<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Classic Elementor Addon Image Zoom
 *
 * @since 1.0.0
 */
class CEA_Elementor_Image_Zoom_Widget extends Widget_Base {
	
    /**
	 * Get widget name.
	 *
	 * Retrieve Cea Image Zoom widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return "ceaimagezoom";
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Cea Image Zoom widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( "Image Zoom", 'classic-elementor-addons-pro' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Cea Image Zoom widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return "ti-zoom-in";
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Cea Image Zoom widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories(): array {
        return [ "classic-elements" ];
    }

    /**
     * Get Keywords
     *
     * Retrieve the list of keywords that used to search for Cea Image Zoom widget
     * 
     * @access public
     * 
     * @return array Widget Keywords
     */
    public function get_keywords(): array {
        return [ 'zozo', 'themes', 'image', 'zoom' ];
    }

    /**
	 * Retrieve the list of scripts the counter widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.3.0
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'cea-custom-front'  ];
	}

    /**
     * Retrieve the list of styles the counter widget depends on.
     *
     * Used to set styles dependencies required to run the widget.
     * 
     * @since 1.3.0
     * @access public
     * 
     * @return array Widget styles dependencies
     */
    public function get_style_depends(): array {
        return [ 'cea-editor' ];
    }

    /**
	 * Register Cea Image Zoom widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls(): void {

        // Content Tab Starts
        $this->start_controls_section(
	    	'general',
	    	[
	    		'label' => esc_html__( 'General', 'classic-elementor-addons-pro' ),
	    		'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
	    	]
	    );
        $this->add_control(
            'select_type',
            [
                'label' => esc_html__( 'Select Type', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'image-zoom' => esc_html__( 'Image Zoom', 'classic-elementor-addons-pro' ),
                    'grid-zoom' => esc_html__( 'Grid Zoom', 'classic-elementor-addons-pro' ),
                ],
                'default' => 'image-zoom',
                'separator' => 'after',
            ]
        );
        $this->add_control(
            'title',
            [
                'label' => esc_html__( 'Title', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'label_block' => true,
                'default' => 'This is Title Text',
            ]
        );
        $this->add_control(
            'sub_title',
            [
                'label' => esc_html__( 'Sub Title', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'label_block' => true,
                'separator' => 'after',
            ]
        );
        $this->add_control(
            'image',
            [
                'label' => esc_html__( 'Image', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
                'condition' => [
                    'select_type' => 'image-zoom',
                ]
            ]
        );
        $repeater = new Repeater();
        $repeater->add_control(
            'image_item',
            [
                'label' => esc_html__( 'Image', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
            ]
        );
        $repeater->add_responsive_control(
            'item_width',
            [
                'label' => esc_html__( 'Width', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ]
                ],
                'selectors' => [
                    '{{CURRENT_ITEM}}.zoom-grid-item' => 'width: {{SIZE}}px;'
                ]
            ]
        );
        $repeater->add_responsive_control(
            'item_position',
            [
                'label' => esc_html__( 'Image Position', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'selectors' => [
                    '{{CURRENT_ITEM}}.zoom-grid-item' => 'top: {{TOP}}{{UNIT}}; left: {{LEFT}}{{UNIT}}; bottom: {{BOTTOM}}{{UNIT}}; right: {{RIGHT}}{{UNIT}};',
                ],
            ]
        );
        $repeater->add_responsive_control(
            'item_index',
            [
                'label' => esc_html__( 'Z - Index', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'min' => 0,
                    'max' => 100,
                ],
                'selectors' => [
                    '{{CURRENT_ITEM}}.zoom-grid-item' => 'z-index: {{SIZE}};'
                ]
            ]
        );
        $this->add_control(
            'image_grid',
            [
                'label' => esc_html__( 'Image Grid', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'condition' => [
                    'select_type' => 'grid-zoom'
                ]
            ]
        );
        $this->end_controls_section();
        // Content Tab Ends

        // Text Style Tab Starts
        $this->start_controls_section(
            'title_style',
            [
                'label' => esc_html__( 'Title', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
			'title_align',
			[
				'label' => esc_html__( 'Alignment', 'classic-elementor-addons-pro' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'default' => 'center',
				'selectors' => [
					'{{WRAPPER}} .image-zoom-text' => 'text-align: {{VALUE}};',
				],
			]
		);
        $this->add_control(
            'title_position',
            [
                'label' => esc_html__( 'Position', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'static' => esc_html__( 'Default', 'classic-elementor-addons-pro' ),
                    'absolute' => esc_html__( 'Absolute', 'classic-elementor-addons-pro' ),
                ],
                'default' => 'static',
                'selectors' => [
                    '{{WRAPPER}} .image-zoom-text' => 'position: {{VALUE}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'title_place',
            [
                'label' => esc_html__( 'Title Position', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'default' => [
                    'unit' => '%',
                    'top'  => 45,
                    'left' => 0,
                    'bottom' => 0,
                    'right' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .image-zoom-text' => 'top: {{TOP}}{{UNIT}}; left: {{LEFT}}{{UNIT}}; bottom: {{BOTTOM}}{{UNIT}}; right: {{RIGHT}}{{UNIT}};',
                ],
                'condition' => [
                    'title_position' => 'absolute',
                ]
            ]
        );
        $this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '{{WRAPPER}} .image-zoom-text',
			]
		);
        $this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'title_shadow',
				'selector' => '{{WRAPPER}} .image-zoom-text',
			]
		);
        $this->add_control(
            'title_clr',
            [
                'label' => esc_html__( 'Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .image-zoom-text' => 'color: {{VALUE}};'
                ],
            ]
        );
        $this->end_controls_section();
        // Text Style Tab Ends

        // Sub Title Style
        $this->start_controls_section(
            'sub_title_style',
            [
                'label' => esc_html__( 'Sub Title', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
			'sub_title_align',
			[
				'label' => esc_html__( 'Alignment', 'classic-elementor-addons-pro' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'default' => 'center',
				'selectors' => [
					'{{WRAPPER}} .image-zoom-sub-title' => 'text-align: {{VALUE}};',
				],
			]
		);
        $this->add_control(
            'sub_title_position',
            [
                'label' => esc_html__( 'Position', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'static' => esc_html__( 'Default', 'classic-elementor-addons-pro' ),
                    'absolute' => esc_html__( 'Absolute', 'classic-elementor-addons-pro' ),
                ],
                'default' => 'static',
                'selectors' => [
                    '{{WRAPPER}} .image-zoom-sub-title' => 'position: {{VALUE}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'sub_title_place',
            [
                'label' => esc_html__( 'Title Position', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'default' => [
                    'unit' => '%',
                    'top'  => 55,
                    'left' => 0,
                    'bottom' => 0,
                    'right' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .image-zoom-sub-title' => 'top: {{TOP}}{{UNIT}}; left: {{LEFT}}{{UNIT}}; bottom: {{BOTTOM}}{{UNIT}}; right: {{RIGHT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'sub_title_typography',
				'selector' => '{{WRAPPER}} .image-zoom-sub-title',
			]
		);
        $this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'sub_title_shadow',
				'selector' => '{{WRAPPER}} .image-zoom-sub-title',
			]
		);
        $this->add_control(
            'sub_title_clr',
            [
                'label' => esc_html__( 'Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .image-zoom-sub-title' => 'color: {{VALUE}};'
                ],
            ]
        );
        $this->end_controls_section();

        // Grid Zoom Effect
        $this->start_controls_section(
            'grid_zoom',
            [
                'label' => esc_html__( 'Grid Blur Effect', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'blur_enter',
            [
                'label' => esc_html__( 'Blur Effect', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'description' => esc_html__( 'Here enter the blur effect of the images. Eg: 3', 'classic-elementor-addons-pro' ),
                'condition' => [
                    'select_type' => 'grid-zoom',
                ]
            ]
        );
        $this->add_control(
            'zoom_in_effect',
            [
                'label' => esc_html__( 'Zoom In Scale', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'description' => esc_html__( 'Here enter the zoom in scale of the images. Eg: 1.3', 'classic-elementor-addons-pro' ),
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 2,
                        'step' => 0.1,
                    ]
                ],
                'condition' => [
                    'select_type' => 'grid-zoom',
                ]
            ]
        );
        $this->end_controls_section();

        // Image Style Starts
        $this->start_controls_section(
            'img_style',
            [
                'label' => esc_html__( 'Image', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'select_type' => 'image-zoom',
                ]
            ]
        );
        $this->add_responsive_control(
            'image_width',
            [
                'label' => esc_html__( 'Image Width', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} .image-zoom-scroll .image-zoom' => 'width: {{SIZE}}{{UNIT}}; height: auto;'
                ],
                'condition' => [
                    'select_type' => 'image-zoom',
                ]
            ]
        );
        $this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
				'include' => [],
				'default' => 'large',
                'condition' => [
                    'select_type' => 'image-zoom',
                ]
			]
		);
        $this->add_control(
            'image_scale',
            [
                'label' => esc_html__( 'Image Animation Scale', 'classic-elementor-addons-pro' ),
                'description' => esc_html__( 'Here enter the expanding scale of the image. Eg: 3', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'condition' => [
                    'select_type' => 'image-zoom',
                ]
            ]
        );
        $this->end_controls_section();
        // Image Style Ends
    }

    /**
	 * Render Tab widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function render() {
        $settings = $this->get_settings_for_display();
        $select_type = $settings[ 'select_type' ];
        $title_text = $settings['title'];
        $sub_title = $settings['sub_title'];
        
        if( $settings['select_type'] == 'image-zoom' ) {
            $img_url = $settings['image']['url'];
            $img_size = wp_get_attachment_image_src( $settings['image']['id'], $settings['thumbnail_size'] );
            $width = ( is_array( $img_size ) && ( $img_size[1] ) ) ? $img_size[1] : null ;
            $height = ( is_array( $img_size ) && ( $img_size[2] ) ) ? $img_size[2] : null ;
            $img_scale = $settings['image_scale'];
            $img_scale = (isset( $img_scale ) && $img_scale != '' ) ? $img_scale : 3;
        
            $this->add_render_attribute( 'image', 'src', $img_url );
            $this->add_render_attribute( 'image', 'width', $width  );
            $this->add_render_attribute( 'image', 'height', $height );
            $this->add_render_attribute( 'image', 'data-scale', $img_scale );
            ?>
            <div class="image-zoom-scroll">
                <h2 class="image-zoom-text"><?php echo $title_text; ?></h2>
                <h6 class="image-zoom-sub-title"><?php echo $sub_title; ?></h6>
                <img class="image-zoom" <?php echo $this->get_render_attribute_string( 'image' ); ?>>
            </div>
            <?php
        }

        if( $settings['select_type'] == 'grid-zoom' ) {
            $blur_effect = $settings['blur_enter'];
            $blur_effect = ( isset( $blur_effect ) && $blur_effect != '' ) ? $blur_effect : 4;
            $zoom_in = $settings['zoom_in_effect']['size'];
            $zoom_in = ( isset( $zoom_in ) && $zoom_in != '' ) ? $zoom_in : 1.3;
            echo'<div zoom_grid_container class="img-grid-container" data-blur="' . $blur_effect . '" data-zoom="' . $zoom_in . '">';
            foreach( $settings['image_grid'] as $img ) {
                $item_url = $img['image_item']['url'];
                echo '<div zoom_grid_item class="zoom-grid-item elementor-repeater-item-' . $img['_id'] . '">';
                echo '<img class="img-grid-item" src="' . $item_url . '" >';
                echo '</div>';
            }
            echo '<h2 class="image-zoom-text">' . $title_text . '</h2>';
            echo '<h6 class="image-zoom-sub-title">' . $sub_title . '</h6>';
            echo '</div>';
        }
    }

}