/*!
 * Averta jQuery Font Selector - A jQuery visual font selecting tool with supporting Google fonts.
 *
 * @version     1.0.0
 * @requires    jQuery 1.9+ | averta.gfonts.js
 * @author      Averta [averta.net]
 * @package     Axiom Framework
 * @copyright   Copyright © 2016-05-24 Averta, all rights reserved
 */

;(function($){

	var api = 'AIzaSyBYkMl1dDDRSmAyHjKInEk9nCgb9-tDQqo',
		url = 'https://www.googleapis.com/webfonts/v1/webfonts',
		list = null,
		options = null;

	var GFonts = function(){},
		waitingList = [];

	GFonts.isLoading = -1;

	GFonts.loadedFonts = [];

	/**
	 * Load and cache list of google fonts
	 * @param  {callback} onLoad
	 * @return {void}
	 */
 	GFonts.getList = function(onLoad){
 		if( list && onLoad ){
 			onLoad.call(null, list)
 			return;
 		}

 		if ( GFonts.isLoading === 1 ) {
 			if ( onLoad ) {
 				waitingList.push(onLoad);
 			}
 			return;
 		}

 		GFonts.isLoading = 1;

 		$.getJSON( url + '?key=' + api, function(data, success, context){
 			GFonts.isLoading = 0;
 			GFonts.listLoaded = true;
	 		GFonts.list = data;
 			if( onLoad ){
 				onLoad.call(null, data, context.responseText);
 			}

 			// call wating list callbacks
 			for ( var i = 0, l = waitingList.length; i !== l; i++ ) {
 				waitingList[i].call(null, data, context.responseText);
 			}

 		});
 	};

 	/**
 	 * generates an option list
 	 * stores variants as data('variants') in each <option></option>
 	 * @return {string}
 	 */
 	GFonts.generateSelectList = function(){
 		if( !list ) return;

 		if( options ) return options;

 		var _options = '';

 		$.each(list.items, function (index, value) {

 			_options += '<option value="' + value.family + '" data-variants="' + value.variants.join(',')  + '">' + value.family + '</option>';
        });

 		options = _options;

        return options;
 	}

 	/**
 	 * Load new google font
 	 * @param  {string} font     font family
 	 * @param  {string} variants  list of font variants (100,200,300,...)
 	 */
 	GFonts.load = function(font, variants){
 		if( GFonts.loadedFonts.indexOf(font) === -1 ){
 			$('head').append('<link rel="stylesheet" href="http://fonts.googleapis.com/css?family=' + escape(font) + ':' + variants + '" >');
			GFonts.loadedFonts.push(font);
		}
 	};

 	window.GFonts = GFonts;

})(jQuery);
;(function ( $, window, document, undefined ) {

	"use strict";

	// Create the defaults once
	var pluginName = 'avertaFontSelector',
		localStorageCached = false,
		cachedGFontsList = null,
		loadedFonts = [],
		defaults = {
			preview     	 	: 'axi-fs-preview',        	// font preview element [class name]
			fontSelect	     	: 'axi-fs-font',         	// fonts select list [class name]
			thicknessSelect  	: 'axi-fs-thickness',    	// font thickness select list [class name]
			previwTextInput  	: 'axi-fs-preview-input',	// font preview text input element [class name]
			container 			: 'axi-fs-container',		// font select div container element [class name]
			lodaing 			: 'axi-fs-loading', 		// font select loading element [class name]

			autoHideElement  	: true,                     // hide HTML select element after init
			insertThickness  	: true,                   	// whether insert font thickness select list element
			insertPreview 	 	: true,						// whether insert font preview
			insertPreviewText	: true,						// whether insert font preview text input element
			previewText 		: 'Lorem ipsum sit amet.',	// preview text

			googleFontsPrefix 	: '-gf-', 					// Google fonts prefix
			systemFontsPrefix 	: '-sys-',					// System fonts prefix
			geaFontsPrefix		: '-gea-',					// Google Early Access fonts prefix
			customFontsPrefix	: '-cus-', 					// Custom fonts prefix

			useGoogleFonts 		: true, 					// whether load google fonts
			systemFonts 		: [],						// system font list DS -> [..,{name:'', thickness:'300,bold,600'},..]
			geaFonts 			: [], 						// Google Early Access fonts DS -> [..,{name:'', thickness:'300,bold,600', url:''},..]
			customFonts 		: [], 						// Custom fonts DS -> [..,{name:'', thickness:'300,bold,600', url:''},..]

			saveToLocalStorage 	: true,						// whether save Google fonts list in localStorage
			lsExpireTime		: 5,						// expire duration for cached fonts in localStorage [days]

			l10n : {										// localization object
				previewTextLabel 		: 'Preview text:',
				fontLabel		 		: 'Font:',
				fontSizeLabel 	 		: 'Size:',
				fontStyleLabel			: 'Style:',
				googleFonts 	 		: 'Google Fonts',
				systemFonts 	 		: 'System Fonts',
				geaFonts 		 		: 'Early Access Fonts',
				customFonts 	 		: 'Custom Fonts'
			}

		};

	// The actual plugin constructor
	function Plugin( element, options ) {
		this.element = element;
		this.$element = $(element);
		this.options = $.extend( {}, defaults, options) ;

		this._defaults = defaults;
		this._name = pluginName;

		this.init();
	}

	$.extend(Plugin.prototype, {

		init : function(){
			var self = this,
				st = self.options;

			if ( st.autoHideElement ) {
				this.$element.css('display', 'none');
			}

			// initial value for previewStr
			self.previewStr = st.previewText;

			// plugin html elements
			self.$container = $('<div class="' + st.container + '"></div>').insertAfter(this.$element).css('display', 'none');
			self.$loading 	= $('<div class="' + st.loading + '"></div>').insertAfter(this.$element)

			// font select html element
			self.$fontSelectList = $('<select class="' + st.fontSelect + '"></select>')
								   .appendTo(self.$container);
			$('<div class="' + st.fontSelect + '-cont"><label>' + st.l10n.fontLabel + '</label></div>')
					.appendTo(self.$container)
					.append(self.$fontSelectList);

			// font thickness html element
			if ( st.insertThickness ) {
				self.$thicknessSelectList = $('<select class=" '+ st.thicknessSelect + '"></select>')
											.on('change', $.proxy(self._onThicknessChange, self));

				$('<div class="' + st.thicknessSelect + '-cont"><label>' + st.l10n.fontStyleLabel	 + '</label></div>')
					.appendTo(self.$container)
					.append(self.$thicknessSelectList);
			}

			// preview element
			if ( st.insertPreview ) {
				self.$preview = $('<div class=" '+ st.preview + '"></div>');

				// preview text input element
				if ( st.insertPreviewText ) {
					self.$previewTextInput = $('<input type="text" class="' + st.previwTextInput + '" value="' + st.previewText + '" />')
										     .on('change keydown', function(){
										     	// update preview test
										     	self.previewStr = $(this).val();
										     	self._updatePreview();
										     });
					$('<div class="' + st.previwTextInput + '-cont"><label>' + st.l10n.previewTextLabel + '</label></div>')
						.appendTo(self.$container)
						.append(self.$previewTextInput);
				}

				self.$preview.appendTo(self.$container);
			}

			// parse input value
			var val = self.$element.val(),
				prefixLen = 0;

			if ( val.length > 1 ){
				// prefix:
				if ( val.indexOf(st.googleFontsPrefix) !== -1 ) {
					self.currentFontType = 'gfont';
					prefixLen = st.googleFontsPrefix.length;
				} else if ( val.indexOf(st.systemFontsPrefix) !== -1 ) {
					self.currentFontType = 'sysfont';
					prefixLen = st.systemFontsPrefix.length;
				} else if ( val.indexOf(st.customFontsPrefix) !== -1 ) {
					self.currentFontType = 'cusfont';
					prefixLen = st.customFontsPrefix.length;
				} else if ( val.indexOf(st.geaFontsPrefix) !== -1 ) {
					self.currentFontType = 'geafont';
					prefixLen = st.geaFontsPrefix.length;
				}

				// is value contains font thickness
				if ( val.indexOf(':') !== -1 ) {
					self.currentThikness = val.slice(val.lastIndexOf(':') + 1);
					self.currentFont = val.slice(prefixLen, val.lastIndexOf(':'));
				} else {
					self.currentFont = val.slice(prefixLen);
				}
			}

			self._getFonts();
			//self._updatePreview();
		},

		/**
		 * calculates numbers of days between two dates
		 * @private
		 * @param  {Date} date1  End date
		 * @param  {Date} date2  Start date
		 * @return {Number}
		 */
		_daysBetween : function (date1, date2) {

		    // The number of milliseconds in one day
		    var ONE_DAY = 1000 * 60 * 60 * 24;

		    // Calculate the difference in milliseconds
		    var difference_ms = Math.abs(date1 - date2);

		    // Convert back to days and return
		    return Math.round(difference_ms/ONE_DAY);

		},

		/**
		 * Prepare google fonts, it first looks localStorage (if enabled) for cached Google fonts if it is not available there or expired
		 * 	loads new list from Google font api.
		 *
		 * @private
		 */
		_getFonts : function () {
			var self = this,
				st = self.options,
				isls = typeof(Storage) !== "undefined"; // is local storage available

			if ( cachedGFontsList || !st.useGoogleFonts ) {
				self.gFontsList = cachedGFontsList;
				self._generateFontList();
				return;
			}

			// if local storage enabled
			if ( st.saveToLocalStorage && isls ) {
				// is it already cached?
				var gfs = localStorage.AxiGoogleFonts,
					expireDate = Date.parse(localStorage.AxiGoogleFontsDate);

				if ( gfs && expireDate && self._daysBetween(new Date().getTime(), expireDate) < st.lsExpireTime) {
					self.gFontsList = JSON.parse(gfs);
					cachedGFontsList = self.gFontsList;
					self._generateFontList();
					return;
				}
			}

			// it is not cached so lets load it
			GFonts.getList(function(list, data){
				// save to local storage
				if ( st.saveToLocalStorage && isls && !localStorageCached ) {
					localStorageCached = true;
					localStorage.AxiGoogleFonts = data;
					localStorage.AxiGoogleFontsDate = new Date();
				}

				self.gFontsList = list;
				cachedGFontsList = list;
				self._generateFontList();
			});


		},

		/**
		 * Generates fonts list select element
		 * @private
		 */
		_generateFontList : function () {
			var self = this,
				st = self.options,
				selectOptions = '', i, l;

			// remove loading
			self.$loading.remove();
			self.$container.css('display', '');

			// system fonts
			if ( st.systemFonts.length !== 0 ) {
	 			selectOptions += '<optgroup label="' + st.l10n.systemFonts + '">';
				for ( i = 0, l = st.systemFonts.length; i !== l; i++ ) {
					selectOptions += '<option value="' + st.systemFonts[i].name + '" data-type="sysfont" data-thickness="' + st.systemFonts[i].thickness  + '">' + st.systemFonts[i].name + '</option>';
				}
				selectOptions += '</optgroup>';
			}

			// google fonts
			if ( st.useGoogleFonts ) {
				cachedGFontsList = self.gFontsList;
				selectOptions += '<optgroup label="' + st.l10n.googleFonts + '">';
				$.each(self.gFontsList.items, function (index, value) {
		 			selectOptions += '<option value="' + value.family + '" data-type="gfont" data-thickness="' + value.variants.join(',')  + '">' + value.family + '</option>';
		        });
		        selectOptions += '</optgroup>';
			}

			// gea fonts
			if ( st.geaFonts.length !== 0 ) {
	 			selectOptions += '<optgroup label="' + st.l10n.geaFonts + '">';
				for ( i = 0, l = st.geaFonts.length; i !== l; i++ ) {
					selectOptions += '<option value="' + st.geaFonts[i].name + '" data-type="geafont" data-thickness="' + st.geaFonts[i].thickness  + '">' + st.geaFonts[i].name + '</option>';
				}
				selectOptions += '</optgroup>';
			}

			// custom fonts
			if ( st.customFonts.length !== 0 ) {
	 			selectOptions += '<optgroup label="' + st.l10n.customFonts + '">';
				for ( i = 0, l = st.customFonts.length; i !== l; i++ ) {
					selectOptions += '<option value="' + st.customFonts[i].name + '" data-type="cusfont" data-thickness="' + st.customFonts[i].thickness  + '">' + st.customFonts[i].name + '</option>';
				}
				selectOptions += '</optgroup>';
			}

			self.$fontSelectList.html(selectOptions)
				.on('change', $.proxy(self._onFontChange, self));

			if ( !self.currentFont ) {
				self.currentFont = self.$fontSelectList.find('option:first').val();
			}

			self.$fontSelectList.val(self.currentFont).trigger('change');
		},

		/**
		 * on font changed in font select list element
		 * @private
		 */
		_onFontChange : function (event) {

			var $this = $(event.currentTarget),
				self = this,
				st = self.options,
				foundThinkess = false,
				$selected = $this.find('option:selected');

			self.currentFont = $this.val();
			self.currentFontType = $selected.data('type');

			if ( st.insertThickness ) {
				// generate thickness select
				var options = '';
				self.currentFontThickness = $selected.data('thickness');
				$.each(self.currentFontThickness.split(','), function (index, thickness) {
					var label = (thickness.indexOf('italic') !== -1 ? thickness.replace(/italic/, ' italic') : thickness);
					if ( thickness === self.currentThikness ) {
						options += '<option value="' + thickness + '" selected>' + label + '</option>';
						foundThinkess = true;
					} else {
						options += '<option value="' + thickness + '">' + label + '</option>';
					}
				});

				self.$thicknessSelectList.html(options);

				if ( !foundThinkess ) {
					self.$thicknessSelectList.find('option:first').attr('selected', 'selected').trigger('change');
				}
			}

			self._updatePreview();
			self._updateInputValue();
		},

		/**
		 * on thickness changed in thickness select element
		 * @private
		 * @param  {jQuery event} event
		 */
		_onThicknessChange : function (event) {
			var $this = $(event.currentTarget),
				self = this

			self.currentThikness = $this.val();
			self._updatePreview();
			self._updateInputValue();
		},

		/**
		 * loads Google early access or custom fonts
		 * @private
		 * @param {string} fontName
		 * @param {string} type
		 */
		_loadFont : function (fontName, type) {
			if ( loadedFonts.indexOf(fontName) !== -1 ) {
				return;
			}

			var st = this.options,
				list, url;

			if ( type === 'cusfont' ) {
				list = st.customFonts
			} else if ( type === 'geafont') {
				list = st.geaFonts;
			} else {
				return;
			}

			// find font
			$.each(list, function(index, font){
				if ( font.name === fontName ) {
					url = font.url;
					return false;
				}
			});

			if ( !url ) {
				return;
			}

			$('head').append('<link rel="stylesheet" href="' + url + '" >');
			loadedFonts.push(fontName);
		},

		/**
		 * updates font preview
		 * @private
		 */
		_updatePreview : function () {

			if ( !this.options.insertPreview ) {
				return;
			}

			var self = this,
				st = self.options,
				prv = self.$preview;

			prv.html(self.previewStr)
			prv.css('font-family', self.currentFont);

			if ( self.currentThikness ) {
				prv.css('font-weight', self.currentThikness.replace(/italic/, ''));

				// italic style
				if ( self.currentThikness.indexOf('italic') !== -1 ) {
					prv.css('font-style', 'italic');
				} else {
					prv.css('font-style', '');
				}
			}

			// load fonts
			if ( st.useGoogleFonts && self.currentFontType === 'gfont' ) {
				GFonts.load(self.currentFont, self.currentFontThickness || '');
			} else if ( self.currentFontType === 'geafont' || self.currentFontType === 'cusfont'){
				self._loadFont(self.currentFont, self.currentFontType);
			}

		},

		/**
		 * updates font selector hidden input value
		 * @private
		 */
		_updateInputValue : function () {
			var self = this,
				st = self.options,
				val = '';

			switch ( self.currentFontType ) {
				case 'sysfont':
					val = st.systemFontsPrefix;
					break;
				case 'gfont':
					val = st.googleFontsPrefix;
					break;
				case 'geafont':
					val = st.geaFontsPrefix;
					break;
				case 'cusfont':
					val = st.customFontsPrefix;
			}

			val += self.currentFont;

			if ( st.insertThickness ) {
				val += ':' + self.currentThikness;
			}

			self.$element.val(val).trigger('change');
		}


	});


	$.fn[pluginName] = function ( options ) {
		var args = arguments;

		// Is the first parameter an object (options), or was omitted,
		// instantiate a new instance of the plugin.
		if (options === undefined || typeof options === 'object') {
		    return this.each(function () {

		        // Only allow the plugin to be instantiated once,
		        // so we check that the element has no plugin instantiation yet
		        if (!$.data(this, 'plugin_' + pluginName)) {

		            // if it has no instance, create a new one,
		            // pass options to our plugin constructor,
		            // and store the plugin instance
		            // in the elements jQuery data object.
		            $.data(this, 'plugin_' + pluginName, new Plugin( this, options ));
		        }
		    });
		}
		// If the first parameter is a string and it doesn't start
		// with an underscore or "contains" the `init`-function,
		// treat this as a call to a public method.
		// } else if (typeof options === 'string' && options[0] !== '_' && options !== 'init') {

		//     // Cache the method call
		//     // to make it possible
		//     // to return a value
		//     var returns;

		//     this.each(function () {
		//         var instance = $.data(this, 'plugin_' + pluginName);

		//         // Tests that there's already a plugin-instance
		//         // and checks that the requested public method exists
		//         if (instance instanceof Plugin && typeof instance[options] === 'function') {

		//             // Call the method of our plugin instance,
		//             // and pass it the supplied arguments.
		//             returns = instance[options].apply( instance, Array.prototype.slice.call( args, 1 ) );
		//         }

		//         // Allow instances to be destroyed via the 'destroy' method
		//         if (options === 'destroy') {
		//           $.data(this, 'plugin_' + pluginName, null);
		//         }
		//     });

		//     // If the earlier cached method
		//     // gives a value back return the value,
		//     // otherwise return this to preserve chainability.
		//     return returns !== undefined ? returns : this;
		// }
	}

}(jQuery, window, document));
