<?php
/**
 * This file displays all CRM activities in a list format and stats.
 *
 * @since 4.4.5
 */

// Fetch all activities from database.
$activities = Realhomes_Crm_Activities::fetch_activities();
?>

<div class="dashboard-crm-page-wrap dashboard-crm-activities grid grid-cols-12 gap-8">

	<div class="col-span-12 xl:col-span-5">
		<h3 class="text-md mb-4"><?php esc_html_e( 'Activities', RH_TEXT_DOMAIN ); ?></h3>
		<div class="bg-white shadow-sm p-10 rounded-md">
			<?php if ( ! empty( $activities ) ) { ?>
				<ul role="list">
					<?php foreach ( $activities as $activity ) {
						$activity_type = $activity->type;

						// Decode the JSON data to an associative array.
						$activity_data = json_decode( $activity->data, true );

						// Check if this is a review
						$is_review = ( 'review' === $activity_type );

						if ( $is_review ) {
							$header_label  = esc_html__( 'Left a Review on', RH_TEXT_DOMAIN );
							$message_label = esc_html__( 'Review', RH_TEXT_DOMAIN );
							$rating        = ! empty( $activity_data['rating'] ) ? intval( $activity_data['rating'] ) : 0;
						} elseif ( 'comment' === $activity_type ) {
							$header_label  = esc_html__( 'Commented on', RH_TEXT_DOMAIN );
							$message_label = esc_html__( 'Comment', RH_TEXT_DOMAIN );
						} elseif ( 'booking' === $activity_type ) {
							$header_label  = esc_html__( 'New Booking for', RH_TEXT_DOMAIN );
						} else {
							$header_label  = esc_html__( 'New Lead from', RH_TEXT_DOMAIN );
							$message_label = esc_html__( 'Message', RH_TEXT_DOMAIN );
						}
						?>
						<li class="rh-crm-item-wrapper">
							<div class="relative pb-16">
								<span class="absolute left-5 top-5 -ml-px h-full w-0.5 bg-gray-200" aria-hidden="true"></span>
								<div class="relative flex items-start space-x-6">
									<div>
										<div class="relative px-1">
											<div class="flex size-12 items-center justify-center rounded-full bg-gray-100 ring-8 ring-white">
												<img class="rounded-full" src="<?php echo get_avatar_url( $activity_data['email'] ) ?>" alt="Avatar">
											</div>
										</div>
									</div>
									<div class="min-w-0 flex-1 py-1.5">
										<div class="text-sm text-gray-500 mb-6 flex justify-between">
											<div>
												<span class="font-medium text-gray-900 mr-2"><?php echo esc_html( $activity_data['name'] ?? '' ); ?></span>
												<?php
												echo $header_label;
												if ( ! empty( $activity_data['source'] ) && ! empty( $activity_data['source_url'] ) ) {
													echo '<a href="' . esc_url( $activity_data['source_url'] ) . '" target="_blank" class="ml-2 font-medium text-[var(--dashboard-primary-color)] hover:text-slate-700">' . esc_html( $activity_data['source'] ) . '</a>';
												}
												?>

												<?php if ( $is_review ) { ?>
													<div class="flex items-center gap-1 mt-4">
														<?php for ( $i = 1; $i <= 5; $i ++ ) {
															$svg_classes = 'w-6 h-6 ';
															$svg_classes .= ( $i <= $rating ) ? 'text-yellow-400 fill-yellow-400' : 'text-gray-300';
															Realhomes_Crm_SVG::the_svg( 'star', $svg_classes );
														} ?>
														<span class="ml-1 text-xs font-medium text-gray-500">(<?php echo esc_html( $rating ); ?>/5)</span>
													</div>
												<?php } ?>
											</div>
											<span class="whitespace-nowrap text-sm"><?php echo esc_html( human_time_diff( strtotime( $activity->created_at ), current_time( 'timestamp' ) ) ) . ' ago'; ?></span>
										</div>
										<div class="mt-8 flex flex-column gap-y-4 sm:gap-y-2 text-sm text-gray-700 border border-gray-200 p-6 rounded-md">
											<?php
											if ( is_array( $activity_data ) ) {
												foreach ( $activity_data as $activity_datum => $activity_datum_value ) {
													// Skip internal fields that shouldn't be displayed
													if ( in_array( $activity_datum, array( 'source', 'source_url', 'source_id', 'comment_id', 'user_id', 'rating', 'name' ) ) ) {
														continue;
													}

													switch ( $activity_datum ) {
														case 'mobile':
															echo '<p class="sm:flex items-start gap-x-2 gap-y-2">
																		<span class="flex items-center shrink-0 min-w-[85px] text-gray-500 mb-2 sm:mb-0">
																			<i class="fas fa-mobile-alt mr-2.5 ml-1"></i>
																			' . esc_html__( 'Mobile', RH_TEXT_DOMAIN ) . '
																		</span>
																		<a href="tel:' . esc_attr( $activity_datum_value ) . '" class="break-all" target="_blank">' . esc_html( $activity_datum_value ) . '</a>
																	</p>';
															break;

														case 'email':
															echo '<p class="sm:flex items-start gap-x-2 gap-y-2">
																		<span class="flex items-center shrink-0 min-w-[85px] text-gray-500 mb-2 sm:mb-0">
																			<i class="far fa-envelope mr-2"></i>
																			' . esc_html__( 'Email', RH_TEXT_DOMAIN ) . '
																		</span>
																		<a href="mailto:' . esc_attr( $activity_datum_value ) . '" class="break-all" target="_blank">' . esc_html( $activity_datum_value ) . '</a>
																	</p>';
															break;

														case 'message':
															echo '<p class="sm:flex items-start gap-x-2 gap-y-2">
																	<span class="flex items-center shrink-0 min-w-[85px] text-gray-500 mb-2 sm:mb-0">
																		<i class="far fa-comment-alt mr-2"></i>
																		' . $message_label . '
																	</span>
																	<span class="break-words leading-[2.2rem]">' . esc_html( $activity_datum_value ) . '</span>
																</p>';
															break;
													}

												}
											}
											?>
											<a data-type="activity" data-id="<?php echo esc_attr( $activity->id ); ?>" class="rh-crm-delete-item absolute bottom-0 right-0 p-3 cursor-pointer">
												<?php Realhomes_Crm_SVG::the_svg( 'cross', 'size-8' ); ?>
											</a>
										</div>
									</div>
								</div>
							</div>
						</li>
					<?php } ?>
				</ul>
			<?php } else {
				Realhomes_Crm_Helper::no_item_found_note( description: esc_html__( 'There are no activities recorded so far :(', RH_TEXT_DOMAIN ), icon_size: 'size-20' );
			} ?>
		</div>
	</div>

	<div class="col-span-12 xl:col-span-7">
		<!--Leads Stats-->
		<h3 class="text-md mb-4"><?php esc_html_e( 'Leads Stats', RH_TEXT_DOMAIN ); ?></h3>
		<div id="rh-crm-leads-stats" class="grid grid-cols-3 divide-x item-center shadow-sm rounded-md py-4 bg-white mb-10"></div>
		<!--Deals Stats-->
		<h3 class="text-md mb-4"><?php esc_html_e( 'Deals Stats', RH_TEXT_DOMAIN ); ?></h3>
		<div id="rh-crm-deals-stats" class="flex justify-evenly item-center shadow-sm rounded-md py-4 bg-white mb-10 divide-x">
			<canvas class="max-w-[70px]" id="rh-crm-deals-chart"></canvas>
		</div>
	</div>

</div><!--.dashboard-crm-page-wrap-->