<?php
/**
 * This file loads all RVR resources and initiate the actions and hooks.
 *
 * @package    realhomes_vacation_rentals
 * @subpackage realhomes_vacation_rentals/assets
 */

/**
 * The class responsible for orchestrating booking widget and related stuff
 */
require_once plugin_dir_path( __FILE__ ) . "/widgets/rvr-booking-widget.php";

if ( ! function_exists( 'rvr_booking_admin_styles' ) ) {
	function rvr_booking_admin_styles( $hook ) {

		wp_enqueue_style( 'rvr_booking_admin', plugins_url( '../assets/css/rvr-booking-admin.css', __FILE__ ) );

		wp_enqueue_script(
			'rvr_booking_admin',
			plugins_url( '../assets/js/rvr-booking-admin.js', __FILE__ ),
			array( 'jquery' ),
			RVR_VERSION,
			true
		);

		// Localize strings for the script
		wp_localize_script(
			'rvr_booking_admin',
			'rvrBookingAdminStrings',
			array(
				'processingLabel'  => esc_html__( 'Processing', RVR_TEXT_DOMAIN ),
				'invoiceGenerated' => esc_html__( 'Invoice Generated', RVR_TEXT_DOMAIN ),
				'invoiceRemoved'   => esc_html__( 'Invoice Removed', RVR_TEXT_DOMAIN ),
				'somethingWrong'   => esc_html__( 'Something went wrong!', RVR_TEXT_DOMAIN )
			)
		);
	}

	add_action( 'admin_enqueue_scripts', 'rvr_booking_admin_styles' );
}

if ( ! function_exists( 'rvr_booking_public_styles' ) ) {
	/**
	 * Enqueue front-end scripts
	 */
	function rvr_booking_public_styles() {

		// Enqueued for some help in the datepicker stuff.
		wp_enqueue_script( 'jquery-ui-datepicker' );

		// DateRangePicker calendar styles and scripts.
		wp_enqueue_style( 'rvr_booking_daterangepicker', plugins_url( '../assets/css/daterangepicker.css', __FILE__ ) );
		wp_enqueue_script( 'moment', plugins_url( '../assets/js/daterangepicker/moment.min.js', __FILE__ ), array( 'jquery' ), RVR_VERSION, true );
		wp_enqueue_script( 'rvr_booking_daterangepicker', plugins_url( '../assets/js/daterangepicker/daterangepicker.min.js', __FILE__ ), array( 'jquery' ), RVR_VERSION, true );

		// Plugin styles and scripts.
		wp_enqueue_style( 'rvr_booking_public', plugins_url( '../assets/css/rvr-booking-public.css', __FILE__ ) );

		if ( defined( 'INSPIRY_DESIGN_VARIATION' ) ) {
			wp_enqueue_style( 'realhomes_vacation_rentals', plugins_url( '../assets/css/' . INSPIRY_DESIGN_VARIATION . '/realhomes-vacation-rentals.css', __FILE__ ) );
		}

		wp_register_script( 'rvr_booking_public', plugins_url( '../assets/js/rvr-booking-public.js', __FILE__ ), array( 'jquery' ), RVR_VERSION, true );

		$rvr_settings           = get_option( 'rvr_settings' );
		$rvr_date_format_method = $rvr_settings['rvr_date_format_method'];
		$rvr_date_format        = esc_html( 'YYYY-MM-DD' );

		if ( 'wordpress' === $rvr_date_format_method ) {
			$rvr_date_format = get_option( 'date_format' );
		} else if ( 'custom' === $rvr_date_format_method ) {
			$rvr_date_format = $rvr_settings['rvr_date_format_custom'];
		}

		wp_localize_script(
			'rvr_booking_public',
			'availability_calendar_data',
			array(
				'day_name'               => array(
					esc_html_x( 'su', 'Sunday', RVR_TEXT_DOMAIN ),
					esc_html_x( 'mo', 'Monday', RVR_TEXT_DOMAIN ),
					esc_html_x( 'tu', 'Tuesday', RVR_TEXT_DOMAIN ),
					esc_html_x( 'we', 'Wednesday', RVR_TEXT_DOMAIN ),
					esc_html_x( 'th', 'Thursday', RVR_TEXT_DOMAIN ),
					esc_html_x( 'fr', 'Friday', RVR_TEXT_DOMAIN ),
					esc_html_x( 'sa', 'Saturday', RVR_TEXT_DOMAIN ),
				),
				'month_name'             => array(
					esc_html__( 'january', RVR_TEXT_DOMAIN ),
					esc_html__( 'february', RVR_TEXT_DOMAIN ),
					esc_html__( 'march', RVR_TEXT_DOMAIN ),
					esc_html__( 'april', RVR_TEXT_DOMAIN ),
					esc_html__( 'may', RVR_TEXT_DOMAIN ),
					esc_html__( 'june', RVR_TEXT_DOMAIN ),
					esc_html__( 'july', RVR_TEXT_DOMAIN ),
					esc_html__( 'august', RVR_TEXT_DOMAIN ),
					esc_html__( 'september', RVR_TEXT_DOMAIN ),
					esc_html__( 'october', RVR_TEXT_DOMAIN ),
					esc_html__( 'november', RVR_TEXT_DOMAIN ),
					esc_html__( 'december', RVR_TEXT_DOMAIN ),
				),
				'rvr_strings'            => array(
					'wrong_date_warning'     => esc_html__( 'Wrong date selection. Try again!', RVR_TEXT_DOMAIN ),
					'reserved_dates_overlap' => esc_html__( 'Selected dates include reserved dates. Please choose different dates.', RVR_TEXT_DOMAIN ),
				),
				'booking_type'           => get_option( 'rvr_settings' )['rvr_booking_type'] ?? 'full_day',
				'rvr_date_format_method' => sanitize_text_field( $rvr_date_format_method ),
				'rvr_date_format'        => sanitize_text_field( $rvr_date_format ),
			)
		);

		wp_enqueue_script( 'rvr_booking_public' );
	}

	add_action( 'wp_enqueue_scripts', 'rvr_booking_public_styles' );
}

if ( ! function_exists( 'rvr_is_enabled' ) ) {
	/**
	 * Check if RVR is enabled from its settings.
	 *
	 * @return bool
	 */
	function rvr_is_enabled() {
		$options = get_option( 'rvr_settings' );

		if ( $options['rvr_activation'] ) {
			return true;
		}

		return false;
	}
}

if ( ! function_exists( 'rvr_body_classes' ) ) {
	/**
	 * RVR body classes.
	 */
	function rvr_body_classes( $classes ) {

		$classes[] = 'rvr-is-enabled';

		return $classes;
	}

	add_filter( 'body_class', 'rvr_body_classes' );
}

if ( ! function_exists( 'rvr_get_reserved_and_booked_dates' ) ) {
	/**
	 * Retrieves all reserved and booked dates for a specific property as a single comma-separated string.
	 *
	 * @since 4.3.8
	 *
	 * @param int $post_id The ID of the property post.
	 *
	 * @return string Comma-separated dates (e.g., "2024-08-08,2024-08-09,2024-08-10").
	 */
	function rvr_get_reserved_and_booked_dates( $post_id ) {
		if ( ! $post_id || ! is_numeric( $post_id ) ) {
			return "0-0-0";
		}

		$reserved_dates = get_post_meta( $post_id, 'rvr_reserved_dates', true );
		$booked_dates   = get_post_meta( $post_id, 'rvr_booked_dates', true );

		// Ensure values are arrays
		$reserved_dates = is_array( $reserved_dates ) ? $reserved_dates : [];
		$booked_dates   = is_array( $booked_dates ) ? $booked_dates : [];

		// Fetch availability table and booking type
		$availability_table = get_post_meta( $post_id, 'rvr_property_availability_table', true );
		$booking_type       = get_option( 'rvr_settings' )['rvr_booking_type'] ?? 'full_day';
		$unavailable_dates  = [];

		if ( ! empty( $availability_table ) && is_array( $availability_table ) ) {
			foreach ( $availability_table as $dates ) {
				$begin = new DateTime( $dates[0] );
				$end   = new DateTime( $dates[1] );

				if ( ( 'full_day' === $booking_type ) || ( 'split_day' === $booking_type && $begin == $end ) ) {
					$end = $end->modify( '+1 day' );
				}

				$interval  = new DateInterval( 'P1D' );
				$daterange = new DatePeriod( $begin, $interval, $end );

				foreach ( $daterange as $date ) {
					$unavailable_dates[] = $date->format( "Y-m-d" );
				}
			}
		}

		// Merge all date arrays **without removing duplicates**
		$all_dates = array_merge( $reserved_dates, $booked_dates, $unavailable_dates );

		// Sort the dates to maintain proper order
		sort( $all_dates );

		// Return as a single comma-separated string or "0-0-0" if empty
		return ! empty( $all_dates ) ? implode( ',', $all_dates ) : "0-0-0";
	}
}
