<?php
/**
 * This class is responsible for the Invoice post type and related stuff.
 *
 * @package    realhomes_vacation_rentals
 * @subpackage realhomes_vacation_rentals/admin
 */

if ( ! class_exists( 'RVR_Invoice' ) ) {
	/**
	 * Class RVR_Invoice
	 *
	 * Responsible for all stuff related to Invoice Post Type.
	 *
	 * @package    realhomes_vacation_rentals
	 * @subpackage realhomes_vacation_rentals/admin
	 */
	class RVR_Invoice {

		/**
		 * Register invoice custom post type
		 */
		public function rvr_invoice_post_type() {

			// Add 'invoice' post type only if WooCommerce payments are enabled.
			if ( ! rvr_is_wc_payment_enabled() ) {
				return;
			}

			$labels = array(
				'name'                  => esc_html_x( 'Invoices', 'Post Type General Name', RVR_TEXT_DOMAIN ),
				'singular_name'         => esc_html_x( 'Invoice', 'Post Type Singular Name', RVR_TEXT_DOMAIN ),
				'menu_name'             => esc_html__( 'Invoices', RVR_TEXT_DOMAIN ),
				'name_admin_bar'        => esc_html__( 'Invoice', RVR_TEXT_DOMAIN ),
				'archives'              => esc_html__( 'Invoice Archives', RVR_TEXT_DOMAIN ),
				'attributes'            => esc_html__( 'Invoice Attributes', RVR_TEXT_DOMAIN ),
				'parent_item_colon'     => esc_html__( 'Parent Invoice:', RVR_TEXT_DOMAIN ),
				'all_items'             => esc_html__( 'All Invoices', RVR_TEXT_DOMAIN ),
				'add_new_item'          => esc_html__( 'Add New Invoice', RVR_TEXT_DOMAIN ),
				'add_new'               => esc_html__( 'Add New', RVR_TEXT_DOMAIN ),
				'new_item'              => esc_html__( 'New Invoice', RVR_TEXT_DOMAIN ),
				'edit_item'             => esc_html__( 'Edit Invoice', RVR_TEXT_DOMAIN ),
				'update_item'           => esc_html__( 'Update Invoice', RVR_TEXT_DOMAIN ),
				'view_item'             => esc_html__( 'View Invoice', RVR_TEXT_DOMAIN ),
				'view_items'            => esc_html__( 'View Invoices', RVR_TEXT_DOMAIN ),
				'search_items'          => esc_html__( 'Search Invoice', RVR_TEXT_DOMAIN ),
				'not_found'             => esc_html__( 'Not found', RVR_TEXT_DOMAIN ),
				'not_found_in_trash'    => esc_html__( 'Not found in Trash', RVR_TEXT_DOMAIN ),
				'featured_image'        => esc_html__( 'Featured Image', RVR_TEXT_DOMAIN ),
				'set_featured_image'    => esc_html__( 'Set featured image', RVR_TEXT_DOMAIN ),
				'remove_featured_image' => esc_html__( 'Remove featured image', RVR_TEXT_DOMAIN ),
				'use_featured_image'    => esc_html__( 'Use as featured image', RVR_TEXT_DOMAIN ),
				'insert_into_item'      => esc_html__( 'Insert into invoice', RVR_TEXT_DOMAIN ),
				'uploaded_to_this_item' => esc_html__( 'Uploaded to this invoice', RVR_TEXT_DOMAIN ),
				'items_list'            => esc_html__( 'Invoices list', RVR_TEXT_DOMAIN ),
				'items_list_navigation' => esc_html__( 'Invoices list navigation', RVR_TEXT_DOMAIN ),
				'filter_items_list'     => esc_html__( 'Filter invoices list', RVR_TEXT_DOMAIN ),
			);
			$args   = array(
				'label'               => esc_html__( 'Invoice', RVR_TEXT_DOMAIN ),
				'description'         => esc_html__( 'Invoices of property bookings.', RVR_TEXT_DOMAIN ),
				'labels'              => $labels,
				'supports'            => array( 'title' ),
				'hierarchical'        => false,
				'public'              => true,
				'show_ui'             => true,
				'show_in_menu'        => 'rvr',
				'menu_position'       => 5,
				'menu_icon'           => 'dashicons-printer',
				'show_in_admin_bar'   => false,
				'show_in_nav_menus'   => false,
				'can_export'          => false,
				'has_archive'         => false,
				'exclude_from_search' => true,
				'publicly_queryable'  => false,
				'rewrite'             => false,
				'capability_type'     => 'post',
				'capabilities'        => array(
					'create_posts' => 'do_not_allow',
				),

				// Disabled REST API Support for Invoices
				//'show_in_rest'        => true,
				//'rest_base'           => apply_filters( 'rvr_invoice_rest_base', esc_html__( 'invoices', RVR_TEXT_DOMAIN ) )
			);
			register_post_type( 'invoice', $args );

		}

		/**
		 * Replace "Enter Title Here" placeholder text.
		 *
		 * @param string $title Default title text.
		 *
		 * @return string
		 */
		public function invoice_title_text( $title ) {
			$screen = get_current_screen();

			if ( 'invoice' === $screen->post_type ) {
				$title = esc_html__( 'Enter invoice ID here', RVR_TEXT_DOMAIN );
			}

			return $title;
		}

		/**
		 * Add Metabox to Display Invoice Payment Information
		 */
		public function add_invoice_payment_metabox() {
			add_meta_box( 'invoice-payment-metabox', esc_html__( 'Invoice Information', RVR_TEXT_DOMAIN ), array(
				$this,
				'invoice_payment_metabox'
			), 'invoice', 'normal', 'default' );
		}

		/**
		 * Invoice Payment Metabox.
		 *
		 * @param object $post Current invoice post.
		 */
		public function invoice_payment_metabox( $post ) {

			$values        = get_post_custom( $post->ID );
			$not_available = esc_html__( 'Not Available', RVR_TEXT_DOMAIN );

			$transaction_id  = isset( $values['transaction_id'] ) ? esc_attr( $values['transaction_id'][0] ) : $not_available;
			$payment_date    = isset( $values['payment_date'] ) ? esc_attr( $values['payment_date'][0] ) : $not_available;
			$payment_method  = isset( $values['payment_method'] ) ? esc_attr( $values['payment_method'][0] ) : $not_available;
			$payer_email     = isset( $values['payer_email'] ) ? esc_attr( $values['payer_email'][0] ) : $not_available;
			$first_name      = isset( $values['first_name'] ) ? esc_attr( $values['first_name'][0] ) : $not_available;
			$last_name       = isset( $values['last_name'] ) ? esc_attr( $values['last_name'][0] ) : $not_available;
			$payment_status  = isset( $values['payment_status'] ) ? esc_attr( $values['payment_status'][0] ) : $not_available;
			$payment_amount  = isset( $values['payment_amount'] ) ? esc_attr( $values['payment_amount'][0] ) : $not_available;
			$amount_currency = isset( $values['amount_currency'] ) ? esc_attr( $values['amount_currency'][0] ) : $not_available;
			$booking_id      = $values['booking_id'][0];
			if ( ! empty( $booking_id ) ) {
				$booking_info = '<a href="' . get_edit_post_link( $booking_id ) . '">' . get_the_title( $booking_id ) . '</a>';
			} else {
				$booking_info = $not_available;
			}
			?>
            <table style="width:100%;">
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Payer First Name', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo esc_html( $first_name ); ?></td>
                </tr>
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Payer Last Name', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo esc_html( $last_name ); ?></td>
                </tr>
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Payer Email', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo esc_html( $payer_email ); ?></td>
                </tr>
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Payment Method', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo esc_html( ucfirst( $payment_method ) ); ?></td>
                </tr>
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Transaction ID', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo esc_html( $transaction_id ); ?></td>
                </tr>
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Payment Date', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo esc_html( $payment_date ); ?></td>
                </tr>
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Amount', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo esc_html( $payment_amount ); ?></td>
                </tr>
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Currency', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo esc_html( $amount_currency ); ?></td>
                </tr>
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Payment Status', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo esc_html( $payment_status ); ?></td>
                </tr>
                <tr>
                    <td style="width:25%; vertical-align: top; border: 1px solid #e1e1e3; padding: 10px;">
                        <strong><?php esc_html_e( 'Booking ID', RVR_TEXT_DOMAIN ); ?></strong>
                    </td>
                    <td style="width:75%; border: 1px solid #e1e1e3; padding: 10px;"><?php echo $booking_info; ?></td>
                </tr>
            </table>
			<?php
		}

		/**
		 * Custom columns for invoices.
		 *
		 * @param array $columns Default columns.
		 *
		 * @return array
		 */
		public function invoice_edit_columns( $columns ) {

			$columns = array(
				'cb'           => '<input type="checkbox" />',
				'title'        => esc_html__( 'Invoice ID', RVR_TEXT_DOMAIN ),
				'booking'      => esc_html__( 'Booking ID', RVR_TEXT_DOMAIN ),
				'payer'        => esc_html__( 'Payer', RVR_TEXT_DOMAIN ),
				'payer_email'  => esc_html__( 'Payer Email', RVR_TEXT_DOMAIN ),
				'amount'       => esc_html__( 'Amount', RVR_TEXT_DOMAIN ),
				'currency'     => esc_html__( 'Currency', RVR_TEXT_DOMAIN ),
				'method'       => esc_html__( 'Method', RVR_TEXT_DOMAIN ),
				'payment_date' => esc_html__( 'Payment Time', RVR_TEXT_DOMAIN ),
			);

			/**
			 * WPML Support.
			 */
			if ( defined( 'ICL_SITEPRESS_VERSION' ) ) {
				global $sitepress;
				$wpml_columns = new WPML_Custom_Columns( $sitepress );
				$columns      = $wpml_columns->add_posts_management_column( $columns );
			}

			/**
			 * Reverse the array for RTL
			 */
			if ( is_rtl() ) {
				$columns = array_reverse( $columns );
			}

			return $columns;
		}

		/**
		 * Setting invoice columns values.
		 *
		 * @param string $column_name Name of the current column.
		 */
		public function invoice_columns_values( $column_name ) {

			global $post, $post_type, $pagenow;

			if ( 'edit.php' === $pagenow && 'invoice' === $post_type ) {

				$meta_data = get_post_custom( $post->ID );

				switch ( $column_name ) {

					case 'booking':
						echo ( ! empty( $meta_data['booking_id'] ) ) ? $meta_data['booking_id'][0] : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'payer':
						echo get_avatar( $meta_data['payer_email'][0], 64 ) . '<br>';
						echo ( ! empty( $meta_data['first_name'] ) ) ? $meta_data['first_name'][0] : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						echo ( ! empty( $meta_data['last_name'] ) ) ? ' ' . $meta_data['last_name'][0] : '';
						break;
					case 'payer_email':
						echo ( ! empty( $meta_data['payer_email'] ) ) ? $meta_data['payer_email'][0] : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'amount':
						echo ( ! empty( $meta_data['payment_amount'] ) ) ? $meta_data['payment_amount'][0] : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'currency':
						echo ( ! empty( $meta_data['amount_currency'] ) ) ? ' ' . $meta_data['amount_currency'][0] : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'method':
						echo ( ! empty( $meta_data['payment_method'] ) ) ? ucfirst( $meta_data['payment_method'][0] ) : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'payment_date':
						the_date( 'Y/m/d' );
						echo '<br>';
						the_time( 'g:i a' );
						break;

					default:
						break;
				}
			}
		}
	}
}
