<?php
/**
 * This class is responsible for the Booking post type and related stuff
 *
 * @package    realhomes_vacation_rentals
 * @subpackage realhomes_vacation_rentals/admin
 */

if ( ! class_exists( 'RVR_Booking' ) ) {
	/**
	 * Class RVR_Booking
	 *
	 * Responsible for all stuff related to Booking Post Type.
	 *
	 * @package    realhomes_vacation_rentals
	 * @subpackage realhomes_vacation_rentals/admin
	 */
	class RVR_Booking {

		/**
		 * Register booking custom post type
		 */
		public function rvr_booking_post_type() {

			$labels = array(
				'name'                  => esc_html_x( 'Bookings', 'Post Type General Name', RVR_TEXT_DOMAIN ),
				'singular_name'         => esc_html_x( 'Booking', 'Post Type Singular Name', RVR_TEXT_DOMAIN ),
				'menu_name'             => esc_html__( 'Bookings', RVR_TEXT_DOMAIN ),
				'name_admin_bar'        => esc_html__( 'Booking', RVR_TEXT_DOMAIN ),
				'archives'              => esc_html__( 'Booking Archives', RVR_TEXT_DOMAIN ),
				'attributes'            => esc_html__( 'Booking Attributes', RVR_TEXT_DOMAIN ),
				'parent_item_colon'     => esc_html__( 'Parent Booking:', RVR_TEXT_DOMAIN ),
				'all_items'             => esc_html__( 'All Bookings', RVR_TEXT_DOMAIN ),
				'add_new_item'          => esc_html__( 'Add New Booking', RVR_TEXT_DOMAIN ),
				'add_new'               => esc_html__( 'Add New', RVR_TEXT_DOMAIN ),
				'new_item'              => esc_html__( 'New Booking', RVR_TEXT_DOMAIN ),
				'edit_item'             => esc_html__( 'Edit Booking', RVR_TEXT_DOMAIN ),
				'update_item'           => esc_html__( 'Update Booking', RVR_TEXT_DOMAIN ),
				'view_item'             => esc_html__( 'View Booking', RVR_TEXT_DOMAIN ),
				'view_items'            => esc_html__( 'View Bookings', RVR_TEXT_DOMAIN ),
				'search_items'          => esc_html__( 'Search Booking', RVR_TEXT_DOMAIN ),
				'not_found'             => esc_html__( 'Not found', RVR_TEXT_DOMAIN ),
				'not_found_in_trash'    => esc_html__( 'Not found in Trash', RVR_TEXT_DOMAIN ),
				'featured_image'        => esc_html__( 'Featured Image', RVR_TEXT_DOMAIN ),
				'set_featured_image'    => esc_html__( 'Set featured image', RVR_TEXT_DOMAIN ),
				'remove_featured_image' => esc_html__( 'Remove featured image', RVR_TEXT_DOMAIN ),
				'use_featured_image'    => esc_html__( 'Use as featured image', RVR_TEXT_DOMAIN ),
				'insert_into_item'      => esc_html__( 'Insert into booking', RVR_TEXT_DOMAIN ),
				'uploaded_to_this_item' => esc_html__( 'Uploaded to this booking', RVR_TEXT_DOMAIN ),
				'items_list'            => esc_html__( 'Bookings list', RVR_TEXT_DOMAIN ),
				'items_list_navigation' => esc_html__( 'Bookings list navigation', RVR_TEXT_DOMAIN ),
				'filter_items_list'     => esc_html__( 'Filter bookings list', RVR_TEXT_DOMAIN ),
			);
			$args   = array(
				'label'               => esc_html__( 'Booking', RVR_TEXT_DOMAIN ),
				'description'         => esc_html__( 'Bookings request for the rental properties.', RVR_TEXT_DOMAIN ),
				'labels'              => $labels,
				'supports'            => array( 'title' ),
				'hierarchical'        => false,
				'public'              => true,
				'show_ui'             => true,
				'show_in_menu'        => 'rvr',
				'menu_position'       => 5,
				'menu_icon'           => 'dashicons-calendar',
				'show_in_admin_bar'   => false,
				'show_in_nav_menus'   => false,
				'can_export'          => false,
				'has_archive'         => false,
				'exclude_from_search' => true,
				'publicly_queryable'  => false,
				'rewrite'             => false,
				'capability_type'     => 'post',
				'capabilities'        => array(
					'create_posts' => 'do_not_allow',
				),
				'map_meta_cap'        => true,

				// Disabled REST API Support for Bookings
				//'show_in_rest'        => true,
				//'rest_base'           => apply_filters( 'rvr_booking_rest_base', esc_html__( 'bookings', RVR_TEXT_DOMAIN ) )
			);
			register_post_type( 'booking', $args );

		}

		/**
		 * Replace "Enter Title Here" placeholder text.
		 *
		 * @param string $title Default title text.
		 *
		 * @return string
		 */
		public function booking_title_text( $title ) {
			$screen = get_current_screen();

			if ( 'booking' === $screen->post_type ) {
				$title = esc_html__( 'Enter booking ID here', RVR_TEXT_DOMAIN );
			}

			return $title;
		}

		/**
		 * Register booking post type metaboxes
		 *
		 * @param $meta_boxes
		 *
		 * @return array|mixed|void
		 */
		public function rvr_booking_meta_boxes( $meta_boxes ) {

			$prefix = 'rvr_';

			$booking_id = false;
			if ( isset( $_GET['post'] ) ) {
				$booking_id = intval( $_GET['post'] );
			} else if ( isset( $_POST['post_ID'] ) ) {
				$booking_id = intval( $_POST['post_ID'] );
			}

			// Prepare invoice detail.
			$invoice_id = get_post_meta( $booking_id, 'rvr_invoice_id', true );
			$invoice_exist = ! empty( $invoice_id ) && ! empty( get_the_title( $invoice_id ) );

			if ( $invoice_exist ) {
				$invoice_info = 'Booking Invoice ID: <a href="' . get_edit_post_link( $invoice_id ) . '">' . get_the_title( $invoice_id ) . '</a>';
			} else {
				$invoice_info = esc_html__( 'Invoice is not available for this booking.', RVR_TEXT_DOMAIN );
			}

			// Prepare additional amenities detail.
			$additional_amenities      = get_post_meta( $booking_id, 'rvr_additional_amenities_paid', true );
			$additional_amenities_info = esc_html__( 'No additional amenities are added to this booking.', RVR_TEXT_DOMAIN );

			if ( ! empty( $additional_amenities ) && is_array( $additional_amenities ) ) {
				$additional_amenities_info = '<div class="rvr-additional-fees-info"><h3>' . esc_html__( 'Additional Amenities Details', RVR_TEXT_DOMAIN ) . '</h3><table>';

				foreach ( $additional_amenities as $amenity_name => $amenity_amount ) {
					$additional_amenities_info .= "<tr><td><strong>{$amenity_name} </strong></td><td>" . ere_format_amount( $amenity_amount ) . "</td>";
				}

				$additional_amenities_info .= '</table></div>';
			}

			// Prepare additional fees detail.
			$additional_fees      = get_post_meta( $booking_id, 'rvr_additional_fees_paid', true );
			$additional_fees_info = esc_html__( 'No additional fees are added to this booking.', RVR_TEXT_DOMAIN );

			if ( ! empty( $additional_fees ) && is_array( $additional_fees ) ) {
				$additional_fees_info = '<div class="rvr-additional-fees-info"><h3>' . esc_html__( 'Additional Fees Details', RVR_TEXT_DOMAIN ) . '</h3><table>';

				foreach ( $additional_fees as $fee_name => $fee_amount ) {
					$additional_fees_info .= "<tr><td><strong>{$fee_name} </strong></td><td>" . ere_format_amount( $fee_amount ) . "</td>";
				}

				$additional_fees_info .= '</table></div>';
			}

			// Invoice Details
			$invoice_details_info = '<div class="rvr-invoice-details">';
			$invoice_details_info .= '<h3>' . esc_html__( 'Invoice Details', RVR_TEXT_DOMAIN ) . '</h3>';
			$invoice_details_info .= '<table>';

			if ( ! $invoice_exist ) {
				$invoice_details_info .= '<tr>';
				$invoice_details_info .= '<td>';
				$invoice_details_info .= '<a type="button" class="rwmb-button generate_booking_invoice button button-danger button hide-if-no-js" data-booking-id="' . esc_attr( $booking_id ) . '">' . esc_html__( 'Generate Invoice', RVR_TEXT_DOMAIN ) . '</a>';
				$invoice_details_info .= '<div class="ere-booking-message-container"></div>';
				$invoice_details_info .= '</td>';
				$invoice_details_info .= '</tr>';
			} else {
				$invoice_details_info .= '<tr>';
				$invoice_details_info .= '<td>';
				$invoice_details_info .= '<p>' . esc_html__( 'Booking Invoice ID: ', RVR_TEXT_DOMAIN ) . '<strong>' . esc_html( $invoice_id ) . '</strong></p>';
				$invoice_details_info .= '<div class="ere-booking-message-container"></div>';
				$invoice_details_info .= '</td>';
				$invoice_details_info .= '</tr>';
			}

			$invoice_details_info .= '</table>';
			$invoice_details_info .= '</div>';

			$rvr_booking_fields = array();

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}booking_status",
				'name'    => esc_html__( 'Booking Status', RVR_TEXT_DOMAIN ),
				'type'    => 'select',
				'options' => array(
					'pending'   => esc_html__( 'Pending', RVR_TEXT_DOMAIN ),
					'rejected'  => esc_html__( 'Rejected', RVR_TEXT_DOMAIN ),
					'cancelled' => esc_html__( 'Cancelled', RVR_TEXT_DOMAIN ),
					'confirmed' => esc_html__( 'Confirmed', RVR_TEXT_DOMAIN ),
				),
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}check_in",
				'name'    => esc_html__( 'Check In Date', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}check_out",
				'name'    => esc_html__( 'Check Out Date', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}adult",
				'name'    => esc_html__( 'Adults', RVR_TEXT_DOMAIN ),
				'type'    => 'number',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}child",
				'name'    => esc_html__( 'Children', RVR_TEXT_DOMAIN ),
				'type'    => 'number',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'         => "{$prefix}request_timestamp",
				'name'       => esc_html__( 'Request Date & Time', RVR_TEXT_DOMAIN ),
				'type'       => 'text',
				'columns'    => 4,
				'attributes' => array(
					'readonly' => true,
				)
			);

			$rvr_booking_fields[] = array(
				'type' => 'heading',
				'desc' => esc_html__( 'Renter Info', RVR_TEXT_DOMAIN )
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}renter_name",
				'name'    => esc_html__( 'Name', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}renter_email",
				'name'    => esc_html__( 'Email', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}renter_phone",
				'name'    => esc_html__( 'Phone', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'type' => 'heading',
				'desc' => esc_html__( 'Pricing', RVR_TEXT_DOMAIN )
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}staying_nights",
				'name'    => esc_html__( 'Staying Nights', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}price_per_night",
				'name'    => esc_html__( 'Price Per Night', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}price_staying_nights",
				'name'    => esc_html__( 'Staying Nights Price', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}extra_guests_cost",
				'name'    => esc_html__( 'Extra Guests Charges', RVR_TEXT_DOMAIN ) . ' <small>(' . esc_html( get_post_meta( $booking_id, 'rvr_extra_guests', true ) ) . ')</small> ',
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}services_charges",
				'name'    => esc_html__( 'Services Charges', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}subtotal_price",
				'name'    => esc_html__( 'Subtotal', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}govt_tax",
				'name'    => esc_html__( 'Government Tax', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'      => "{$prefix}total_price",
				'name'    => esc_html__( 'Total Price', RVR_TEXT_DOMAIN ),
				'type'    => 'text',
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'type' => 'heading',
				'desc' => esc_html__( 'Property Information', RVR_TEXT_DOMAIN )
			);

			$rvr_booking_fields[] = array(
				'id'         => "{$prefix}property_custom_id",
				'name'       => esc_html__( 'Property ID', RVR_TEXT_DOMAIN ),
				'type'       => 'text',
				'columns'    => 4,
				'attributes' => array(
					'readonly' => true,
				)
			);

			$rvr_booking_fields[] = array(
				'id'         => "{$prefix}property_title",
				'name'       => esc_html__( 'Property Title', RVR_TEXT_DOMAIN ),
				'type'       => 'text',
				'columns'    => 8,
				'attributes' => array(
					'readonly' => true,
				)
			);

			$rvr_booking_fields[] = array(
				'id'         => "{$prefix}property_url",
				'name'       => esc_html__( 'Property URL', RVR_TEXT_DOMAIN ),
				'type'       => 'url',
				'columns'    => 12,
				'attributes' => array(
					'readonly' => true,
				)
			);

			$rvr_booking_fields[] = array(
				'id'   => "{$prefix}additional_amenities_info",
				'type' => 'heading',
				'desc' => $additional_amenities_info,
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'   => "{$prefix}additional_fees_info",
				'type' => 'heading',
				'desc' => $additional_fees_info,
				'columns' => 4
			);

			$rvr_booking_fields[] = array(
				'id'   => "{$prefix}invoice_details_heading",
				'type' => 'heading',
				'desc' => $invoice_details_info,
				'columns' => 4
			);

			$meta_boxes[] = array(
				'id'     => 'booking-meta-box',
				'title'  => esc_html__( 'Booking', RVR_TEXT_DOMAIN ),
				'pages'  => array( 'booking' ),
				'fields' => $rvr_booking_fields
			);

			// apply a filter before returning meta boxes
			$meta_boxes = apply_filters( 'rvr_booking_meta_boxes', $meta_boxes );

			return $meta_boxes;

		}

		/**
		 * Custom columns for bookings
		 *
		 * @param $columns
		 *
		 * @return array
		 */
		public function booking_edit_columns( $columns ) {

			$columns = array(

				'cb'             => '<input type="checkbox" />',
				'title'          => esc_html__( 'Booking ID', RVR_TEXT_DOMAIN ),
				'renter'         => esc_html__( 'Renter', RVR_TEXT_DOMAIN ),
				'renter_info'    => esc_html__( 'Renter Info', RVR_TEXT_DOMAIN ),
				'check_in'       => esc_html__( 'Check In', RVR_TEXT_DOMAIN ),
				'check_out'      => esc_html__( 'Check Out', RVR_TEXT_DOMAIN ),
				'property_title' => esc_html__( 'Booked Property', RVR_TEXT_DOMAIN ),
				'total_price'    => esc_html__( 'Total', RVR_TEXT_DOMAIN ),
				'status'         => esc_html__( 'Status', RVR_TEXT_DOMAIN ),
				'booking_date'   => esc_html__( 'Received Time', RVR_TEXT_DOMAIN ),
			);

			/**
			 * WPML Support
			 */
			if ( defined( 'ICL_SITEPRESS_VERSION' ) ) {
				global $sitepress;
				$wpml_columns = new WPML_Custom_Columns( $sitepress );
				$columns      = $wpml_columns->add_posts_management_column( $columns );
			}

			/**
			 * Reverse the array for RTL
			 */
			if ( is_rtl() ) {
				$columns = array_reverse( $columns );
			}

			return $columns;
		}

		/**
		 * Setting booking columns values
		 *
		 * @param $column_name
		 */
		public function booking_columns_values( $column_name ) {

			global $post, $post_type, $pagenow;;

			if ( $pagenow == 'edit.php' && $post_type == 'booking' ) {

				$meta_data     = get_post_custom( $post->ID );
				$property_id   = ! empty( $meta_data['rvr_property_id'][0] ) ? $meta_data['rvr_property_id'][0] : '';
				$property_link = '';

				if ( ! empty( $property_id ) ) {

					$property_title = get_the_title( $property_id );

					if ( ! empty( $property_title ) ) {
						$property_link = '<a href="' . get_the_permalink( $property_id ) . '" target="_blank">' . $property_title . '</a>';
					}
				}

				switch ( $column_name ) {

					case 'renter':
						echo ( ! empty( $meta_data['rvr_renter_email'][0] ) ) ? get_avatar( $meta_data['rvr_renter_email'][0], 64 ) . '<br>' : '';
						echo ( ! empty( $meta_data['rvr_renter_name'] ) ) ? esc_html( $meta_data['rvr_renter_name'][0] ) : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'renter_info':
						$not_available = true;

						if ( ! empty( $meta_data['rvr_renter_email'][0] ) ) {
							echo $meta_data['rvr_renter_email'][0];
							$not_available = false;
						}

						if ( ! empty( $meta_data['rvr_renter_phone'][0] ) ) {
							echo "<br><br>";
							echo $meta_data['rvr_renter_phone'][0];
							$not_available = false;
						}
						echo ( $not_available ) ? esc_html__( 'NA', RVR_TEXT_DOMAIN ) : '';
						break;
					case 'phone':
						echo ( ! empty( $meta_data['rvr_renter_phone'] ) ) ? $meta_data['rvr_renter_phone'][0] : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'check_in':
						echo ( ! empty( $meta_data['rvr_check_in'] ) ) ? $meta_data['rvr_check_in'][0] : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'check_out':
						echo ( ! empty( $meta_data['rvr_check_out'] ) ) ? $meta_data['rvr_check_out'][0] : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'property_title':
						echo ( ! empty( $property_link ) ) ? $property_link : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'total_price':
						echo ( ! empty( $meta_data['rvr_total_price'] ) ) ? ere_format_amount( $meta_data['rvr_total_price'][0] ) : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'status':
						echo ( ! empty( $meta_data['rvr_booking_status'] ) ) ? ucfirst( $meta_data['rvr_booking_status'][0] ) : esc_html__( 'NA', RVR_TEXT_DOMAIN );
						break;
					case 'booking_date':
						the_date( 'Y/m/d' );
						echo '<br>';
						the_time( 'g:i a' );
						break;

					default:
						break;
				}
			}
		}

		/**
		 * Make booking `Status` & `Received` columns sortable
		 *
		 * @param $columns
		 *
		 * @return mixed
		 */
		public function booking_sortable_columns( $columns ) {
			$columns['status']       = 'status';
			$columns['booking_date'] = 'booking_date';

			return $columns;
		}

		/**
		 * Add booking `Status` column sorting key
		 *
		 * @param $query
		 */
		public function booking_status_orderby( $query ) {
			if ( ! is_admin() ) {
				return;
			}

			$orderby = $query->get( 'orderby' );

			if ( 'status' == $orderby ) {
				$query->set( 'meta_key', 'rvr_booking_status' );
				$query->set( 'orderby', 'meta_value' );
			}
		}
	}
}
