<?php
/**
 * Trait RHEA_Single_Property_V2_Title_Section_Trait
 *
 * Provides methods for managing and rendering section title controls and styles
 * for single property sections in Elementor.
 *
 * @since 2.4.0
 */
trait RHEA_Single_Property_V2_Title_Section_Trait {

	public $is_old_widget = false;

	/**
	 * Enables support for title variations by toggling the `$is_old_widget` property.
	 *
	 * This method allows widgets to support multiple title variations, particularly for legacy or advanced styling.
	 *
	 * @return void
	 */
	public function enable_title_variation_support() {
		$this->is_old_widget = true;
	}

	/**
	 * Adds content tab controls for the section title in Elementor.
	 *
	 * This method creates controls that allow users to define the content
	 * for the section title, such as text and default values.
	 *
	 * @param string $default_value Default section title value.
	 */
	public function title_section_content_tab_controls( $default_value = '' ) {

		$this->start_controls_section(
			'section_title_content',
			[
				'label' => esc_html__( 'Section Title', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'section_title',
			[
				'label'   => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => $default_value,
			]
		);

		if ( $this->is_old_widget ) {
			$this->add_control(
				'section_title_variation',
				[
					'label'   => esc_html__( 'Variation', RHEA_TEXT_DOMAIN ),
					'type'    => \Elementor\Controls_Manager::SELECT,
					'default' => '1',
					'options' => [
						'1' => esc_html__( 'One', RHEA_TEXT_DOMAIN ),
						'2' => esc_html__( 'Two', RHEA_TEXT_DOMAIN ),
					],
				]
			);
		} else {
			$this->add_control(
				'section_title_variation',
				[
					'label'   => esc_html__( 'Variation', RHEA_TEXT_DOMAIN ),
					'type'    => \Elementor\Controls_Manager::HIDDEN,
					'default' => '2',
				]
			);
		}

		// Finalize section controls.
		$this->end_controls_section();
	}

	/**
	 * Adds style tab controls for the section title in Elementor.
	 *
	 * This method creates styling controls that allow users to customize
	 * the appearance of the section title, including typography, colors,
	 * margins, and vertical line styles.
	 */
	public function title_section_style_tab_controls() {

		$this->start_controls_section(
			'section_title_styles',
			[
				'label' => esc_html__( 'Section Title', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'section_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-section-heading, {{WRAPPER}} .rh_property__heading' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'section_title_typography',
				'label'     => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'    => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector'  => '{{WRAPPER}} .rh-single-property-section-heading, {{WRAPPER}} .rh_property__heading',
				'separator' => 'after',
			]
		);

		$this->add_responsive_control(
			'section_title_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-section-heading, {{WRAPPER}} .rh_property__heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

			$this->add_control(
				'vertical_line_heading',
				[
					'label'     => esc_html__( 'Vertical Line', RHEA_TEXT_DOMAIN ),
					'type'      => \Elementor\Controls_Manager::HEADING,
					'condition' => [ 'section_title_variation' => '2' ],
				]
			);

			$this->add_control(
				'section_title_vertical_line_color',
				[
					'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
					'type'      => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rh-single-property-section-heading:before' => 'background-color: {{VALUE}}',
					],
					'condition' => [ 'section_title_variation' => '2' ],
				]
			);

			$this->add_responsive_control(
				'section_title_vertical_line_space',
				[
					'label'     => esc_html__( 'Space', RHEA_TEXT_DOMAIN ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .rh-single-property-section-heading-common' => 'gap: {{SIZE}}px',
					],
					'condition' => [ 'section_title_variation' => '2' ],
				]
			);

			$this->add_responsive_control(
				'section_title_vertical_line_width',
				[
					'label'     => esc_html__( 'Width', RHEA_TEXT_DOMAIN ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .rh-single-property-section-heading:before' => 'width: {{SIZE}}px',
					],
					'condition' => [ 'section_title_variation' => '2' ],
				]
			);

			$this->add_responsive_control(
				'section_title_vertical_line_height',
				[
					'label'     => esc_html__( 'Height', RHEA_TEXT_DOMAIN ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .rh-single-property-section-heading:before' => 'height: {{SIZE}}px',
					],
					'condition' => [ 'section_title_variation' => '2' ],
				]
			);

			$this->add_responsive_control(
				'section_title_vertical_line_border_radius',
				[
					'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min'  => 0,
							'max'  => 50,
							'step' => 1,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .rh-single-property-section-heading:before' => 'border-radius: {{SIZE}}px',
					],
					'condition' => [ 'section_title_variation' => '2' ],
				]
			);

			$this->add_responsive_control(
				'section_title_vertical_line_margin',
				[
					'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
					'type'      => \Elementor\Controls_Manager::DIMENSIONS,
					'selectors' => [
						'{{WRAPPER}} .rh-single-property-section-heading:before' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'condition' => [ 'section_title_variation' => '2' ],
				]
			);

		// Finalize section controls.
		$this->end_controls_section();
	}

	/**
	 * Renders a title section with dynamic HTML classes and content.
	 *
	 * @param array $settings
	 *
	 * @return void Outputs the HTML for the title section or returns early if no title is provided.
	 */
	public function render_section_title( $settings ) {

		if ( empty( $settings['section_title'] ) ) {
			return; // Exit early if no title is provided
		}

		if ( '2' === $settings['section_title_variation'] ) {
			$container_classes = 'rh-single-property-section-heading rh-single-property-section-heading-common';
		} else {
			$container_classes = 'rh_property__heading';
		}

		// Print the section title
		printf( '<h4 class="%s">%s</h4>', esc_attr( $container_classes ), esc_html( $settings['section_title'] ) );
	}
}