<?php
/**
 * Handles integration of Elementor Mega Menu within WordPress navigation menus.
 *
 * This class allows administrators to link Elementor templates to top-level
 * WordPress menu items, enabling the creation of mega menus with customizable layouts
 * (default, fullwidth, or custom width). It also manages front-end rendering and caching.
 *
 * @since 4.4.5
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class RHEA_Elementor_Mega_Menu
 *
 * Integrates Elementor library templates as mega menu content within WordPress menus.
 */
class RHEA_Elementor_Mega_Menu {

	/**
	 * Whether WPML (multilingual plugin) is active.
	 *
	 * @var bool
	 */
	private $wpml_is_active = false;

	/**
	 * Constructor: Hooks into admin and front-end actions.
	 */
	public function __construct() {

		if ( is_admin() ) {
			// Admin-side hooks for menu item field management.
			add_action( 'wp_nav_menu_item_custom_fields', [ $this, 'add_select_post_dropdown' ], 10, 5 );
			add_action( 'wp_update_nav_menu_item', [ $this, 'save_selected_post_id' ], 10, 2 );
		} else {
			// Front-end hooks for menu rendering and styling.
			add_filter( 'walker_nav_menu_start_el', [ $this, 'inject_selected_post_content' ], 10, 4 );
			add_filter( 'nav_menu_css_class', [ $this, 'add_custom_li_classes' ], 10, 4 );
		}

		// Clear cached Elementor library data on updates.
		add_action( 'wp_update_nav_menu', [ __CLASS__, 'clear_elementor_library_cache' ] );
		add_action( 'customize_save_after', [ __CLASS__, 'clear_elementor_library_cache' ] );
		add_action( 'save_post_elementor_library', [ __CLASS__, 'clear_elementor_library_cache' ] );
		add_action( 'delete_post_elementor_library', [ __CLASS__, 'clear_elementor_library_cache' ] );

		// Detect WPML availability.
		if ( function_exists( 'rhea_wpml_is_active' ) ) {
			$this->wpml_is_active = rhea_wpml_is_active();
		}
	}

	/**
	 * Get Elementor plugin instance.
	 *
	 * @return \Elementor\Plugin
	 */
	public static function elementor_plugin() {
		return \Elementor\Plugin::$instance;
	}

	/**
	 * Clears the Elementor library transient cache.
	 *
	 * This ensures that any updates to templates or menus
	 * are reflected in dropdowns without delay.
	 *
	 * @return void
	 */
	public static function clear_elementor_library_cache() {
		delete_transient( 'rhea_elementor_library_options' );

		// Clear object cache if available.
		if ( function_exists( 'wp_cache_delete' ) ) {
			wp_cache_delete( 'rhea_elementor_library_options', 'transient' );
		}
	}

	/**
	 * Retrieves Elementor library templates for the dropdown.
	 *
	 * @return array Associative array of [template_id => template_title].
	 */
	public static function get_elementor_library() {
		$cache_key    = 'rhea_elementor_library_options';
		$return_array = get_transient( $cache_key );

		if ( false === $return_array ) {
			$return_array = [ 'none' => esc_html__( 'None', RH_TEXT_DOMAIN ) ];
			$query        = get_posts( [
				'post_type'      => 'elementor_library',
				'posts_per_page' => 150,
				'post_status'    => [ 'publish' ],
				'fields'         => 'ids',
			] );

			foreach ( $query as $post_id ) {
				$return_array[ $post_id ] = get_the_title( $post_id );
			}

			set_transient( $cache_key, $return_array, HOUR_IN_SECONDS );
		}

		return $return_array;
	}

	/**
	 * Adds custom CSS classes to menu items containing a mega menu.
	 *
	 * @param array   $classes Array of CSS classes for the menu item.
	 * @param WP_Post $item    The current menu item object.
	 * @param object  $args    Nav menu arguments.
	 * @param int     $depth   Menu depth level.
	 *
	 * @return array Modified CSS classes.
	 */
	public function add_custom_li_classes( $classes, $item, $args, $depth ) {
		if ( 0 === $depth ) {
			$linked_id = get_post_meta( $item->ID, '_menu_item_mega_menu', true );

			if ( ! empty( $linked_id ) && 'none' !== $linked_id ) {
				$classes[] = 'has-realhomes-mega-menu';

				$layout = get_post_meta( $item->ID, '_menu_item_mega_menu_layout', true );
				switch ( $layout ) {
					case 'fullwidth':
						$classes[] = 'realhomes-mega-menu-fullwidth';
						break;
					case 'custom':
						$classes[] = 'realhomes-mega-menu-custom-width';
						break;
					default:
						$classes[] = 'realhomes-mega-menu-default-width';
				}
			}
		}

		return $classes;
	}

	/**
	 * Adds custom dropdown fields in the menu editor (admin side).
	 *
	 * @param int    $item_id           Menu item ID.
	 * @param object $menu_item         Menu item data object.
	 * @param int    $depth             Menu depth.
	 * @param array  $args              Menu arguments.
	 * @param int    $current_object_id Current object ID.
	 *
	 * @return void
	 */
	public function add_select_post_dropdown( $item_id, $menu_item, $depth, $args, $current_object_id ) {
		// Only for top-level menu items.
		if ( ! empty( $menu_item->menu_item_parent ) ) {
			return;
		}

		$saved_post_id = get_post_meta( $item_id, '_menu_item_mega_menu', true );
		$layout_option = get_post_meta( $item_id, '_menu_item_mega_menu_layout', true );
		$custom_width  = get_post_meta( $item_id, '_menu_item_mega_menu_custom_width', true );
		$posts         = self::get_elementor_library();

		ob_start();
		?>
        <p class="field-linked-post description description-wide">
            <label for="edit-menu-item-linked-post-<?php echo esc_attr( $item_id ); ?>">
				<?php esc_html_e( 'Select Elementor Mega Menu Template', RHEA_TEXT_DOMAIN ); ?><br>
                <select id="edit-menu-item-linked-post-<?php echo esc_attr( $item_id ); ?>" class="widefat code" name="menu-item-linked-post[<?php echo esc_attr( $item_id ); ?>]">
					<?php foreach ( $posts as $id => $title ) : ?>
                        <option value="<?php echo esc_attr( $id ); ?>" <?php selected( $saved_post_id, $id ); ?>>
							<?php echo esc_html( $title ); ?>
                        </option>
					<?php endforeach; ?>
                </select>
            </label>
        </p>

        <p class="field-fullwidth description description-wide">
            <label for="edit-menu-item-layout-<?php echo esc_attr( $item_id ); ?>">
				<?php esc_html_e( 'Mega Menu Layout', RHEA_TEXT_DOMAIN ); ?><br>
                <select id="edit-menu-item-layout-<?php echo esc_attr( $item_id ); ?>" class="widefat code" name="menu-item-layout[<?php echo esc_attr( $item_id ); ?>]">
                    <option value="default" <?php selected( $layout_option, 'default' ); ?>><?php esc_html_e( 'Default', RHEA_TEXT_DOMAIN ); ?></option>
                    <option value="fullwidth" <?php selected( $layout_option, 'fullwidth' ); ?>><?php esc_html_e( 'Fullwidth', RHEA_TEXT_DOMAIN ); ?></option>
                    <option value="custom" <?php selected( $layout_option, 'custom' ); ?>><?php esc_html_e( 'Custom Width', RHEA_TEXT_DOMAIN ); ?></option>
                </select>
            </label>
        </p>

        <p class="field-custom-width description description-wide" style="<?php echo ( 'fullwidth' !== $layout_option ) ? '' : 'display:none;'; ?>">
            <label for="edit-menu-item-custom-width-<?php echo esc_attr( $item_id ); ?>">
				<?php esc_html_e( 'Custom Width (px)', RHEA_TEXT_DOMAIN ); ?><br>
                <input type="number" id="edit-menu-item-custom-width-<?php echo esc_attr( $item_id ); ?>" class="widefat code" name="menu-item-custom-width[<?php echo esc_attr( $item_id ); ?>]" value="<?php echo esc_attr( $custom_width ); ?>" />
            </label>
        </p>

        <script>
            ( function ( $ ) {
                $( '#edit-menu-item-layout-<?php echo esc_js( $item_id ); ?>' ).on( 'change', function () {
                    const val = $( this ).val();
                    $( '#edit-menu-item-custom-width-<?php echo esc_js( $item_id ); ?>' ).closest( 'p' ).toggle( val !== 'fullwidth' );
                } );
            } )( jQuery );
        </script>
		<?php
		echo ob_get_clean();
	}

	/**
	 * Saves selected Elementor template, layout, and width for a menu item.
	 *
	 * @param int $menu_id         Menu ID.
	 * @param int $menu_item_db_id Menu item ID.
	 *
	 * @return void
	 */
	public function save_selected_post_id( $menu_id, $menu_item_db_id ) {
		if ( ! current_user_can( 'edit_theme_options' ) ) {
			return;
		}

		// Save linked Elementor template.
		if ( isset( $_POST['menu-item-linked-post'][ $menu_item_db_id ] ) ) {
			$post_id = intval( $_POST['menu-item-linked-post'][ $menu_item_db_id ] );
			update_post_meta( $menu_item_db_id, '_menu_item_mega_menu', $post_id );
		} else {
			delete_post_meta( $menu_item_db_id, '_menu_item_mega_menu' );
		}

		// Save layout type.
		if ( isset( $_POST['menu-item-layout'][ $menu_item_db_id ] ) ) {
			$layout = sanitize_text_field( $_POST['menu-item-layout'][ $menu_item_db_id ] );
			update_post_meta( $menu_item_db_id, '_menu_item_mega_menu_layout', $layout );
		} else {
			delete_post_meta( $menu_item_db_id, '_menu_item_mega_menu_layout' );
		}

		// Save custom width if provided.
		if ( ! empty( $_POST['menu-item-custom-width'][ $menu_item_db_id ] ) ) {
			$width = absint( $_POST['menu-item-custom-width'][ $menu_item_db_id ] );
			update_post_meta( $menu_item_db_id, '_menu_item_mega_menu_custom_width', $width );
		} else {
			delete_post_meta( $menu_item_db_id, '_menu_item_mega_menu_custom_width' );
		}
	}

	/**
	 * Injects Elementor template content into menu output on the front-end.
	 *
	 * @param string  $item_output Menu item output.
	 * @param WP_Post $item        Menu item object.
	 * @param int     $depth       Menu depth.
	 * @param object  $args        Menu arguments.
	 *
	 * @return string Modified menu output.
	 */
	public function inject_selected_post_content( $item_output, $item, $depth, $args ) {
		$item_id   = $item->ID;
		$linked_id = get_post_meta( $item_id, '_menu_item_mega_menu', true );

		if ( empty( $linked_id ) || 'none' === $linked_id ) {
			return $item_output;
		}

		// Handle WPML translation.
		if ( $this->wpml_is_active ) {
			$current_language = apply_filters( 'wpml_current_language', null );
			$linked_id        = apply_filters( 'wpml_object_id', $linked_id, 'elementor_library', false, $current_language );
		}

		// Retrieve Elementor content.
		if ( $linked_id ) {
			$content = self::elementor_plugin()->frontend->get_builder_content_for_display( $linked_id );

			if ( $content ) {
				$styles = '';
				$layout = get_post_meta( $item_id, '_menu_item_mega_menu_layout', true );

				// Apply custom width if defined.
				if ( ! empty( $layout ) && 'fullwidth' !== $layout ) {
					$width = get_post_meta( $item_id, '_menu_item_mega_menu_custom_width', true );

					if ( ! empty( $width ) ) {
						$styles = sprintf( 'style="--realhomes-mega-menu-custom-width:%dpx;"', absint( $width ) );
					}
				}

				$item_output .= '<div class="realhomes-mega-menu realhomes-mega-sub-menu" ' . $styles . '>' . $content . '</div>';
			}
		}

		return $item_output;
	}
}

new RHEA_Elementor_Mega_Menu();
