<?php
/**
 * Single Property Sub Properties Widget V1.
 *
 * @since 2.4.5
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Single_Property_Sub_Properties_V1 extends \Elementor\Widget_Base {

	use RHEA_Single_Property_V2_Title_Section_Trait;

	private static $is_rvr_enabled;

	private static $property_id;

	public function __construct( array $data = [], ?array $args = null ) {
		parent::__construct( $data, $args );

		if ( is_null( self::$is_rvr_enabled ) ) {
			self::$is_rvr_enabled = rhea_is_rvr_enabled();
		}

		// Initialize static properties if not already set.
		if ( is_null( self::$property_id ) ) {
			self::$property_id = $this->initialize_property_id();
		}
	}

	public function get_name() {
		return 'rh-single-property-sub-properties-v1';
	}

	public function get_title() {
		return esc_html__( 'RH: Single Property Sub Properties V1', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		// More classes for icons can be found at https://pojome.github.io/elementor-icons/
		return 'eicon-posts-grid rh-ultra-widget';
	}

	public function get_categories() {
		return [ 'ultra-real-homes' ];
	}

	protected function register_controls() {

		$this->title_section_content_tab_controls( esc_html__( 'Sub Properties', RHEA_TEXT_DOMAIN ) );

		$this->start_controls_section(
			'properties_section',
			[
				'label' => esc_html__( 'Properties', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'layout',
			[
				'label'              => esc_html__( 'Layout', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'options'            => [
					'list'  => esc_html__( 'List', RHEA_TEXT_DOMAIN ),
					'table' => esc_html__( 'Table', RHEA_TEXT_DOMAIN ),
				],
				'default'            => 'list',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'orderby',
			[
				'label'              => esc_html__( 'Order By', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'options'            => [
					'date'        => esc_html__( 'Time - Recent First', RHEA_TEXT_DOMAIN ),
					'low-to-high' => esc_html__( 'Price - Low to High', RHEA_TEXT_DOMAIN ),
					'high-to-low' => esc_html__( 'Price - High to Low', RHEA_TEXT_DOMAIN ),
					'random'      => esc_html__( 'Random', RHEA_TEXT_DOMAIN ),
				],
				'default'            => 'date',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'posts_per_page',
			[
				'label'              => esc_html__( 'Properties Per Page', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::NUMBER,
				'min'                => 1,
				'max'                => 12,
				'step'               => 1,
				'default'            => 6,
				'frontend_available' => true,
			]
		);

		// Table Fields Visibility Controls
		$this->add_control(
			'table_fields_visibility_heading',
			[
				'label'     => esc_html__( 'Table Fields Visibility', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'show_property_type',
			[
				'label'        => esc_html__( 'Show Property Type', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'layout' => 'table',
				],
			]
		);

		$this->add_control(
			'show_meta_fields',
			[
				'label'        => esc_html__( 'Show Meta Fields', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'layout' => 'table',
				],
			]
		);

		$this->add_control(
			'show_availability_date_field',
			[
				'label'        => esc_html__( 'Show Availability Date', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => [
					'layout' => 'table',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'meta_settings',
			[
				'label'     => esc_html__( 'Meta Settings', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_CONTENT,
				'condition' => [
					'layout' => 'list',
				],
			]
		);

		$get_meta = array(
			'bedrooms'   => esc_html__( 'Bedrooms', RHEA_TEXT_DOMAIN ),
			'bathrooms'  => esc_html__( 'Bathrooms', RHEA_TEXT_DOMAIN ),
			'area'       => esc_html__( 'Area', RHEA_TEXT_DOMAIN ),
			'garage'     => esc_html__( 'Garages/Parking', RHEA_TEXT_DOMAIN ),
			'year-built' => esc_html__( 'Year Built', RHEA_TEXT_DOMAIN ),
			'lot-size'   => esc_html__( 'Lot Size', RHEA_TEXT_DOMAIN ),
		);

		$meta_defaults = array(
			array(
				'rhea_property_meta_display' => 'bedrooms',
				'rhea_meta_repeater_label'   => esc_html__( 'Bedrooms', RHEA_TEXT_DOMAIN ),
			),
			array(
				'rhea_property_meta_display' => 'bathrooms',
				'rhea_meta_repeater_label'   => esc_html__( 'Bathrooms', RHEA_TEXT_DOMAIN ),
			),
			array(
				'rhea_property_meta_display' => 'area',
				'rhea_meta_repeater_label'   => esc_html__( 'Area', RHEA_TEXT_DOMAIN ),
			),
		);

		if ( self::$is_rvr_enabled ) {
			$get_meta['guests']   = esc_html__( 'Guests Capacity', RHEA_TEXT_DOMAIN );
			$get_meta['min-stay'] = esc_html__( 'Min Stay', RHEA_TEXT_DOMAIN );

			$meta_defaults[] = array(
				'rhea_property_meta_display' => 'guests',
				'rhea_meta_repeater_label'   => esc_html__( 'Guests', RHEA_TEXT_DOMAIN ),
			);
		}

		$get_meta = apply_filters( 'rhea_custom_fields_meta_icons', $get_meta );

		$meta_repeater = new \Elementor\Repeater();

		$meta_repeater->add_control(
			'rhea_property_meta_display',
			[
				'label'              => esc_html__( 'Select Meta', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'default'            => '',
				'options'            => $get_meta,
				'frontend_available' => true,
			]
		);

		$meta_repeater->add_control(
			'rhea_meta_repeater_label',
			[
				'label'   => esc_html__( 'Meta Label', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Add Label', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'rhea_add_meta_select',
			[
				'label'              => esc_html__( 'Add Meta', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::REPEATER,
				'fields'             => $meta_repeater->get_controls(),
				'default'            => $meta_defaults,
				'title_field'        => ' {{{ rhea_meta_repeater_label }}}',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		$this->title_section_style_tab_controls();

		$this->start_controls_section(
			'property_card_section',
			[
				'label'     => esc_html__( 'Property Card', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'layout' => [ 'list' ],
				],
			]
		);

		$this->add_responsive_control(
			'property_card_spacing',
			[
				'label'     => esc_html__( 'Cards Spacing', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-list-layout' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'property_list_card_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-sub-property-card'                                      => 'border-radius: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-list-layout .rh-property-thumb-wrapper,
					 {{WRAPPER}} .rh-single-property-sub-properties-v1-list-layout .rh-property-thumb-wrapper > a'      => 'border-radius: {{SIZE}}{{UNIT}} 0 0 {{SIZE}}{{UNIT}};',
					'.rtl {{WRAPPER}} .rh-single-property-sub-properties-v1-list-layout .rh-property-thumb-wrapper,
					 .rtl {{WRAPPER}} .rh-single-property-sub-properties-v1-list-layout .rh-property-thumb-wrapper > a' => 'border-radius: 0 {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}} 0;',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'property_card_box_shadow',
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-sub-property-card',
			]
		);

		$this->add_control(
			'property_content_wrapper_heading',
			[
				'label' => esc_html__( 'Content Wrapper', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_responsive_control(
			'property_card_content_wrapper_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-sub-property-card-inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'property_card_content_bg_color',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}}  .rh-single-property-sub-properties-v1 .rh-property-sub-property-card' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'property_title_heading',
			[
				'label' => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-sub-properties-v1 .property-title',
			]
		);

		$this->add_control(
			'property_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .property-title a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_title_hover_color',
			[
				'label'     => esc_html__( 'Color Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .property-title a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'property_header_bottom_border_width',
			[
				'label'     => esc_html__( 'Bottom Border Width', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-sub-property-card-header' => 'border-bottom-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'property_header_bottom_border_color',
			[
				'label'     => esc_html__( 'Bottom Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-sub-property-card-header' => 'border-bottom-color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'property_title_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .property-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'property_price_heading',
			[
				'label' => esc_html__( 'Property Price', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_price_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .ere-price-display,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .property-current-price
				',
			]
		);

		$this->add_control(
			'property_price_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .ere-price-display,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .property-current-price
					 ' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_old_price_typography',
				'label'    => esc_html__( 'Old Price, Pre, and Postfix', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '
				 {{WRAPPER}} .rh-single-property-sub-properties-v1 .property-old-price,
				 {{WRAPPER}} .rh-single-property-sub-properties-v1 .ere-price-prefix,
				 {{WRAPPER}} .rh-single-property-sub-properties-v1 .ere-price-slash,
				 {{WRAPPER}} .rh-single-property-sub-properties-v1 .ere-price-postfix
				 ',
			]
		);

		$this->add_control(
			'property_old_price_color',
			[
				'label'     => esc_html__( 'Old Price, Pre, and Postfix Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
				 {{WRAPPER}} .rh-single-property-sub-properties-v1 .property-old-price,
				 {{WRAPPER}} .rh-single-property-sub-properties-v1 .ere-price-prefix,
				 {{WRAPPER}} .rh-single-property-sub-properties-v1 .ere-price-slash,
				 {{WRAPPER}} .rh-single-property-sub-properties-v1 .ere-price-postfix
				 ' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_added_date_heading',
			[
				'label' => esc_html__( 'Added Date', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_added_date_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-added-date',
			]
		);

		$this->add_control(
			'property_added_date_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-added-date' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_meta_heading',
			[
				'label' => esc_html__( 'Meta', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_meta_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .figure,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .rhea-meta-labels,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .label,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .rhea-meta-icons-labels,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .figure,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .rhea-meta-labels
				',
			]
		);

		$this->add_control(
			'property_meta_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .figure,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .rhea-meta-labels,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .label,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .rhea-meta-icons-labels,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .figure,
				{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .rhea-meta-labels
					' => 'color: {{VALUE}}
				',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_meta_postfix_typography',
				'label'    => esc_html__( 'Postfix Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .label',
			]
		);

		$this->add_control(
			'property_meta_postfix_color',
			[
				'label'     => esc_html__( 'Postfix Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_meta_svg_color',
			[
				'label'     => esc_html__( 'SVG Icon Fill', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra svg,
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card__meta .rhea_guests,
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rhea_ultra_prop_card__meta .rh-ultra-dark,
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rhea_ultra_prop_card__meta svg' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_meta_svg_stroke',
			[
				'label'     => esc_html__( 'SVG Icon Stroke', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra svg path,
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra svg rect,
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_guests path,
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_guests rect,
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta svg path,
					{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra svg rect' => 'stroke: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'property_meta_icon_size',
			[
				'label'     => esc_html__( 'Meta Icon Size (px)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 128,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra svg'                                                 => 'width: {{SIZE}}{{UNIT}} !important;',
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta svg,
					 {{WRAPPER}} .rh-single-property-sub-properties-v1 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta img' => 'width: {{SIZE}}{{UNIT}}; height:auto;',
				],
			]
		);

		$this->add_responsive_control(
			'property_meta_items_spacing',
			[
				'label'     => esc_html__( 'Meta Items Spacing', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh_prop_card_meta_wrap_ultra' => 'column-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'property_type_and_status_heading',
			[
				'label' => esc_html__( 'Property Type and Status Buttons', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_control(
			'property_type_and_status_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-type-and-status a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_type_and_status_hover_color',
			[
				'label'     => esc_html__( 'Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-type-and-status a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_type_and_status_background_color',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-type-and-status a' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_type_and_status_hover_background_color',
			[
				'label'     => esc_html__( 'Background Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-type-and-status a:hover' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_type_and_status_border_color',
			[
				'label'     => esc_html__( 'Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-type-and-status a' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_type_and_status_hover_border_color',
			[
				'label'     => esc_html__( 'Border Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-type-and-status a:hover' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_type_and_status_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-sub-properties-v1 .rh-property-type-and-status a',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'properties_table_section',
			[
				'label'     => esc_html__( 'Properties Table', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'layout' => 'table',
				],
			]
		);

		$this->add_responsive_control(
			'table_border_radius',
			[
				'label'     => esc_html__( 'Table Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout table'                              => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}; overflow: hidden;',
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout thead th:first-child'               => 'border-top-left-radius: {{TOP}}{{UNIT}};',
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout thead th:last-child'                => 'border-top-right-radius: {{RIGHT}}{{UNIT}};',
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout tbody tr:last-child td:first-child' => 'border-bottom-left-radius: {{BOTTOM}}{{UNIT}};',
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout tbody tr:last-child td:last-child'  => 'border-bottom-right-radius: {{LEFT}}{{UNIT}};',
					'body.rtl {{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout thead th:first-child' => 'border-top-left-radius: 0; border-top-right-radius: {{RIGHT}}{{UNIT}};',
					'body.rtl {{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout thead th:last-child'  => 'border-top-right-radius: 0; border-top-left-radius: {{TOP}}{{UNIT}};',
					'body.rtl {{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout tbody tr:last-child td:first-child' => 'border-bottom-left-radius: 0; border-bottom-right-radius: {{LEFT}}{{UNIT}};',
					'body.rtl {{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout tbody tr:last-child td:last-child'  => 'border-bottom-right-radius: 0; border-bottom-left-radius: {{BOTTOM}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'table_border',
				'label'    => esc_html__( 'Table Border', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout table',
			]
		);

		$this->add_control(
			'table_header_heading',
			[
				'label'     => esc_html__( 'Table Header', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'table_header_padding',
			[
				'label'      => esc_html__( 'Header Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout thead th' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'table_header_text_color',
			[
				'label'     => esc_html__( 'Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout thead th' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'table_header_bg_color',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout thead th' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'table_header_border',
				'label'    => esc_html__( 'Header Border', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout thead th',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'table_header_typography',
				'label'     => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'selector'  => '{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout thead th',
				'separator' => 'after',
			]
		);

		$this->add_control(
			'table_body_heading',
			[
				'label'     => esc_html__( 'Table Body', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'table_cell_padding',
			[
				'label'      => esc_html__( 'Cell Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout tbody td' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'table_cell_border',
				'label'    => esc_html__( 'Cell Border', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout tbody td',
			]
		);

		$this->add_control(
			'table_row_bg',
			[
				'label'     => esc_html__( 'Row Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout tbody tr' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'table_row_alternate_bg',
			[
				'label'     => esc_html__( 'Alternate Row Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout tbody tr:nth-child(even)' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'table_row_hover_bg',
			[
				'label'     => esc_html__( 'Row Hover Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout tbody tr:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_text_color',
			[
				'label'     => esc_html__( 'Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout td' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'meta_typography',
				'label'     => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'selector'  => '{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout td',
				'separator' => 'after',
			]
		);

		$this->add_control(
			'title_heading',
			[
				'label'     => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'title_text_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout .property-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_text_hover_color',
			[
				'label'     => esc_html__( 'Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout .property-title:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'title_typography',
				'label'     => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'selector'  => '{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout .property-title',
				'separator' => 'after',
			]
		);

		$this->add_control(
			'price_heading',
			[
				'label'     => esc_html__( 'Price', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'price_text_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout .sub-property-price' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'price_typography',
				'label'     => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'selector'  => '{{WRAPPER}} .rh-single-property-sub-properties-v1-table-layout .sub-property-price',
				'separator' => 'after',
			]
		);

		$this->end_controls_section();


	}

	protected function render() {
		global $settings;

		$settings  = $this->get_settings_for_display();
		$widget_id = $this->get_id();
		$layout    = $settings['layout'];

		$sub_properties_args = array(
			'post_type'      => 'property',
			'post_status'    => 'publish',
			'posts_per_page' => $settings['posts_per_page'],
			'post_parent'    => self::$property_id,
		);

		if ( 'low-to-high' == $settings['orderby'] ) {
			$sub_properties_args['orderby']  = 'meta_value_num';
			$sub_properties_args['meta_key'] = 'REAL_HOMES_property_price';
			$sub_properties_args['order']    = 'ASC';
		} else if ( 'high-to-low' == $settings['orderby'] ) {
			$sub_properties_args['orderby']  = 'meta_value_num';
			$sub_properties_args['meta_key'] = 'REAL_HOMES_property_price';
			$sub_properties_args['order']    = 'DESC';
		} else if ( 'random' == $settings['orderby'] ) {
			$sub_properties_args['orderby'] = 'rand';
		}

		$sub_properties_args  = apply_filters( 'realhomes_children_properties', $sub_properties_args );
		$sub_properties_query = new WP_Query( $sub_properties_args );

		if ( $sub_properties_query->have_posts() ) {
			if ( ! empty( $settings['section_title'] ) ) {
				?>
                <h4 class="rh-single-property-section-heading rh-single-property-section-heading-common"><?php echo esc_html( $settings['section_title'] ); ?></h4>
				<?php
			}
			?>
            <div id="rh-single-property-sub-properties-v1-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-sub-properties-v1 rh-single-property-sub-properties-v1-<?php echo esc_attr( $layout ); ?>-layout">
				<?php
				if ( 'table' === $layout ) {

					$show_property_type     = 'yes' === $settings['show_property_type'];
					$show_meta_fields       = 'yes' === $settings['show_meta_fields'];
					$show_availability_date = 'yes' === $settings['show_availability_date_field'];
					?>
                    <div class="sub-properties-list-table-container">
                        <table class="table sub-properties-list-table">
                            <thead>
                            <tr>
                                <th class="sub-property-title"><?php esc_html_e( 'Title', RHEA_TEXT_DOMAIN ); ?></th>
                                <th class="sub-property-price"><?php esc_html_e( 'Price', RHEA_TEXT_DOMAIN ); ?></th>
								<?php
								if ( $show_property_type ) {
									?>
                                    <th class="sub-property-type"><?php esc_html_e( 'Property Type', RHEA_TEXT_DOMAIN ); ?></th>
									<?php
								}

								if ( $show_meta_fields ) {
									?>
                                    <th class="sub-property-beds"><?php esc_html_e( 'Beds', RHEA_TEXT_DOMAIN ); ?></th>
                                    <th class="sub-property-baths"><?php esc_html_e( 'Baths', RHEA_TEXT_DOMAIN ); ?></th>
                                    <th class="sub-property-size"><?php esc_html_e( 'Area', RHEA_TEXT_DOMAIN ); ?></th>
									<?php
								}

								if ( $show_availability_date ) {
									?>
                                    <th class="sub-property-availability"><?php esc_html_e( 'Availability Date', RHEA_TEXT_DOMAIN ); ?></th>
									<?php
								}
								?>
                            </tr>
                            </thead>
                            <tbody>
							<?php
							while ( $sub_properties_query->have_posts() ) {
								$sub_properties_query->the_post();

								$property_id        = get_the_ID();
								$property_size      = get_post_meta( $property_id, 'REAL_HOMES_property_size', true );
								$size_postfix       = realhomes_get_area_unit( $property_id );
								$property_bedrooms  = get_post_meta( $property_id, 'REAL_HOMES_property_bedrooms', true );
								$property_bathrooms = get_post_meta( $property_id, 'REAL_HOMES_property_bathrooms', true );
								?>
                                <tr>
                                    <td><a class="property-title" href="<?php the_permalink(); ?>"><?php the_title(); ?></a></td>
                                    <td>
										<?php
										if ( function_exists( 'ere_property_price' ) ) {
											?>
                                            <p class="sub-property-price"><?php ere_property_price(); ?></p>
											<?php
										}
										?>
                                    </td>
									<?php
									if ( $show_property_type ) {
										?>
                                        <td><?php echo inspiry_get_property_types_string( $property_id ); ?></td>
										<?php
									}

									if ( $show_meta_fields ) {
										?>
                                        <td><?php echo esc_html( $property_bedrooms ); ?></td>
                                        <td><?php echo esc_html( $property_bathrooms ); ?></td>
                                        <td>
                                            <p class="sub-property-property-size"><?php echo esc_html( $property_size );
												if ( ! empty( $size_postfix ) ) {
													echo '<sup>' . esc_html( $size_postfix ) . '</sup>';
												}
												?>
                                            </p>
                                        </td>
										<?php
									}

									if ( $show_availability_date ) {
										?>
                                        <td><?php esc_html_e( 'Call for availability', RHEA_TEXT_DOMAIN ); ?></td>
										<?php
									}
									?>
                                </tr>
								<?php
							}
							?>
                            </tbody>
                        </table>
                    </div>
					<?php
				} else {
					while ( $sub_properties_query->have_posts() ) {
						$sub_properties_query->the_post();

						$property_id    = get_the_ID();
						$property_image = 'modern-property-child-slider';
						$property_link  = get_the_permalink( $property_id );
						?>
                        <article class="rh-property-sub-property-card">
                            <figure class="rh-property-thumb-wrapper rhea-trigger-animate">
								<?php
								if ( has_post_thumbnail( $property_id ) ) {
									$property_thumbnail = get_the_post_thumbnail_url( $property_id, $property_image );
								} else {
									$property_thumbnail = inspiry_get_raw_placeholder_url( $property_image );
								}

								printf( '<a href="%s" style="background-image: url(%s);"></a>', esc_url( $property_link ), $property_thumbnail );
								?>
                            </figure>
                            <div class="rh-property-sub-property-card-inner">
                                <header class="rh-property-sub-property-card-header">
                                    <h3 class="property-title"><a href="<?php echo esc_url( $property_link ); ?>"><?php the_title(); ?></a></h3>
									<?php
									if ( function_exists( 'ere_property_price' ) ) {
										?>
                                        <div class="rh-property-price-group">
											<?php echo ere_get_property_price( $property_id, true, true ); ?>
                                        </div>
										<?php
									}
									?>
                                </header>
                                <div class="rh-property-additional-info">
									<?php rhea_get_template_part( 'assets/partials/ultra/grid-card-meta' ); ?>
                                    <div class="rh-property-type-and-status">
										<?php
										$property_status = get_the_terms( $property_id, 'property-status' );
										if ( ! empty( $property_status ) && ! is_wp_error( $property_status ) ) {
											?>
                                            <a href="<?php echo get_term_link( $property_status[0]->term_id ); ?>"><?php echo esc_html( $property_status[0]->name ); ?></a>
											<?php
										}
										$property_type = get_the_terms( $property_id, "property-type" );
										if ( ! empty( $property_type ) && ! is_wp_error( $property_type ) ) {
											?>
                                            <a href="<?php echo get_term_link( $property_type[0]->term_id ); ?>"><?php echo esc_html( $property_type[0]->name ); ?></a>
											<?php
										}
										?>
                                    </div>
                                </div>

                                <p class="rh-property-added-date">
									<?php
									if ( ! empty( $settings['property_added_date_label'] ) ) {
										echo '<span class="added-title">' . esc_html( $settings['property_added_date_label'] ) . '</span>';
									}

									echo get_the_date();
									?>
                                </p>
                            </div>
                        </article>
						<?php
					}
				}

				wp_reset_postdata();
				?>
            </div><!-- .rh-single-property-sub-properties-v1 -->
			<?php
		}
	}

	private function initialize_property_id() {
		if ( rhea_is_preview_mode() && rhea_get_sample_property_id() ) {
			return rhea_get_sample_property_id();
		}

		return get_the_ID();
	}
}