<?php
/**
 * Single Property RVR Features Widget V2.
 *
 * @since 2.4.0
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Single_Property_RVR_Features_V2 extends \Elementor\Widget_Base {

	use RHEA_Single_Property_V2_Title_Section_Trait;

	private static $property_id;
	private static $features_list;

	public function __construct( $data = [], $args = null ) {
		parent::__construct( $data, $args );

		if ( is_null( self::$property_id ) ) {
			self::$property_id = $this->initialize_property_id();
		}

		if ( is_null( self::$features_list ) ) {
			self::$features_list = $this->features_list();
		}
	}

	public function get_name() {
		return 'rh-single-property-rvr-features-v2';
	}

	public function get_title() {
		return esc_html__( 'RH: Single Property RVR Features V2', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		return 'eicon-bullet-list rh-ultra-widget';
	}

	public function get_categories() {
		return [ 'ultra-realhomes-single-property' ];
	}

	protected function register_controls() {

		$this->title_section_content_tab_controls();

		foreach ( self::$features_list as $id => $label ) {
			$this->start_controls_section(
				$id . '_section',
				[
					'label' => $label,
					'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				]
			);

			$enable_section = $id . '_enable';

			$this->add_control(
				$enable_section,
				[
					'label'        => esc_html__( 'Enable', RHEA_TEXT_DOMAIN ),
					'type'         => \Elementor\Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
					'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);

			$this->add_control(
				$id . '_label',
				[
					'label'     => esc_html__( 'Custom Label', RHEA_TEXT_DOMAIN ),
					'type'      => \Elementor\Controls_Manager::TEXT,
					'default'   => $label,
					'condition' => [
						$enable_section => 'yes',
					],
				]
			);

			$this->add_control(
				$id . '_show_icon',
				[
					'label'        => esc_html__( 'Show Panel Icon', RHEA_TEXT_DOMAIN ),
					'type'         => \Elementor\Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
					'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
					'return_value' => 'yes',
					'default'      => 'yes',
					'condition'    => [
						$enable_section => 'yes',
					],
				]
			);

			$this->add_control(
				$id . '_icon',
				[
					'label'       => esc_html__( 'Custom Panel Icon', RHEA_TEXT_DOMAIN ),
					'type'        => \Elementor\Controls_Manager::ICONS,
					'skin'        => 'inline',
					'label_block' => false,
					'condition'   => [
						$enable_section    => 'yes',
						$id . '_show_icon' => 'yes',
					],
				]
			);

			$this->end_controls_section();
		}

		$this->start_controls_section(
			'tabs_container',
			[
				'label' => esc_html__( 'Container', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'content_container_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_container_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_container_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_container_background',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'name'     => 'content_container_box_shadow',
				'selector' => '{{WRAPPER}} .rh-single-property-rvr-features-v2',
			]
		);

		$this->end_controls_section();

		$this->title_section_style_tab_controls();

		$this->start_controls_section(
			'tabs',
			[
				'label' => esc_html__( 'Tabs', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'tabs_items_gap',
			[
				'label'     => esc_html__( 'Items Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-list' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_items__inner_content_gap',
			[
				'label'     => esc_html__( 'Items Inner Content Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-list li' => 'column-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'tabs_items_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-list li' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'tabs_active_item_color',
			[
				'label'     => esc_html__( 'Color (Active and Hover)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-list li:hover,
					 {{WRAPPER}} .rh-single-property-rvr-features-v2-tab-list li.active' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'tabs_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-list li',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'tabs_panels',
			[
				'label' => esc_html__( 'Panels', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'tabs_panels_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_panels_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_panels_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-content' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_panels_background',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-content' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'name'     => 'tabs_panels_box_shadow',
				'selector' => '{{WRAPPER}} .rh-single-property-rvr-features-v2-tab-content',
			]
		);

		$this->add_control(
			'panels_title',
			[
				'label'     => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'panels_title_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-label' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'panels_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'panels_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-rvr-features-v2-label',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'panels_lists',
			[
				'label' => esc_html__( 'Panels Lists', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'panels_list_items_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list li,
					{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list li a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'panels_list_link_color',
			[
				'label'     => esc_html__( 'Links Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list li a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'panels_list_icon_color',
			[
				'label'     => esc_html__( 'Icons', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list .outdoor-features-icon path,
					 {{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list .outdoor-features-icon circle' => 'fill: {{VALUE}}',
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list i'                             => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'panels_list_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list li',
			]
		);

		$this->add_control(
			'panels_list_title',
			[
				'label'     => esc_html__( 'List Title', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'panels_list_title_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'panels_list_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list-title' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'panels_list_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-list-title',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'tabs_panels_icons',
			[
				'label' => esc_html__( 'Panels Icons', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'icon_size',
			[
				'label'     => esc_html__( 'Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-icon img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'svg_color',
			[
				'label'     => esc_html__( 'Fill', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-icon svg,
					 {{WRAPPER}} .rh-single-property-rvr-features-v2-feature-icon svg path,
					 {{WRAPPER}} .rh-single-property-rvr-features-v2-feature-icon svg rect' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'svg_stroke',
			[
				'label'     => esc_html__( 'Stroke', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-icon svg path,
					{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-icon svg rect' => 'stroke: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'svg_circle_color',
			[
				'label'     => esc_html__( 'Circle Element Fill (If any)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-rvr-features-v2-feature-icon svg circle' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function render() {

		// Check if login required to vew property detail.
		if ( inspiry_prop_detail_login() ) {
			return;
		}

		$settings  = $this->get_settings_for_display();
		$widget_id = $this->get_id();
		?>
        <div id="rh-single-property-rvr-features-v2-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-rvr-features-v2 <?php realhomes_printable_section( 'features' ); ?>">
            <div id="rh-single-property-rvr-features-v2-tabs-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-rvr-features-v2-tabs">
                <div class="rh-single-property-rvr-features-v2-tabs-inner">
					<?php
					if ( ! empty( $settings['section_title'] ) ) {
						?>
                        <h4 class="rh-single-property-section-heading rh-single-property-section-heading-common"><?php echo esc_html( $settings['section_title'] ); ?></h4>
						<?php
					}

					$this->get_tabs( $settings );
					?>
                </div>
                <div class="rh-single-property-rvr-features-v2-tab-content">
					<?php $this->get_tabs_content( $settings ); ?>
                </div>
            </div>
        </div>
		<?php
		$this->get_script( $widget_id );
	}

	private function initialize_property_id() {
		if ( rhea_is_preview_mode() && rhea_get_sample_property_id() ) {
			return rhea_get_sample_property_id();
		}

		return get_the_ID();
	}

	public function features_list() {
		return array(
			'rvr_amenities'        => esc_html__( 'Amenities', RHEA_TEXT_DOMAIN ),
			'rvr_surroundings'     => esc_html__( 'Surroundings', RHEA_TEXT_DOMAIN ),
			'rvr_outdoor_features' => esc_html__( 'Outdoor Features', RHEA_TEXT_DOMAIN ),
			'rvr_services'         => esc_html__( 'Services', RHEA_TEXT_DOMAIN ),
			'rvr_policies'         => esc_html__( 'Property Rules', RHEA_TEXT_DOMAIN ),
		);
	}

	public function get_features_list( $settings ) {
		$features = array();

		foreach ( self::$features_list as $id => $label ) {
			if ( 'yes' === $settings[ $id . '_enable' ] ) {

				if ( ! empty( $settings[ $id . '_label' ] ) ) {
					$label = $settings[ $id . '_label' ];
				}

				$features[ $id ] = $label;
			}
		}

		return $features;
	}

	public function get_tabs( $settings ) {
		$features = $this->get_features_list( $settings );

		if ( empty( $features ) ) {
			return;
		}

		$tab_counter = 1;
		?>
        <ol class="rh-single-property-rvr-features-v2-tab-list">
			<?php
			foreach ( $features as $feature ) {
				printf( '<li class="rh-single-property-rvr-features-v2-tab"><span class="counter">%s</span><span class="feature-label">%s</span></li>', $this->counter_prefix( $tab_counter ), esc_html( $feature ) );
				$tab_counter++;
			}
			?>
        </ol>
		<?php
	}

	public function get_tabs_content( $settings ) {
		$features = $this->get_features_list( $settings );

		if ( empty( $features ) ) {
			return;
		}

		$content_counter = 1;
		foreach ( $features as $meta_key => $feature ) {
			?>
            <div class="rh-single-property-rvr-features-v2-tab-panel">
				<?php
				if ( ! empty( $feature ) ) {
					printf( '<h5 class="rh-single-property-rvr-features-v2-label"><span class="counter">%s</span><span class="feature-label">%s</span></h5>', $this->counter_prefix( $content_counter ), esc_html( $feature ) );
				}
				?>
                <div class="rh-single-property-rvr-features-v2-tab-panel-inner">
					<?php
					if ( is_callable( [ $this, $meta_key ] ) ) {
						call_user_func( [ $this, $meta_key ], $meta_key, $settings );
					}
					?>
                </div>
            </div>
			<?php
			$content_counter++;
		}
	}

	public function rvr_amenities( $meta_key, $settings ) {
		$is_link  = get_option( 'inspiry_property_features_display', 'link' );
		$features = get_the_terms( self::$property_id, 'property-feature' );

		if ( ! empty( $features ) && ! is_wp_error( $features ) ) {
			?>
            <ul class="rh-single-property-rvr-features-v2-feature-list">
				<?php
				foreach ( $features as $feature ) {
					?>
                    <li>
                        <i class="fas fa-check"></i>
						<?php
						if ( 'link' === $is_link ) {
							printf( '<a href="%s">%s</a>', esc_url( get_term_link( $feature->slug, 'property-feature' ) ), esc_html( $feature->name ) );
						} else {
							printf( '<span class="feature-item-label">%s</span>', esc_html( $feature->name ) );
						}
						?>
                    </li>
					<?php
				}
				?>
            </ul>
			<?php
		}

		if ( 'yes' === $settings[ $meta_key . '_show_icon' ] ) {
			?>
            <div class="rh-single-property-rvr-features-v2-feature-icon">
				<?php
				if ( ! empty( $settings[ $meta_key . '_icon' ]['value'] ) ) {
					\Elementor\Icons_Manager::render_icon( $settings[ $meta_key . '_icon' ] );
				} else {
					?>
                    <svg width="205" height="204" viewBox="0 0 205 204" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <circle cx="150.009" cy="94.9904" r="18.017" fill="#BA9BFB" />
                        <circle cx="37.4366" cy="41.3846" r="6.14078" transform="rotate(38.8928 37.4366 41.3846)" fill="#BA9BFB" />
                        <circle cx="53.9069" cy="48.3593" r="3.68447" transform="rotate(38.8928 53.9069 48.3593)" fill="#BA9BFB" />
                        <path d="M121.993 118.284C121.993 125.681 115.665 131.877 107.607 131.877C99.5489 131.877 93.221 125.681 93.221 118.284C93.221 110.887 99.5489 104.691 107.607 104.691C115.665 104.691 121.993 110.887 121.993 118.284Z" stroke="#1D1D1B" stroke-width="4.91373" />
                        <path d="M67.1653 124.302V180.16C67.1653 182.874 69.3652 185.074 72.079 185.074H89.1106C91.8244 185.074 94.0244 182.874 94.0244 180.16V153.868C94.0244 151.155 96.2243 148.955 98.9381 148.955H117.801C120.515 148.955 122.715 151.155 122.715 153.868V180.16C122.715 182.874 124.915 185.074 127.628 185.074H144.05C146.763 185.074 148.963 182.874 148.963 180.16V124.302" stroke="#1D1D1B" stroke-width="4.91373" stroke-linecap="round" />
                        <path d="M43.6021 121.659L107.625 76.7405L172.538 121.659" stroke="#1D1D1B" stroke-width="4.91373" stroke-linecap="round" />
                        <path d="M130.949 92.5939V75.0483C130.949 72.3345 133.149 70.1345 135.862 70.1345H140.592C143.306 70.1345 145.506 72.3345 145.506 75.0483V102.502" stroke="#1D1D1B" stroke-width="4.91373" stroke-linecap="round" />
                        <path d="M141.101 35.2401L141.101 35.2409C142.703 37.2791 143.533 39.6719 143.533 42.1819C143.533 48.5446 137.974 53.9814 130.821 53.9814C123.668 53.9814 118.109 48.5446 118.109 42.1819C118.109 39.6719 118.939 37.2791 120.541 35.2409C120.592 35.1762 120.641 35.1094 120.69 35.0401L126.925 26.3408L126.925 26.3404L126.937 26.3236C126.945 26.3129 126.959 26.2942 126.975 26.2702L130.821 20.9049L140.949 35.0353C140.999 35.1074 141.05 35.1757 141.101 35.2401Z" stroke="black" stroke-width="4.91373" />
                        <path d="M38.38 86.7109C40.9347 79.5052 46.1264 73.0735 53.6306 68.9449C61.0722 64.8507 69.5133 63.7822 77.3393 65.2767M50.2316 90.9964C51.8129 86.3125 55.1072 82.1165 59.985 79.4329C64.8002 76.7837 70.2608 76.1631 75.3081 77.2" stroke="black" stroke-width="4.91373" />
                    </svg>
					<?php
				}
				?>
            </div>
			<?php
		}
	}

	public function rvr_surroundings( $meta_key, $settings ) {
		$features = get_post_meta( self::$property_id, 'rvr_surroundings', true );
		if ( ! empty( $features ) ) {
			?>
            <ul class="rh-single-property-rvr-features-v2-feature-list feature-list-full feature-list-space-between">
				<?php
				foreach ( $features as $feature ) {
					?>
                    <li>
						<?php
						if ( ! empty( $feature['rvr_surrounding_point'] ) ) {
							?>
                            <span class="surrounding-point">
                                <i class="fas fa-map-marker-alt"></i>
                                <?php echo esc_html( $feature['rvr_surrounding_point'] ); ?>
                            </span>
							<?php
						}

						if ( ! empty( $feature['rvr_surrounding_point_distance'] ) ) {
							?>
                            <span class="surrounding-point-distance"><?php echo esc_html( $feature['rvr_surrounding_point_distance'] ); ?></span>
							<?php
						}
						?>
                    </li>
					<?php
				}
				?>
            </ul>
			<?php
		}

		if ( 'yes' === $settings[ $meta_key . '_show_icon' ] ) {
			?>
            <div class="rh-single-property-rvr-features-v2-feature-icon">
				<?php
				if ( ! empty( $settings[ $meta_key . '_icon' ]['value'] ) ) {
					\Elementor\Icons_Manager::render_icon( $settings[ $meta_key . '_icon' ] );
				} else {
					?>
                    <svg width="124" height="124" viewBox="0 0 124 124" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <circle cx="82" cy="67" r="11" fill="#BA9BFB" />
                        <path d="M66.9709 53.3354C66.9709 55.5802 65.0646 57.4957 62.5929 57.4957C60.1211 57.4957 58.2148 55.5802 58.2148 53.3354C58.2148 51.0905 60.1211 49.175 62.5929 49.175C65.0646 49.175 66.9709 51.0905 66.9709 53.3354Z" stroke="#1D1D1B" stroke-width="3" />
                        <path d="M44.1875 56.4485V82.0002C44.1875 83.6571 45.5306 85.0002 47.1875 85.0002H53.2122C54.8691 85.0002 56.2122 83.6571 56.2122 82.0002V71.0308C56.2122 69.3739 57.5554 68.0308 59.2122 68.0308H66.0568C67.7136 68.0308 69.0568 69.3739 69.0568 71.0308V82.0002C69.0568 83.6571 70.3999 85.0002 72.0568 85.0002H77.8082C79.465 85.0002 80.8082 83.6571 80.8082 82.0002V56.4485" stroke="#1D1D1B" stroke-width="3" stroke-linecap="round" />
                        <path d="M33.6367 55.207L62.2993 34.1035L91.3609 55.207" stroke="#1D1D1B" stroke-width="3" stroke-linecap="round" />
                        <path d="M72.7402 41.5517V34C72.7402 32.3431 74.0834 31 75.7402 31H76.2575C77.9143 31 79.2575 32.3431 79.2575 34V46.2069" stroke="#1D1D1B" stroke-width="3" stroke-linecap="round" />
                        <path d="M78.5 13.7866C85.3748 16.189 91.5648 20.0505 96.6988 25.0001M47 13.2891C39.5247 15.6369 32.8059 19.6931 27.3012 25.0001M13.4493 49C12.5023 52.8442 12 56.8635 12 61C12 65.1366 12.5023 69.1559 13.4493 73.0001M27.3012 97.0001C32.8059 102.307 39.5247 106.363 47 108.711M78.5 108.214C85.3748 105.811 91.5648 101.95 96.6988 97.0001M110.551 73.0001C111.498 69.1559 112 65.1366 112 61C112 56.8635 111.498 52.8442 110.551 49" stroke="#1D1D1B" stroke-width="3" />
                        <circle cx="62" cy="111" r="9.5" fill="white" stroke="#1D1D1B" stroke-width="3" />
                        <circle cx="19" cy="85" r="9.5" fill="white" stroke="#1D1D1B" stroke-width="3" />
                        <circle cx="19" cy="37" r="9.5" fill="white" stroke="#1D1D1B" stroke-width="3" />
                        <circle cx="62" cy="14" r="9.5" fill="white" stroke="#1D1D1B" stroke-width="3" />
                        <circle cx="104" cy="37" r="9.5" fill="white" stroke="#1D1D1B" stroke-width="3" />
                        <circle cx="104" cy="85" r="9.5" fill="white" stroke="#1D1D1B" stroke-width="3" />
                        <circle cx="17.2711" cy="10.272" r="3.74915" transform="rotate(38.8928 17.2711 10.272)" fill="#BA9BFB" />
                        <circle cx="27.3267" cy="14.5304" r="2.24949" transform="rotate(38.8928 27.3267 14.5304)" fill="#BA9BFB" />
                    </svg>
					<?php
				}
				?>
            </div>
			<?php
		}
	}

	public function rvr_outdoor_features( $meta_key, $settings ) {
		$features = get_post_meta( self::$property_id, 'rvr_outdoor_features', true );
		if ( ! empty( $features ) ) {
			?>
            <ul class="rh-single-property-rvr-features-v2-feature-list feature-list-full outdoor-feature-item">
				<?php
				foreach ( $features as $feature ) {
					?>
                    <li>
                        <svg class="outdoor-features-icon" width="18" height="14" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 307.2 307.2">
                            <circle cx="153.6" cy="153.6" r="68.3"></circle>
                            <path d="M153.6 0c-9.4 0-17.1 7.6-17.1 17.1v17.1c0 9.4 7.6 17.1 17.1 17.1s17.1-7.6 17.1-17.1V17.1C170.7 7.6 163 0 153.6 0z"></path>
                            <path d="M153.6 256c-9.4 0-17.1 7.6-17.1 17.1v17.1c0 9.4 7.6 17.1 17.1 17.1s17.1-7.6 17.1-17.1v-17.1C170.7 263.6 163 256 153.6 256z"></path>
                            <path d="M290.1 136.5h-17.1c-9.4 0-17.1 7.6-17.1 17.1s7.6 17.1 17.1 17.1h17.1c9.4 0 17.1-7.6 17.1-17.1S299.6 136.5 290.1 136.5z"></path>
                            <path d="M34.1 136.5H17.1C7.6 136.5 0 144.2 0 153.6s7.6 17.1 17.1 17.1h17.1c9.4 0 17.1-7.6 17.1-17.1S43.6 136.5 34.1 136.5z"></path>
                            <path d="M261.8 45c-6.6-6.4-17.1-6.4-23.7 0l-12.1 12.1c-6.7 6.7-6.7 17.5 0 24.1 3.2 3.2 7.5 5 12.1 5 4.5 0 8.9-1.8 12.1-5l12.1-12.1C268.8 62.3 268.6 51.5 261.8 45z"></path>
                            <path d="M81.6 226c-6.5-6.8-17.4-7-24.1-0.4 -0.1 0.1-0.3 0.3-0.4 0.4L45 238.1c-6.8 6.5-7 17.4-0.4 24.1 6.5 6.8 17.4 7 24.1 0.4 0.1-0.1 0.3-0.3 0.4-0.4l12.1-12.1C88 243.6 88.2 232.8 81.6 226z"></path>
                            <path d="M262.2 238.1l-12.1-12.1c-6.8-6.5-17.6-6.4-24.1 0.4 -6.4 6.6-6.4 17.1 0 23.7l12.1 12.1c6.8 6.5 17.6 6.4 24.1-0.4C268.6 255.2 268.6 244.7 262.2 238.1z"></path>
                            <path d="M81.6 57.5c-0.1-0.1-0.3-0.3-0.4-0.4L69.1 45c-6.8-6.5-17.6-6.4-24.1 0.4 -6.4 6.6-6.4 17.1 0 23.7l12.1 12.1c6.5 6.8 17.4 7 24.1 0.4C88 75.1 88.2 64.3 81.6 57.5z"></path>
                        </svg>
                        <span><?php echo esc_html( $feature ); ?></span>
                    </li>
					<?php
				}
				?>
            </ul>
			<?php
		}

		if ( 'yes' === $settings[ $meta_key . '_show_icon' ] ) {
			?>
            <div class="rh-single-property-rvr-features-v2-feature-icon">
				<?php
				if ( ! empty( $settings[ $meta_key . '_icon' ]['value'] ) ) {
					\Elementor\Icons_Manager::render_icon( $settings[ $meta_key . '_icon' ] );
				} else {
					?>
                    <svg width="124" height="124" viewBox="0 0 124 124" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <circle cx="18" cy="28" r="11" fill="#BA9BFB" />
                        <path d="M16.8288 52.5762H6.63672V67.0596H16.8288M99.7063 52.5762H109.094V67.0596H99.7063" stroke="black" stroke-width="3" />
                        <path d="M37.2129 52.5762H47.9414M37.2129 67.0596H47.9414" stroke="black" stroke-width="3" />
                        <path d="M68.3262 52.5762H79.0546M68.3262 67.0596H79.0546" stroke="black" stroke-width="3" />
                        <path d="M16.8288 75.1057H6.63672V89.5892H16.8288M99.7063 75.1057H109.094V89.5892H99.7063" stroke="black" stroke-width="3" />
                        <path d="M37.2129 75.1057H47.9414M37.2129 89.5892H47.9414" stroke="black" stroke-width="3" />
                        <path d="M68.3262 75.1057H79.0546M68.3262 89.5892H79.0546" stroke="black" stroke-width="3" />
                        <path d="M26.9205 22L16.8301 40.3655V103H37.2142V40.3655L26.9205 22Z" stroke="black" stroke-width="3" />
                        <path d="M58.0299 22L47.9395 40.3655V103H68.3236V40.3655L58.0299 22Z" stroke="black" stroke-width="3" />
                        <path d="M89.1432 22L79.0527 40.3655V103H99.4368V40.3655L89.1432 22Z" stroke="black" stroke-width="3" />
                        <path d="M83.9693 96C75.1575 96 68 89.0633 68 80.5C68 71.9367 75.1575 65 83.9693 65C87.548 65 90.8457 66.1334 93.5054 68.0748C96.1543 66.1334 99.452 65 103.031 65C111.842 65 119 71.9367 119 80.5C119 89.0633 111.842 96 103.031 96H83.9693Z" fill="white" stroke="black" stroke-width="3" stroke-miterlimit="10" />
                        <path d="M77.9693 106C69.1575 106 62 99.0633 62 90.5C62 81.9367 69.1575 75 77.9693 75C81.548 75 84.8457 76.1334 87.5054 78.0748C90.1543 76.1334 93.452 75 97.0307 75C105.842 75 113 81.9367 113 90.5C113 99.0633 105.842 106 97.0307 106H77.9693Z" fill="white" stroke="black" stroke-width="3" stroke-miterlimit="10" />
                        <circle cx="85.623" cy="12.8476" r="3.74915" transform="rotate(158.893 85.623 12.8476)" fill="#BA9BFB" />
                        <circle cx="76.9078" cy="19.4268" r="2.24949" transform="rotate(158.893 76.9078 19.4268)" fill="#BA9BFB" />
                    </svg>
					<?php
				}
				?>
            </div>
			<?php
		}
	}

	public function rvr_services( $meta_key, $settings ) {
		$included_heading     = $settings['included_title'] ?? esc_html__( 'Included', RHEA_TEXT_DOMAIN );
		$not_included_heading = $settings['not_included_title'] ?? esc_html__( 'Not Included', RHEA_TEXT_DOMAIN );
		$rvr_included         = get_post_meta( self::$property_id, 'rvr_included', true );
		$rvr_not_included     = get_post_meta( self::$property_id, 'rvr_not_included', true );
		?>
        <div class="rh-single-property-rvr-features-v2-feature-lists-wrapper">
			<?php
			if ( ! empty( $rvr_included ) ) {
				?>
                <h4 class="rh-single-property-rvr-features-v2-feature-list-title"><?php echo esc_html( $included_heading ); ?></h4>
                <ul class="rh-single-property-rvr-features-v2-feature-list feature-list-full feature-item-included">
					<?php
					foreach ( $rvr_included as $rvr_include ) {
						echo '<li><i class="fas fa-check"></i>' . esc_html( $rvr_include ) . '</li>';
					}
					?>
                </ul>
				<?php
			}

			if ( ! empty( $rvr_not_included ) ) {
				?>
                <h4 class="rh-single-property-rvr-features-v2-feature-list-title"><?php echo esc_html( $not_included_heading ); ?></h4>
                <ul class="rh-single-property-rvr-features-v2-feature-list feature-list-full feature-item-not-included">
					<?php
					foreach ( $rvr_not_included as $rvr_not_include ) {
						echo '<li><i class="fas fa-times"></i>' . esc_html( $rvr_not_include ) . '</li>';
					}
					?>
                </ul>
				<?php
			}
			?>
        </div>
		<?php
		if ( 'yes' === $settings[ $meta_key . '_show_icon' ] ) {
			?>
            <div class="rh-single-property-rvr-features-v2-feature-icon">
				<?php
				if ( ! empty( $settings[ $meta_key . '_icon' ]['value'] ) ) {
					\Elementor\Icons_Manager::render_icon( $settings[ $meta_key . '_icon' ] );
				} else {
					?>
                    <svg width="150" height="150" viewBox="0 0 124 124" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <circle cx="96" cy="27" r="11" fill="#BA9BFB" />
                        <path d="M27.6701 87.1488L18 92.0781L28.1551 112L37.8251 107.071L27.6701 87.1488Z" stroke="#1D1D1B" stroke-width="3" stroke-miterlimit="10" />
                        <path d="M28.916 89.2147L33.5973 86.4785C40.1364 82.6573 47.8477 81.9474 54.8807 84.532L63.2038 87.5874C65.526 88.4409 66.595 91.3288 65.4597 93.6821C64.5382 95.5807 62.474 96.4662 60.6088 95.7562L51.8139 92.4376" stroke="#1D1D1B" stroke-width="3" stroke-miterlimit="10" />
                        <path d="M35.916 103.732C35.916 103.732 37.8984 100.774 42.1194 102.659C43.2953 103.181 45.2852 103.862 47.5464 104.573C55.898 107.197 65.0185 106.167 72.5183 101.775L86.0784 93.8299C88.0607 92.6701 88.5205 90.0895 87.0582 88.357C85.8146 86.8855 83.606 86.5666 81.9629 87.6177L70.536 93.6632" stroke="#1D1D1B" stroke-width="3" stroke-miterlimit="10" />
                        <path d="M41.916 73H105.916" stroke="#1D1D1B" stroke-width="3" stroke-miterlimit="10" />
                        <path d="M97.916 72V35" stroke="#1D1D1B" stroke-width="3" stroke-miterlimit="10" />
                        <path d="M49.916 72V35" stroke="#1D1D1B" stroke-width="3" stroke-miterlimit="10" />
                        <path d="M41.916 41L73.381 15L105.916 41" stroke="#1D1D1B" stroke-width="3" stroke-miterlimit="10" />
                        <path d="M67.916 72V50H79.916V72" stroke="#1D1D1B" stroke-width="3" stroke-miterlimit="10" />
                        <path d="M60.916 25.3833V13H52.916V32" stroke="#1D1D1B" stroke-width="3" stroke-miterlimit="10" />
                        <circle cx="23.2711" cy="24.272" r="3.74915" transform="rotate(38.8928 23.2711 24.272)" fill="#BA9BFB" />
                        <circle cx="33.3267" cy="28.5304" r="2.24949" transform="rotate(38.8928 33.3267 28.5304)" fill="#BA9BFB" />
                    </svg>
					<?php
				}
				?>
            </div>
			<?php
		}
	}

	public function rvr_policies( $meta_key, $settings ) {
		$features = get_post_meta( self::$property_id, 'rvr_policies', true );
		if ( ! empty( $features ) ) {
			?>
            <ul class="rh-single-property-rvr-features-v2-feature-list feature-list-full">
				<?php
				foreach ( $features as $feature_detail ) {
					?>
                    <li>
						<?php
						if ( ! empty( $feature_detail['rvr_policy_icon'] ) ) {
							?>
                            <i class="<?php echo esc_attr( $feature_detail['rvr_policy_icon'] ); ?>"></i>
							<?php
						} else {
                            ?>
                            <i class="fas fa-check"></i>
                            <?php
						}

						echo esc_html( $feature_detail['rvr_policy_detail'] );
						?>
                    </li>
					<?php
				}
				?>
            </ul>
			<?php
		}

		if ( 'yes' === $settings[ $meta_key . '_show_icon' ] ) {
			?>
            <div class="rh-single-property-rvr-features-v2-feature-icon">
				<?php
				if ( ! empty( $settings[ $meta_key . '_icon' ]['value'] ) ) {
					\Elementor\Icons_Manager::render_icon( $settings[ $meta_key . '_icon' ] );
				} else {
					?>
                    <svg width="124" height="124" viewBox="0 0 124 124" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <circle cx="83" cy="16" r="11" fill="#BA9BFB" />
                        <circle cx="99.2711" cy="35.272" r="3.74915" transform="rotate(38.8928 99.2711 35.272)" fill="#BA9BFB" />
                        <circle cx="109.327" cy="39.5304" r="2.24949" transform="rotate(38.8928 109.327 39.5304)" fill="#BA9BFB" />
                        <path d="M85 99.5281V101C85 103.761 82.7614 106 80 106H17C14.2386 106 12 103.761 12 101V15C12 12.2386 14.2386 10 17 10H80C82.7614 10 85 12.2386 85 15V42.0674" stroke="#1D1D1B" stroke-width="3" />
                        <path d="M22 28H75" stroke="#1D1D1B" stroke-width="3" stroke-linecap="round" />
                        <path d="M22 43H66" stroke="#1D1D1B" stroke-width="3" stroke-linecap="round" />
                        <path d="M22 58H58" stroke="#1D1D1B" stroke-width="3" stroke-linecap="round" />
                        <circle cx="84.8206" cy="68.5945" r="5.03153" stroke="#1D1D1B" stroke-width="3" />
                        <path d="M69.1387 71.0432V92.7744C69.1387 94.4313 70.4818 95.7744 72.1387 95.7744H76.5543C78.2112 95.7744 79.5543 94.4312 79.5543 92.7744V84.0757C79.5543 82.4188 80.8975 81.0757 82.5543 81.0757H87.6802C89.337 81.0757 90.6802 82.4188 90.6802 84.0757V92.7744C90.6802 94.4313 92.0233 95.7744 93.6802 95.7744H97.8591C99.516 95.7744 100.859 94.4312 100.859 92.7744V71.0432" stroke="#1D1D1B" stroke-width="3" stroke-linecap="round" />
                        <path d="M60 69.9678L84.8272 51.6882L110 69.9678" stroke="#1D1D1B" stroke-width="3" stroke-linecap="round" />
                        <path d="M93.873 58.1398V51.8226C93.873 50.2637 95.1368 49 96.6956 49V49C98.2545 49 99.5182 50.2637 99.5182 51.8226V62.172" stroke="#1D1D1B" stroke-width="3" stroke-linecap="round" />
                    </svg>
					<?php
				}
				?>
            </div>
			<?php
		}
	}

	public function counter_prefix( $counter ): string {
		return ( $counter < 10 ? '0' . $counter : $counter );
	}

	public function get_script( $widget_id ) {
		?>
        <script>
            jQuery( document ).ready( function ( $ ) {
                "use strict";

                const $tabsContainer = $( '#rh-single-property-rvr-features-v2-<?php echo esc_attr( $widget_id ); ?>' ),
                      $tabs          = $tabsContainer.find( '.rh-single-property-rvr-features-v2-tab' ),
                      $panels        = $tabsContainer.find( '.rh-single-property-rvr-features-v2-tab-panel' );

                // Initialize - Set the first tab and panel as active
                $tabs.first().addClass( 'active' );
                $panels.hide().first().show();

                // Handle click events on tabs
                $tabs.on( 'click', function () {
                    const $clickedTab = $( this ),
                          tabIndex    = $clickedTab.index();

                    // Remove active class from all tabs and add to the clicked tab
                    $tabs.removeClass( 'active' );
                    $clickedTab.addClass( 'active' );

                    // Hide all panels and show the one corresponding to the clicked tab
                    $panels.hide().eq( tabIndex ).fadeIn();
                } );
            } );
        </script>
		<?php
	}
}