<?php
/**
 * Single Property Mortgage Calculator Widget V2.
 *
 * @since 2.4.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Single_Property_Mortgage_Calculator_V2 extends \Elementor\Widget_Base {

	use RHEA_Single_Property_V2_Title_Section_Trait;

	private static $property_id;

	public function __construct( array $data = [], ?array $args = null ) {
		parent::__construct( $data, $args );

		if ( is_null( self::$property_id ) ) {
			self::$property_id = $this->initialize_property_id();
		}
	}

	public function get_name() {
		return 'rh-single-property-mortgage-calculator-v2';
	}

	public function get_title() {
		return esc_html__( 'RH: Single Property Mortgage Calculator V2', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		// More classes for icons can be found at https://pojome.github.io/elementor-icons/
		return 'eicon-number-field rh-ultra-widget';
	}

	public function get_categories() {
		return [ 'ultra-realhomes-single-property' ];
	}

	protected function register_controls() {

		$this->title_section_content_tab_controls( esc_html__( 'Mortgage Calculator', RHEA_TEXT_DOMAIN ) );

		$this->title_section_style_tab_controls();

		$this->start_controls_section(
			'mc_container_styles',
			[
				'label' => esc_html__( 'Container', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'mc_container_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_property__mc' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'mc_container_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_property__mc' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'mc_container_items_row_gap',
			[
				'label'     => esc_html__( 'Items Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_property__mc' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'mc_container_bg',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_property__mc' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'mc_container_box_shadow',
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_property__mc',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'mc_fields_section_styles',
			[
				'label' => esc_html__( 'Fields', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'mc_fields_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_mc_field' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'mc_fields_label_color',
			[
				'label'     => esc_html__( 'Label Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_mc_field label,
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_mc_field input' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'mc_fields_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_mc_field label, {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh_mc_field input',
			]
		);

		$this->add_control(
			'mc_term_dropdown_section_heading',
			[
				'label'     => esc_html__( 'Term Field', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'mc_fields_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .bootstrap-select .dropdown-toggle' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'mc_fields_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .bootstrap-select .dropdown-toggle' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'chevron_margin',
			[
				'label'     => esc_html__( 'Chevron Icon Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rhea-dropdown-style-chevron .bootstrap-select.inspiry_select_picker_trigger>.dropdown-toggle .caret:before' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'chevron_font_size',
			[
				'label'     => esc_html__( 'Chevron Icon Font Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-dropdown-style-chevron .bootstrap-select.inspiry_select_picker_trigger>.dropdown-toggle .caret:before' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'mc_fields_color',
			[
				'label'     => esc_html__( 'Caret Icon Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .bootstrap-select .dropdown-toggle' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'mc_caret_icon_hover_color',
			[
				'label'     => esc_html__( 'Caret Icon Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .bootstrap-select.open > .dropdown-toggle,
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .bootstrap-select .dropdown-toggle:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'mc_fields_border_color',
			[
				'label'     => esc_html__( 'Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .bootstrap-select .dropdown-toggle' => 'border-color: {{VALUE}}',
					'{{WRAPPER}} .inspiry_select_picker_trigger>.dropdown-toggle span.caret'                    => 'border-top-color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'mc_fields_border_width',
			[
				'label'     => esc_html__( 'Border Width', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 10,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .bootstrap-select .dropdown-toggle' => 'border-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'mc_fields_bg_color',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .bootstrap-select .dropdown-toggle' => 'background-color: {{VALUE}}'
				],
			]
		);

		$this->add_control(
			'mc_dropdown_color',
			[
				'label'     => esc_html__( 'Options Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown .dropdown-menu li a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'mc_dropdown_active_color',
			[
				'label'     => esc_html__( 'Option Active Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown .dropdown-menu li.selected .text'           => 'color: {{VALUE}}',
					'{{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown .dropdown-menu li.selected span.check-mark' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'mc_dropdown_hover_color',
			[
				'label'     => esc_html__( 'Options Hover Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown .dropdown-menu li:hover' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'mc_dropdown_hover_text_color',
			[
				'label'     => esc_html__( 'Options Hover Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select .dropdown-menu li:hover a, 
					 {{WRAPPER}} .bootstrap-select .dropdown-menu li.selected:hover .text,
					 {{WRAPPER}} .bootstrap-select .dropdown-menu li.selected:hover span.check-mark' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'mc_fields_term_select_typography',
				'label'    => esc_html__( 'Term Field Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '.bootstrap-select .dropdown-toggle .filter-option-inner-inner',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'mc_fields_options_typography',
				'label'    => esc_html__( 'Options Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .bootstrap-select .dropdown-menu li a span.text',
			]
		);

		$this->add_control(
			'mc_slider_fields_section_heading',
			[
				'label'     => esc_html__( 'Slider Fields', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'mc_slider_fields_spacing',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh-mc-slider-fields' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'mc_slider_fields_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 input[type=range]::-webkit-slider-thumb,					 
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh-mc-slider-fields:before,
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh-mc-slider-fields:after' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'mc_slider_fields_borer_color',
			[
				'label'     => esc_html__( 'Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 input[type=range]::-webkit-slider-thumb,
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh-mc-slider-fields:before,
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh-mc-slider-fields:after' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'mc_slider_fields_bg_color',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 input[type=range]::-webkit-slider-thumb,
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh-mc-slider-fields:before,
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh-mc-slider-fields:after' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'mc_slider_fields_track_bg_color',
			[
				'label'     => esc_html__( 'Track Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} input[type=range]::-webkit-slider-runnable-track' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'mc_fields_tips_section_heading',
			[
				'label'     => esc_html__( 'Tips', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'mc_fields_tips_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh-mc-slider-fields .tippable',
			]
		);

		$this->add_control(
			'mc_fields_tips_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .rh-mc-slider-fields .tippable' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'mc_graph_&_content',
			[
				'label' => esc_html__( 'Graph and Content', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'mc_content_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_right_side' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'mc_content_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_right_side' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'mc_content_items_row_gap',
			[
				'label'     => esc_html__( 'Items Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_right_side' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'mc_content_bg',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_right_side' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'mc_content_box_shadow',
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_right_side',
			]
		);

		$this->add_control(
			'mc_description_section_heading',
			[
				'label'     => esc_html__( 'Description ', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'mc_description_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_term_interest' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'mc_description_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_term_interest',
			]
		);

		$this->add_control(
			'mc_mortgage_payment_section_heading',
			[
				'label'     => esc_html__( 'Mortgage Payment', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'mc_payment_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_cost_total' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'mc_payment_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_cost_total, 
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_cost_over_graph,
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_cost_over_graph strong' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'mc_payment_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_cost_total',
			]
		);

		$this->add_control(
			'mc_graph_section_heading',
			[
				'label'     => esc_html__( 'Graph', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'mc_graph_text_color',
			[
				'label'     => esc_html__( 'Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_cost li,
					 {{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_cost li strong' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'mc_graph_text_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-mortgage-calculator-v2 .mc_cost li',
			]
		);

		$this->add_control(
			'mc_graph_interest_color',
			[
				'label'     => esc_html__( 'Interest Field Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2' => '--mc-interest-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'mc_graph_tax_color',
			[
				'label'     => esc_html__( 'Tax Field Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2' => '--mc-tax-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'mc_graph_hoa_color',
			[
				'label'     => esc_html__( 'Other Dues Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-mortgage-calculator-v2' => '--mc-hoa-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

	}

	protected function render() {

		// Check if login required to vew property detail.
		if ( inspiry_prop_detail_login() ) {
			return;
		}

		$settings  = $this->get_settings_for_display();
		$widget_id = $this->get_id();

		$property_rent_status = get_option( 'inspiry_mortgage_calculator_statuses' ); // Get property enabled statuses for the mortgage calculator.
		if ( has_term( $property_rent_status, 'property-status', self::$property_id ) ) { // Display Mortgage Calculator only if current property has enabled status.
			if ( ! empty( $settings['section_title'] ) ) {
				?>
                <h4 class="rh-single-property-section-heading rh-single-property-section-heading-common"><?php echo esc_html( $settings['section_title'] ); ?></h4>
				<?php
			}
			?>
            <div id="rh-single-property-mortgage-calculator-v2-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-mortgage-calculator-v2">
                <div class="rh_property__mc">
					<?php
					// Default data for the mortgage calculator.
					$mc_default['terms']       = array_map( 'intval', explode( ',', get_option( 'inspiry_mc_terms', '30,20,15,10,5' ) ) );
					$mc_default['term']        = intval( get_option( 'inspiry_mc_term_default', '10' ) );
					$mc_default['interest']    = floatval( get_option( 'inspiry_mc_interest_default', '3.5' ) );
					$mc_default['downpayment'] = floatval( get_option( 'inspiry_mc_downpayment_default', '20' ) );
					$mc_default['tax']         = intval( get_post_meta( self::$property_id, 'inspiry_property_tax', true ) );
					$mc_default['fee']         = intval( get_post_meta( self::$property_id, 'inspiry_additional_fee', true ) );
					$mc_default['price']       = intval( get_post_meta( self::$property_id, 'REAL_HOMES_property_price', true ) );

					if ( empty( $mc_default['terms'] ) ) {
						$mc_default['terms'] = array( '30', '20', '15', '10' );
					}

					if ( empty( $mc_default['price'] ) ) {
						$mc_default['price'] = intval( get_option( 'inspiry_mc_price_default', '0' ) );

						if ( '' === $mc_default['price'] ) {
							$mc_default['price'] = '0';
						}
					}

					// Currency conversion in case Currency Switcher is enabled.
					if ( function_exists( 'realhomes_currency_switcher_enabled' ) && realhomes_currency_switcher_enabled() ) {
						$base_currency    = realhomes_get_base_currency();
						$current_currency = realhomes_get_current_currency();

						$mc_default['price_converted'] = realhomes_convert_currency( $mc_default['price'], $base_currency, $current_currency );
						$mc_default['tax_converted']   = realhomes_convert_currency( $mc_default['tax'], $base_currency, $current_currency );
						$mc_default['fee_converted']   = realhomes_convert_currency( $mc_default['fee'], $base_currency, $current_currency );

						$currencies_data   = realhomes_get_currencies_data();
						$currency_sign     = $currencies_data[ $current_currency ]['symbol'];
						$currency_position = $currencies_data[ $current_currency ]['position'];
					} else {
						$mc_default['price_converted'] = $mc_default['price'];
						$mc_default['tax_converted']   = $mc_default['tax'];
						$mc_default['fee_converted']   = $mc_default['fee'];
						$currency_sign                 = ere_get_currency_sign();
						$currency_position             = get_option( 'theme_currency_position', 'before' );
					}

					// Fields labels.
					$term_label        = get_option( 'inspiry_mc_term_field_label', esc_html__( 'Term', RHEA_TEXT_DOMAIN ) );
					$interest_label    = get_option( 'inspiry_mc_interest_field_label', esc_html__( 'Interest', RHEA_TEXT_DOMAIN ) );
					$price_label       = get_option( 'inspiry_mc_price_field_label', esc_html__( 'Home Price', RHEA_TEXT_DOMAIN ) );
					$downpayment_label = get_option( 'inspiry_mc_downpayment_field_label', esc_html__( 'Down Payment', RHEA_TEXT_DOMAIN ) );
					$principle_label   = get_option( 'inspiry_mc_principle_field_label', esc_html__( 'Principal and Interest', RHEA_TEXT_DOMAIN ) );
					$cost_prefix       = get_option( 'inspiry_mc_cost_prefix', esc_html__( 'Monthly', RHEA_TEXT_DOMAIN ) );
					?>
                    <!-- Calculator left side -->
                    <div class="mc_left_side rhea-dropdown-style-chevron">

                        <!-- Term -->
                        <div class="rh_mc_field">
                            <label for="mc_term"><?php echo esc_html( $term_label ); ?></label>
                            <div class="rh_form__item">
                                <select id="mc_term" name="mc_term" class="rh-ultra-select-dropdown mc_term inspiry_select_picker_trigger show-tick">
									<?php
									foreach ( $mc_default['terms'] as $mc_term ) {
										echo '<option value="' . esc_attr( $mc_term ) . '" ' . selected( $mc_default['term'], $mc_term, false ) . '>' . esc_html( $mc_term ) . ' ' . esc_html__( 'Years Fixed', RHEA_TEXT_DOMAIN ) . '</option>';
									}
									?>
                                </select>
                            </div>
                        </div>

                        <!-- Interest Rate -->
                        <div class="rh_mc_field">
                            <label for="mc_interest"><?php echo esc_html( $interest_label ); ?></label>
                            <input id="mc_interest" class="mc_interset" type="text" value="<?php echo esc_attr( $mc_default['interest'] ); ?>%">
                            <div class="rh_form__item rh-mc-slider-fields">
                                <input class="mc_interset_slider" type="range" min="0" max="100" step="0.1" value="<?php echo esc_attr( $mc_default['interest'] ); ?>">
                                <div class="tippable tippable-left"><span>%0</span></div>
                                <div class="tippable tippable-right"><span>100%</span></div>
                            </div>
                        </div>

                        <!-- Home Price -->
                        <div class="rh_mc_field">
                            <label for="mc_home_price"><?php echo esc_html( $price_label ); ?></label>
                            <input id="mc_home_price" class="mc_home_price" type="text" value="<?php echo esc_html( $mc_default['price_converted'] ); ?>">
                            <div class="rh_form__item rh-mc-slider-fields">
                                <input class="mc_home_price_slider" type="range" min="0" max="<?php echo esc_html( $mc_default['price_converted'] ); ?>" step="1" value="<?php echo esc_html( $mc_default['price_converted'] ); ?>">
                                <input class="mc_currency_sign" type="hidden" value="<?php echo esc_attr( $currency_sign ); ?>">
                                <input class="mc_sign_position" type="hidden" value="<?php echo esc_attr( $currency_position ); ?>">
                                <div class="tippable tippable-left"><span><?php echo esc_attr( $currency_sign ); ?>0</span></div>
                                <div class="tippable tippable-right"><span><?php echo esc_attr( $currency_sign ); ?><?php echo esc_html( $mc_default['price_converted'] ); ?></span></div>
                            </div>
                        </div>

                        <!-- Down Payment -->
                        <div class="rh_mc_field">
                            <label for="mc_downpayment"><?php echo esc_html( $downpayment_label ); ?></label>
                            <div class="rh-mc-fields-half">
                                <input id="mc_downpayment" class="mc_downpayment" type="text" value="<?php echo esc_html( ( $mc_default['price_converted'] * $mc_default['downpayment'] ) / 100 ); ?>">
                                <input class="mc_downpayment_percent" type="text" value="<?php echo esc_html( $mc_default['downpayment'] ); ?>%">
                            </div>
                            <div class="rh_form__item rh-mc-slider-fields">
                                <input class="mc_downpayment_slider" type="range" min="0" max="100" step="1" value="<?php echo esc_html( $mc_default['downpayment'] ); ?>">
                                <div class="tippable tippable-left"><span><?php echo esc_attr( $currency_sign ); ?>0</span></div>
                                <div class="tippable tippable-right"><span><?php echo esc_attr( $currency_sign ); ?><?php echo esc_html( $mc_default['price_converted'] ); ?></span></div>
                            </div>
                        </div>

                    </div><!-- End of the left side -->

					<?php $graph_type = get_option( 'inspiry_mc_graph_type', 'circle' ); ?>
                    <div class="mc_right_side <?php echo 'graph_' . esc_attr( $graph_type ); ?>">
						<?php
						$tax_field_display = get_option( 'inspiry_mc_first_field_display', 'true' );
						$hoa_field_display = get_option( 'inspiry_mc_second_field_display', 'true' );

						if ( 'true' !== $tax_field_display ) {
							$mc_default['tax_converted'] = '0';
							$mc_default['tax']           = '0';
						} else {
							$tax_field_label = get_option( 'inspiry_mc_first_field_title', esc_html__( 'Property Taxes', RHEA_TEXT_DOMAIN ) );
						}

						if ( 'true' !== $hoa_field_display ) {
							$mc_default['fee_converted'] = '0';
							$mc_default['fee']           = '0';
						} else {
							$hoa_field_label = get_option( 'inspiry_mc_second_field_title', esc_html__( 'Other Dues', RHEA_TEXT_DOMAIN ) );
						}

						$graph_type = get_option( 'inspiry_mc_graph_type', 'circle' );
						if ( 'circle' === $graph_type ) {
							?>
                            <div class="mc_term_interest">
                                <div class="rh-mc-label-box">
                                    <span class="mc_term_value rh-mc-value"><?php echo esc_html( $mc_default['term'] ); ?></span>
                                    <span class="rh-mc-label">
                                        <?php esc_html_e( 'Years Fixed', RHEA_TEXT_DOMAIN ); ?>
                                    </span>
                                </div>
                                <div class="rh-mc-label-box">
                                    <span class="mc_interest_value rh-mc-value test"><?php echo esc_attr( $mc_default['interest'] ); ?></span><span class="rh-mc-percent">%</span>
                                    <span class="rh-mc-label"><?php echo esc_html( $interest_label ); ?></span>
                                </div>
                            </div>
                            <div class="mc_cost_graph_circle">
                                <svg class="mc_graph_svg" width="220" height="220" viewPort="0 0 100 100">
                                    <circle class="mc_graph_hoa" r="90" cx="110" cy="110" fill="transparent" stroke-dasharray="565.48" stroke-dashoffset="0"></circle>
                                    <circle class="mc_graph_tax" r="90" cx="110" cy="110" fill="transparent" stroke-dasharray="565.48" stroke-dashoffset="0"></circle>
                                    <circle class="mc_graph_interest" r="90" cx="110" cy="110" fill="transparent" stroke-dasharray="565.48" stroke-dashoffset="0"></circle>
                                </svg>
                                <div class="mc_cost_over_graph" data-cost-prefix=" <?php echo esc_html( $cost_prefix ); ?>">
									<?php
									// Temporary Text Added for Editor Side only
									if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
										?>
                                        <strong>$999</strong>
										<?php
										echo esc_html( $cost_prefix );
									}
									?>
                                </div>
                            </div>
							<?php
						} else {
							?>
                            <div class="mc_term_interest">
								<?php echo esc_html( $cost_prefix ); ?>
								<?php esc_html_e( 'payment on a', RHEA_TEXT_DOMAIN ); ?>
                                <span class="mc_term_value"><?php echo esc_html( $mc_default['term'] ); ?></span>
								<?php esc_html_e( 'year loan with', RHEA_TEXT_DOMAIN ); ?>
                                <span class="mc_interest_value"><?php echo esc_attr( $mc_default['interest'] ); ?></span><span>%</span>
								<?php esc_html_e( 'fixed', RHEA_TEXT_DOMAIN ); ?>
								<?php echo esc_html( $interest_label ); ?>
								<?php esc_html_e( 'is:', RHEA_TEXT_DOMAIN ); ?>
                            </div>
                            <div class="mc_cost_total"><span></span></div>
                            <div class="mc_cost_graph">
                                <ul class="clearfix">
                                    <li class="mc_graph_interest"><span></span></li>
                                    <li class="mc_graph_tax"><span></span></li>
                                    <li class="mc_graph_hoa"><span></span></li>
                                </ul>
                            </div>
							<?php
						}
						?>

                        <div class="mc_cost">
                            <ul>
                                <li class="mc_cost_interest">
                                    <strong class="label"><?php echo esc_html( $principle_label ); ?></strong>
                                    <span></span>
                                </li>
								<?php
								if ( ! empty( $mc_default['tax'] ) ) {
									?>
                                    <li class="mc_cost_tax">
                                        <strong class="label"><?php echo esc_html( $tax_field_label ); ?></strong>
                                        <span><?php echo esc_html( ere_format_amount( $mc_default['tax'] ) ); ?></span>
                                    </li>
									<?php
								}

								if ( ! empty( $mc_default['fee'] ) ) {
									?>
                                    <li class="mc_cost_hoa">
                                        <strong class="label"><?php echo esc_html( $hoa_field_label ); ?></strong>
                                        <span><?php echo esc_html( ere_format_amount( $mc_default['fee'] ) ); ?></span>
                                    </li>
									<?php
								}
								?>
                            </ul>
                            <input class="mc_cost_tax_value" type="hidden" value="<?php echo esc_html( $mc_default['tax_converted'] ); ?>">
                            <input class="mc_cost_hoa_value" type="hidden" value="<?php echo esc_html( $mc_default['fee_converted'] ); ?>">
                        </div>

                    </div><!-- End of the right side -->

                </div>
            </div>
			<?php
		} else {
			rhea_print_no_result_for_editor();
		}

		if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
			?>
            <script type="application/javascript">
                rheaSelectPicker( '.inspiry_select_picker_trigger' );
            </script>
			<?php
		}
	}

	private function initialize_property_id() {
		if ( rhea_is_preview_mode() && rhea_get_sample_property_id() ) {
			return rhea_get_sample_property_id();
		}

		return get_the_ID();
	}
}