<?php
/**
 * Single Property Agent Widget V2.
 *
 * @since 2.4.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Single_Property_Agent_V2 extends \Elementor\Widget_Base {

	use RHEA_Single_Property_V2_Title_Section_Trait;

	private static $property_id;

	public function __construct( $data = [], $args = null ) {
		parent::__construct( $data, $args );

		if ( is_null( self::$property_id ) ) {
			self::$property_id = $this->initialize_property_id();
		}
	}

	public function get_name() {
		return 'rh-single-property-agent-v2';
	}

	public function get_title() {
		return esc_html__( 'RH: Single Property Agent V2', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		return 'eicon-person rh-ultra-widget';
	}

	public function get_categories() {
		return [ 'ultra-realhomes-single-property' ];
	}

	protected function register_controls() {

		$this->start_controls_section(
			'agent_card_section',
			[
				'label' => esc_html__( 'Basic Settings', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'section_title',
			[
				'label'     => esc_html__( 'Section Title', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'separator' => 'after'
			]
		);

		$this->add_control(
			'max_number_of_agents',
			[
				'label'   => esc_html__( 'Max Number of Agent', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 10,
				'step'    => 1,
				'default' => 1,
			]
		);

		$this->add_control(
			'show_agent_card',
			[
				'label'        => esc_html__( 'Show Agent Card', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'show_rating',
			[
				'label'        => esc_html__( 'Show Rating', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => [
					'show_agent_card' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_description',
			[
				'label'        => esc_html__( 'Show Description', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'show_agent_card' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_contact_info',
			[
				'label'        => esc_html__( 'Show Contact Info', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'show_agent_card' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_social_links',
			[
				'label'        => esc_html__( 'Show Social Links', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'show_agent_card' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_agent_form',
			[
				'label'        => esc_html__( 'Show Agent Contact Form', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'agents_labels',
			[
				'label'     => esc_html__( 'Labels', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'show_agent_card' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_label',
			[
				'label'     => esc_html__( 'Agent Label', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Agent', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_card' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_office_contact_label',
			[
				'label'     => esc_html__( 'Office Contact', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Office:', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_card'   => 'yes',
					'show_contact_info' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_office_contact_mobile',
			[
				'label'     => esc_html__( 'Mobile Contact', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Mobile:', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_card'   => 'yes',
					'show_contact_info' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_office_contact_fax',
			[
				'label'     => esc_html__( 'Fax Contact', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Fax:', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_card'   => 'yes',
					'show_contact_info' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_office_contact_whatsapp',
			[
				'label'     => esc_html__( 'WhatsApp Contact', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'WhatsApp:', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_card'   => 'yes',
					'show_contact_info' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_office_contact_email',
			[
				'label'     => esc_html__( 'Email Contact', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Email:', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_card'   => 'yes',
					'show_contact_info' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_form_title',
			[
				'label'     => esc_html__( 'Form Title', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Drop Your Message', RHEA_TEXT_DOMAIN ),
				'separator'   => 'before',
				'condition' => [
					'show_agent_form' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_name_placeholder',
			[
				'label'     => esc_html__( 'Name Field Placeholder', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Name', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_form' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_email_placeholder',
			[
				'label'     => esc_html__( 'Email Field Placeholder', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Email', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_form' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_phone_placeholder',
			[
				'label'     => esc_html__( 'Phone Field Placeholder', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Phone', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_form' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_textarea_placeholder',
			[
				'label'     => esc_html__( 'Textarea Field Placeholder', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Message', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_form' => 'yes',
				],
			]
		);

		$this->add_control(
			'agent_submit_button_text',
			[
				'label'     => esc_html__( 'Submit Button Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Send Message', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'show_agent_form' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->title_section_style_tab_controls();

		$this->start_controls_section(
			'agent_container_styles',
			[
				'label' => esc_html__( 'Container', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'agent_container_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'agent_container_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'agent_container_box_shadow',
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-inner',
			]
		);

		$this->add_control(
			'agent-container-bg',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'agent_container_items_row_gap',
			[
				'label'     => esc_html__( 'Items Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'agent_styles',
			[
				'label' => esc_html__( 'Agent Info', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'agent_thumbnail',
			[
				'label' => esc_html__( 'Agent Thumbnail', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_responsive_control(
			'agent_thumbnail_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-thumb' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'agent_thumbnail_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-thumb, {{WRAPPER}} .rh-single-property-agent-v2-agent-thumb img' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->add_control(
			'agent_verification_badge_color',
			[
				'label'     => esc_html__( 'Verification Badge Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh_main_verification__icon' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_verification_badge_tick_color',
			[
				'label'     => esc_html__( 'Verification Badge Tick Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh_main_verification__icon svg' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_title',
			[
				'label'     => esc_html__( 'Agent Title', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-agent-title',
			]
		);

		$this->add_control(
			'agent_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-title a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_title_hover_color',
			[
				'label'     => esc_html__( 'Color Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-title a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_label_heading',
			[
				'label'     => esc_html__( 'Agent Label', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_label_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-agent-label',
			]
		);

		$this->add_control(
			'agent_label_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_rating_heading',
			[
				'label'     => esc_html__( 'Agent Rating', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'agent_rating_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2 .stars-avg-rating' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_rating_text_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2 .rating-span',
			]
		);

		$this->add_control(
			'agent_rating_text_color',
			[
				'label'     => esc_html__( 'Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2 .rating-span' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_rating_color',
			[
				'label'     => esc_html__( 'Rating Stars', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2 .rating-stars' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_rated_star_color',
			[
				'label'     => esc_html__( 'Rated Stars', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2 .rating-stars .rated' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'agent_description_styles',
			[
				'label' => esc_html__( 'Description', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'agent_thumbnail_margin_top',
			[
				'label'     => esc_html__( 'Margin Top', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-description' => 'margin-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_description_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-agent-description',
			]
		);

		$this->add_control(
			'agent_description_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-description' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'agent_contact_details_styles',
			[
				'label' => esc_html__( 'Contact Details', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'agent_contact_details_margin_top',
			[
				'label'     => esc_html__( 'Margin Top', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-contact-info' => 'margin-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'agent_contact_details_items_row_gap',
			[
				'label'     => esc_html__( 'Items Row Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-contact-info' => 'row-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_contact_details_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-agent-contact-info p',
			]
		);

		$this->add_control(
			'agent_contact_details_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-contact-info p' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_contact_details_link_typography',
				'label'    => esc_html__( 'Link Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-agent-contact-info p a',
			]
		);

		$this->add_control(
			'agent_contact_details_link_color',
			[
				'label'     => esc_html__( 'Link Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-contact-info a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_details_link_hover_color',
			[
				'label'     => esc_html__( 'Link Color Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-contact-info a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'agent_social_links_styles',
			[
				'label' => esc_html__( 'Social Links', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'agent_social_links_margin_top',
			[
				'label'     => esc_html__( 'Margin Top', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-social-links' => 'margin-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'agent_social_links_items_column_gap',
			[
				'label'     => esc_html__( 'Items Column Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-social-links' => 'column-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'agent_social_links_items_row_gap',
			[
				'label'     => esc_html__( 'Items Row Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-social-links' => 'row-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'agent_social_links_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-social-links a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_social_links_hover_color',
			[
				'label'     => esc_html__( 'Color Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-agent-social-links a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'agent_contact_form_styles',
			[
				'label' => esc_html__( 'Contact Form', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_title_section_heading',
			[
				'label'     => esc_html__( 'Form Title', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_title_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_contact_form_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-form-title',
			]
		);

		$this->add_control(
			'agent_contact_form_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-title' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_fields_section_heading',
			[
				'label'     => esc_html__( 'Fields', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_fields_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form p, {{WRAPPER}} .rh_inspiry_gdpr' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_fields_padding',
			[
				'label'      => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-field' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_fields_border_radius',
			[
				'label'      => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-field' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_fields_bg_color',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-field,
					 {{WRAPPER}} .rh-inspiry-gdpr input:checked + label:before, 
					 {{WRAPPER}} .rh_inspiry_gdpr input:checked + label:before,
					 {{WRAPPER}} .comment-form-cookies-consent input:checked + label:before' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_fields_border_color',
			[
				'label'     => esc_html__( 'Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-field,
					 {{WRAPPER}} .rh-single-property-agent-v2-inner  .rh_inspiry_gdpr label:before' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_contact_form_fields_text_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-form-field',
			]
		);

		$this->add_control(
			'agent_contact_form_fields_text_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-field,
					 {{WRAPPER}} .rh-inspiry-gdpr input:checked + label:before, 
					 {{WRAPPER}} .rh_inspiry_gdpr input:checked + label:before' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_fields_placeholder_color',
			[
				'label'     => esc_html__( 'Placeholder Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-form-wrapper ::-webkit-input-placeholder' => 'color: {{VALUE}}',
					'{{WRAPPER}} .rh-single-property-agent-v2-form-wrapper :-ms-input-placeholder'      => 'color: {{VALUE}}',
					'{{WRAPPER}} .rh-single-property-agent-v2-form-wrapper ::placeholder'               => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_gdpr_title_section_heading',
			[
				'label'     => esc_html__( 'GDPR', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_gdpr_title_margin_bottom',
			[
				'label'     => esc_html__( 'Title Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner  .gdpr-checkbox-label' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_contact_form_gdpr_title_typography',
				'label'    => esc_html__( 'Title Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-inner  .gdpr-checkbox-label',
			]
		);

		$this->add_control(
			'agent_contact_form_gdpr_title_color',
			[
				'label'     => esc_html__( 'Title Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner  .gdpr-checkbox-label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_gdpr_checkbox_margin',
			[
				'label'     => esc_html__( 'Checkbox Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh_inspiry_gdpr label:before' => 'Margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_contact_form_gdpr_description_typography',
				'label'    => esc_html__( 'Description Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-inner  .rh-gdpr-text-inner-wrapper',
			]
		);

		$this->add_control(
			'agent_contact_form_gdpr_description_color',
			[
				'label'     => esc_html__( 'Description Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner  .rh-gdpr-text-inner-wrapper' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_submit_button_section_heading',
			[
				'label'     => esc_html__( 'Submit Button', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_submit_button_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner  .submit-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'agent_contact_form_submit_button_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner  .submit-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_contact_form_submit_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-agent-v2-inner  .submit-button',
			]
		);

		$this->add_control(
			'agent_contact_form_submit_button_text_color',
			[
				'label'     => esc_html__( 'Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner  .submit-button' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_submit_button_text_color_hover',
			[
				'label'     => esc_html__( 'Text Color Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner  .submit-button:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_submit_button_bg_color',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner  .submit-button' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_form_submit_button_bg_hover_color',
			[
				'label'     => esc_html__( 'Background Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-agent-v2-inner  .submit-button:hover' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'agent_contact_error_color',
			[
				'label'     => esc_html__( 'Error Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .error-container label' => 'color: {{VALUE}}',
				],
				'separator' => 'before'
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		// Check if login required to vew property detail.
		if ( inspiry_prop_detail_login() ) {
			return;
		}

		$settings       = $this->get_settings_for_display();
		$widget_id      = $this->get_id();
		$display_option = get_post_meta( self::$property_id, 'REAL_HOMES_agent_display_option', true );

		if ( ( 'none' === $display_option ) ) {
			return;
		}

		if ( 'yes' !== $settings['show_agent_card'] && 'yes' !== $settings['show_agent_form'] ) {
			return;
		}

		if ( ! empty( $settings['section_title'] ) ) {
			?>
            <h4 class="rh-single-property-section-heading rh-single-property-section-heading-common"><?php echo esc_html( $settings['section_title'] ); ?></h4>
			<?php
		}
		?>
        <div id="rh-single-property-agent-v2-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-agent-v2 <?php echo realhomes_printable_section( 'agent', false ) . ' ' . ( ! empty( $args['agent_class'] ) ? esc_attr( $args['agent_class'] ) : '' ); ?>">
			<?php
			if ( 'my_profile_info' == $display_option ) {
				$profile_args                        = array();
				$profile_args['display_author']      = true;
				$profile_args['agent_id']            = '';
				$profile_args['author_id']           = get_the_author_meta( 'ID' );
				$profile_args['author_display_name'] = get_the_author_meta( 'display_name' );
				$profile_args['agent_title_text']    = esc_html__( 'Submitted by', RHEA_TEXT_DOMAIN ) . ' ' . get_the_author_meta( 'display_name' );
				$profile_args['profile_image_id']    = intval( get_the_author_meta( 'profile_image_id' ) );
				$profile_args['agents_count']        = 1;
				$profile_args['agent_mobile']        = get_the_author_meta( 'mobile_number' );
				$profile_args['agent_whatsapp']      = get_the_author_meta( 'mobile_whatsapp' );
				$profile_args['agent_office_phone']  = get_the_author_meta( 'office_number' );
				$profile_args['agent_office_fax']    = get_the_author_meta( 'fax_number' );
				$profile_args['agent_email']         = get_the_author_meta( 'user_email' );
				$profile_args['agent_description']   = get_framework_custom_excerpt( get_the_author_meta( 'description' ), 18 );
				$this->display_agent( $profile_args, $profile_args['author_id'], $settings );

			} else {
				$property_agents = get_post_meta( self::$property_id, 'REAL_HOMES_agents' );

				// Remove invalid ids.
				$property_agents = array_filter( $property_agents, function ( $agent_id ) {
					return ( $agent_id > 0 );
				} );

				// Remove duplicated ids.
				$property_agents = array_unique( $property_agents );
				if ( empty( $property_agents ) ) {
					rhea_print_no_result_for_editor();

					return;
				}

				$agents_count   = count( $property_agents );
				$agents_counter = 1;
				$max_agents     = intval( $settings['max_number_of_agents'] );

				foreach ( $property_agents as $agent ) {
					if ( 0 < intval( $agent ) ) {
						$agent_args                       = array();
						$agent_args['agent_id']           = intval( $agent );
						$agent_args['agent_label']        = $settings['agent_label'];
						$agent_args['agents_count']       = $agents_count;
						$agent_args['agent_title_text']   = esc_html( get_the_title( $agent_args['agent_id'] ) );
						$agent_args['agent_mobile']       = get_post_meta( $agent_args['agent_id'], 'REAL_HOMES_mobile_number', true );
						$agent_args['agent_whatsapp']     = get_post_meta( $agent_args['agent_id'], 'REAL_HOMES_whatsapp_number', true );
						$agent_args['agent_office_phone'] = get_post_meta( $agent_args['agent_id'], 'REAL_HOMES_office_number', true );
						$agent_args['agent_office_fax']   = get_post_meta( $agent_args['agent_id'], 'REAL_HOMES_fax_number', true );
						$agent_args['agent_email']        = get_post_meta( $agent_args['agent_id'], 'REAL_HOMES_agent_email', true );

						// Excerpt for agent is bit tricky as we have to get excerpt if available otherwise post contents
						$temp_agent_excerpt = get_post_field( 'post_excerpt', $agent_args['agent_id'] );
						if ( is_wp_error( $temp_agent_excerpt ) || empty( $temp_agent_excerpt ) ) {
							$agent_args['agent_excerpt'] = get_post_field( 'post_content', $agent_args['agent_id'] );
						} else {
							$agent_args['agent_excerpt'] = $temp_agent_excerpt;
						}

						$agent_args['agent_description'] = get_framework_custom_excerpt( $agent_args['agent_excerpt'], 18 );
						$this->display_agent( $agent_args, $agent_args['agent_id'], $settings );
					}

					if ( $max_agents === $agents_counter ) {
						break;
					}

					$agents_counter++;
				}
			}
			?>
        </div>
		<?php
		$this->get_script( $widget_id );
	}

	protected function display_agent( $args, $agent_id, $settings ) {
		?>
        <div id="rh-single-property-agent-<?php echo esc_attr( $agent_id ); ?>" class="rh-single-property-agent-v2-inner">
			<?php
			if ( 'yes' === $settings['show_agent_card'] ) {
				?>
                <div class="rh-single-property-agent-v2-agent-card">
                    <div class="rh-single-property-agent-v2-agent-thumb">
						<?php
						if ( ! empty( $args['display_author'] ) ) {
							if ( isset( $args['profile_image_id'] ) && ( 0 < $args['profile_image_id'] ) ) {
								?>
                                <a class="agent-image" href="<?php echo esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ); ?>">
									<?php echo wp_get_attachment_image( $args['profile_image_id'], 'agent-image' ); ?>
                                </a>
								<?php
							} else if ( isset( $args['agent_email'] ) ) {
								?>
                                <a class="agent-image" href="<?php echo esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ); ?>">
									<?php echo get_avatar( $args['agent_email'], '210' ); ?>
                                </a>
								<?php
							}
						} else if ( isset( $args['agent_id'] ) && ( ! empty( get_the_post_thumbnail( $args['agent_id'] ) ) ) ) {
							?>
                            <a class="agent-image" href="<?php echo esc_url( get_permalink( $args['agent_id'] ) ); ?>">
								<?php echo get_the_post_thumbnail( $args['agent_id'], 'agent-image' ); ?>
                            </a>
							<?php
						} else {
							?>
                            <a class="agent-image agent-property-placeholder" href="<?php echo esc_url( get_permalink( $args['agent_id'] ) ); ?>">
                                <i class="fas fa-user-tie"></i>
                            </a>
							<?php
						}
						?>
                    </div>
					<?php
					if ( isset( $args['display_author'] ) && ( $args['display_author'] ) ) {
						$agent_url = get_author_posts_url( get_the_author_meta( 'ID' ) );
					} else {
						$agent_url = get_permalink( $args['agent_id'] );
					}
					?>
                    <div class="rh-single-property-agent-v2-agent-title-wrapper">
						<?php
						if ( ! empty( $args['agent_title_text'] ) ) {
							?>
                            <h3 class="rh-single-property-agent-v2-agent-title">
                                <a href="<?php echo esc_url( $agent_url ) ?>"><?php echo esc_html( $args['agent_title_text'] ); ?></a>
								<?php
								if ( 0 < intval( $args['agent_id'] ) && function_exists( 'realhomes_verification_badge' ) ) {
									realhomes_verification_badge( 'agent', $args['agent_id'] );
								}
								?>
                            </h3>
							<?php
						}

						if ( ! empty( $args['agent_label'] ) ) {
							?>
                            <h4 class="rh-single-property-agent-v2-agent-label"><?php echo esc_html( $args['agent_label'] ) ?></h4>
							<?php
						}
						?>
                    </div>
					<?php
					// Display average rating for the agent
					if ( 'yes' === $settings['show_rating'] ) {
						inspiry_rating_average( $settings, $agent_id );
					}

					if ( 'yes' === $settings['show_description'] && ! empty( $args['agent_description'] ) ) {
						?>
                        <p class="rh-single-property-agent-v2-agent-description"><?php echo esc_html( $args['agent_description'] ); ?></p>
						<?php
					}

					if ( 'yes' === $settings['show_contact_info'] ) {
						?>
                        <div class="rh-single-property-agent-v2-agent-contact-info">
							<?php
							if ( ! empty( $args['agent_office_phone'] ) ) {
								?>
                                <p class="contact office">
									<?php
									if ( ! empty( $settings['agent_office_contact_label'] ) ) {
										?>
                                        <span><?php echo esc_html( $settings['agent_office_contact_label'] ); ?></span>
										<?php
									}
									?>
                                    <a href="tel:<?php echo esc_html( $args['agent_office_phone'] ); ?>">
										<?php echo esc_html( $args['agent_office_phone'] ); ?>
                                    </a>
                                </p>
								<?php
							}

							if ( ! empty( $args['agent_whatsapp'] ) ) {
								?>
                                <p class="contact whatsapp">
									<?php
									if ( ! empty( $settings['agent_office_contact_whatsapp'] ) ) {
										?>
                                        <span><?php echo esc_html( $settings['agent_office_contact_whatsapp'] ); ?></span>
										<?php
									}
									?>
                                    <a href="https://wa.me/<?php echo esc_html( $args['agent_whatsapp'] ); ?>">
										<?php echo esc_html( $args['agent_whatsapp'] ); ?>
                                    </a>
                                </p>
								<?php
							}

							if ( ! empty( $args['agent_mobile'] ) ) {
								?>
                                <p class="contact mobile">
									<?php
									if ( ! empty( $settings['agent_office_contact_mobile'] ) ) {
										?>
                                        <span><?php echo esc_html( $settings['agent_office_contact_mobile'] ); ?></span>
										<?php
									}
									?>
                                    <a href="tel:<?php echo esc_html( $args['agent_mobile'] ); ?>">
										<?php echo esc_html( $args['agent_mobile'] ); ?>
                                    </a>
                                </p>
								<?php
							}

							if ( ! empty( $args['agent_email'] ) ) {
								?>
                                <p class="contact email">
									<?php
									if ( ! empty( $settings['agent_office_contact_email'] ) ) {
										?>
                                        <span><?php echo esc_html( $settings['agent_office_contact_email'] ); ?></span>
										<?php
									}
									?>
                                    <a href="mailto:<?php echo esc_attr( antispambot( $args['agent_email'] ) ); ?>">
										<?php echo esc_html( antispambot( $args['agent_email'] ) ); ?>
                                    </a>
                                </p>
								<?php
							}

							if ( ! empty( $args['agent_office_fax'] ) ) {
								?>
                                <p class="contact fax">
									<?php
									if ( ! empty( $settings['agent_office_contact_fax'] ) ) {
										?>
                                        <span><?php echo esc_html( $settings['agent_office_contact_fax'] ); ?></span>
										<?php
									}
									?>
                                    <a href="fax:<?php echo esc_html( $args['agent_office_fax'] ); ?>">
										<?php echo esc_html( $args['agent_office_fax'] ); ?>
                                    </a>
                                </p>
								<?php
							}
							?>
                        </div>
						<?php
					}

					if ( 'yes' === $settings['show_social_links'] ) {
						$facebook_url  = get_post_meta( $agent_id, 'REAL_HOMES_facebook_url', true );
						$twitter_url   = get_post_meta( $agent_id, 'REAL_HOMES_twitter_url', true );
						$linked_in_url = get_post_meta( $agent_id, 'REAL_HOMES_linked_in_url', true );
						$instagram_url = get_post_meta( $agent_id, 'inspiry_instagram_url', true );
						$youtube_url   = get_post_meta( $agent_id, 'inspiry_youtube_url', true );
						$tiktok_url    = get_post_meta( $agent_id, 'realhomes_tiktok_url', true );
						$pinterest_url = get_post_meta( $agent_id, 'inspiry_pinterest_url', true );
						$agent_website = get_post_meta( $agent_id, 'REAL_HOMES_website', true );

						if ( ! empty( $facebook_url ) || ! empty( $twitter_url ) || ! empty( $linked_in_url ) || ! empty( $instagram_url ) || ! empty( $youtube_url ) || ! empty( $tiktok_url ) || ! empty( $pinterest_url ) || ! empty( $agent_website ) ) {
							?>
                            <div class="rh-single-property-agent-v2-agent-social-links">
								<?php
								if ( ! empty( $facebook_url ) ) {
									?><a class="facebook" target="_blank" href="<?php echo esc_url( $facebook_url ); ?>"><i class="fab fa-facebook-f fa-lg"></i></a><?php
								}

								if ( ! empty( $twitter_url ) ) {
									?>
                                    <a class="twitter" target="_blank" href="<?php echo esc_url( $twitter_url ); ?>">
                                        <svg aria-hidden="true" class="e-font-icon-svg e-fab-x-twitter" viewBox="0 0 512 512" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M389.2 48h70.6L305.6 224.2 487 464H345L233.7 318.6 106.5 464H35.8L200.7 275.5 26.8 48H172.4L272.9 180.9 389.2 48zM364.4 421.8h39.1L151.1 88h-42L364.4 421.8z"></path>
                                        </svg>
                                    </a>
									<?php
								}

								if ( ! empty( $linked_in_url ) ) {
									?><a class="linkedin" target="_blank" href="<?php echo esc_url( $linked_in_url ); ?>"><i class="fab fa-linkedin-in fa-lg"></i></a><?php
								}

								if ( ! empty( $tiktok_url ) ) {
									?><a class="tiktok" target="_blank" href="<?php echo esc_url( $tiktok_url ); ?>"><i class="fab fa-tiktok fa-lg"></i></a><?php
								}

								if ( ! empty( $pinterest_url ) ) {
									?><a class="pinterest" target="_blank" href="<?php echo esc_url( $pinterest_url ); ?>"><i class="fab fa-pinterest-p fa-lg"></i></a><?php
								}

								if ( ! empty( $instagram_url ) ) {
									?><a class="instagram" target="_blank" href="<?php echo esc_url( $instagram_url ); ?>"><i class="fab fa-instagram fa-lg"></i></a><?php
								}

								if ( ! empty( $youtube_url ) ) {
									?><a class="youtube" target="_blank" href="<?php echo esc_url( $youtube_url ); ?>"><i class="fab fa-youtube fa-lg"></i></a><?php
								}

								if ( ! empty( $agent_website ) ) {
									?>
                                    <a class="website-icon" target="_blank" href="<?php echo esc_url( $agent_website ); ?>"><i class="fas fa-globe fa-lg"></i></a>
									<?php
								}
								?>
                            </div>
							<?php
						}
					}
					?>
                </div>
				<?php
			}

			if ( 'yes' === $settings['show_agent_form'] ) {
				if ( inspiry_get_agent_custom_form( $args['agent_id'] ) ) {
					inspiry_agent_custom_form( $args['agent_id'] );

				} else if ( ! empty( $args['agent_email'] ) ) {
					$user_name = $user_email = $user_phone = '';
					if ( is_user_logged_in() ) {
						$current_user = wp_get_current_user();
						$user_name    = ! empty( $current_user->display_name ) ? $current_user->display_name : $current_user->user_login;
						$user_email   = $current_user->user_email;
						$user_phone   = get_user_meta( $current_user->ID, 'mobile_number', true );
					}
					?>
                    <div class="rh-single-property-agent-v2-form-wrapper">
	                    <?php
	                    if ( ! empty( $settings['agent_form_title'] ) ) {
		                    ?>
                            <h4 class="rh-single-property-agent-v2-form-title"><?php echo esc_html( $settings['agent_form_title'] ); ?></h4>
		                    <?php
	                    }
	                    ?>
                        <form class="rh-single-property-agent-v2-form" method="post" action="<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>">
                            <p>
                                <input id="rh-enquiry-name" type="text" name="name" placeholder="<?php echo esc_html( $settings['agent_name_placeholder'] ); ?>" class="required rh-single-property-agent-v2-form-field" value="<?php echo esc_attr( $user_name ); ?>" title="<?php esc_html_e( '* Please provide your name', RHEA_TEXT_DOMAIN ); ?>" />
                            </p>
                            <p>
                                <input id="rh-enquiry-email" type="text" name="email" placeholder="<?php echo esc_html( $settings['agent_email_placeholder'] ); ?>" class="required rh-single-property-agent-v2-form-field" value="<?php echo esc_attr( $user_email ); ?>" title="<?php esc_html_e( '* Please provide valid email address', RHEA_TEXT_DOMAIN ); ?>" />
                            </p>
                            <p>
                                <input id="rh-enquiry-phone" type="text" name="phone" placeholder="<?php echo esc_html( $settings['agent_phone_placeholder'] ); ?>" class="required rh-single-property-agent-v2-form-field" value="<?php echo esc_attr( $user_phone ); ?>" title="<?php esc_html_e( '* Please provide valid phone number', RHEA_TEXT_DOMAIN ); ?>" />
                            </p>
                            <p>
                                <textarea id="rh-enquiry-message" cols="40" rows="6" name="message" placeholder="<?php echo esc_html( $settings['agent_textarea_placeholder'] ); ?>" class="required rh-single-property-agent-v2-form-field" title="<?php esc_html_e( '* Please provide your message', RHEA_TEXT_DOMAIN ); ?>"><?php
	                                printf( esc_textarea( realhomes_get_agent_default_message() ), esc_html( get_the_title( self::$property_id ) ) );
	                                ?></textarea>
                            </p>
							<?php
							if ( function_exists( 'ere_gdpr_agreement' ) ) {
								ere_gdpr_agreement( array(
									'id'              => 'rh_inspiry_gdpr',
									'container_class' => 'rh_inspiry_gdpr',
									'title_class'     => 'gdpr-checkbox-label'
								) );
							}

							if ( class_exists( 'Easy_Real_Estate' ) ) {
								if ( ere_is_reCAPTCHA_configured() ) {
									$recaptcha_type = get_option( 'inspiry_reCAPTCHA_type', 'v2' );
									?>
                                    <div class="rh_modal__recaptcha">
                                        <div class="inspiry-recaptcha-wrapper clearfix g-recaptcha-type-<?php echo esc_attr( $recaptcha_type ); ?>">
                                            <div class="inspiry-google-recaptcha"></div>
                                        </div>
                                    </div>
									<?php
								}
							}
							?>
                            <input type="hidden" name="nonce" value="<?php echo esc_attr( wp_create_nonce( 'agent_message_nonce' ) ); ?>" />
                            <input type="hidden" name="target" value="<?php echo esc_attr( antispambot( $args['agent_email'] ) ); ?>">
							<?php
							if ( ! empty( $args['agent_id'] ) ) {
								?>
                                <input type="hidden" name="agent_id" value="<?php echo esc_attr( $args['agent_id'] ); ?>">
								<?php
							}
							if ( ! empty( $args['author_id'] ) ) {
								?>
                                <input type="hidden" name="author_id" value="<?php echo esc_attr( $args['author_id'] ) ?>">
								<?php
							}
							?>
                            <input type="hidden" name="author_name" value="<?php echo esc_attr( $args['agent_title_text'] ); ?>">
                            <input type="hidden" name="property_id" value="<?php echo esc_attr( self::$property_id ) ?>">
                            <input type="hidden" name="action" value="send_message_to_agent" />
                            <input type="hidden" name="property_title" value="<?php echo esc_attr( get_the_title( self::$property_id ) ); ?>" />
                            <input type="hidden" name="property_permalink" value="<?php echo esc_url_raw( get_permalink( self::$property_id ) ); ?>" />
                            <button type="submit" name="submit" class="submit-button">
                                <span class="btn-text"><?php echo esc_html( $settings['agent_submit_button_text'] ); ?></span>
                                <span id="ajax-loader-<?php echo esc_attr( $agent_id ); ?>" class="ajax-loader"><?php inspiry_safe_include_svg( '/icons/ball-triangle.svg' ); ?></span>
                            </button>
                            <div class="error-container"></div>
                            <div class="message-container"></div>
                        </form>
                    </div>
					<?php
				}
			}
			?>
        </div>
		<?php
	}

	protected function get_script( $widget_id ) {
		?>
        <script>
            jQuery( document ).ready( function ( $ ) {
                "use strict";

                /**
                 * Validate and handle the submission of a form.
                 * @param {string} formDiv - The ID of the form container.
                 */
                function rhValidateForm( formDiv ) {
                    const $form            = $( "#" + formDiv ).find( '.rh-single-property-agent-v2-form' ),
                          submitButton     = $form.find( '.submit-button' ),
                          ajaxLoader       = $form.find( '.ajax-loader' ),
                          messageContainer = $form.find( '.message-container' ),
                          errorContainer   = $form.find( ".error-container" );

                    const formOptions = {
                        beforeSubmit : function () {
                            submitButton.prop( 'disabled', true );
                            ajaxLoader.fadeIn( 'fast' ).css( "display", "inline-block" );
                            messageContainer.fadeOut( 'fast' );
                            errorContainer.fadeOut( 'fast' );
                        },
                        success      : function ( ajaxResponse ) {
                            const response = $.parseJSON( ajaxResponse );

                            ajaxLoader.fadeOut( 'fast' );
                            submitButton.prop( 'disabled', false );

                            if ( response.success ) {
                                // Reset the form and show success message
                                $form.trigger( 'reset' );
                                messageContainer.html( response.message ).fadeIn( 'fast' );

                                // Call reset function for reCAPTCHA if available
                                if ( typeof inspiryResetReCAPTCHA === 'function' ) {
                                    inspiryResetReCAPTCHA();
                                }

                                // Handle redirection if agentData is defined
                                if ( typeof agentData !== 'undefined' && agentData.redirectPageUrl ) {
                                    setTimeout( () => {
                                        window.location.replace( agentData.redirectPageUrl );
                                    }, 1000 );
                                } else {
                                    setTimeout( () => {
                                        messageContainer.fadeOut( 'slow' );
                                    }, 3000 );
                                }
                            } else {
                                // Show error message
                                errorContainer.html( response.message ).fadeIn( 'fast' );
                            }
                        }
                    };

                    // Initialize form validation
                    $form.validate( {
                        errorLabelContainer : errorContainer,
                        submitHandler       : function ( form ) {
                            $( form ).ajaxSubmit( formOptions );
                        }
                    } );
                }

                // Check if validation and ajaxSubmit are available
                if ( $.fn.validate && $.fn.ajaxSubmit ) {
                    const agentDivs = $( '#rh-single-property-agent-v2-<?php echo esc_attr( $widget_id ); ?> .rh-single-property-agent-v2-inner' );
                    if ( agentDivs.length ) {
                        agentDivs.each( function () {
                            const formDiv = $( this ).attr( "id" );
                            if ( formDiv ) {
                                rhValidateForm( formDiv );
                            }
                        } );
                    }
                }
            } );
        </script>
		<?php
	}

	private function initialize_property_id() {
		if ( rhea_is_preview_mode() && rhea_get_sample_property_id() ) {
			return rhea_get_sample_property_id();
		}

		return get_the_ID();
	}
}