<?php
/**
 * Elementor Properties Widget V14.
 *
 * @since 2.4.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Properties_Widget_V14 extends \Elementor\Widget_Base {

	use RHEA_Properties_Partials;
	use RHEA_Properties_Action_buttons;

	private $current_widget_id;

	private $is_rvr_enabled;

	public function __construct( array $data = [], ?array $args = null ) {
		parent::__construct( $data, $args );

		$this->is_rvr_enabled = rhea_is_rvr_enabled();

		wp_register_script( 'rh-properties-widget-v14', RHEA_PLUGIN_URL . 'elementor/js/properties-widget-v14.js', [ 'elementor-frontend' ], RHEA_VERSION, true );

		wp_localize_script( 'rh-properties-widget-v14', 'ajax_object', [
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce'    => wp_create_nonce( 'filter_properties_nonce' ),
		] );
	}

	public function get_name() {
		return 'rh-properties-widget-v14';
	}

	public function get_title() {
		return esc_html__( 'RH: Properties V14', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		// More classes for icons can be found at https://pojome.github.io/elementor-icons/
		return 'eicon-posts-grid rh-ultra-widget';
	}

	public function get_keywords() {
		return [ 'RealHomes', 'Ultra', 'Carousel', 'List', 'Grid', 'Properties Carousel', 'Properties List', 'Properties Grid' ];
	}

	public function get_categories() {
		return [ 'ultra-real-homes' ];
	}

	public function get_script_depends() {
		return [ 'rh-properties-widget-v14' ];
	}

	public function get_style_depends() {
		return [ 'swiper' ];
	}

	protected function register_controls() {

		$allowed_html = array(
			'a'      => array(
				'href'  => array(),
				'title' => array()
			),
			'br'     => array(),
			'em'     => array(),
			'strong' => array(),
		);

		$this->start_controls_section(
			'properties_section',
			[
				'label' => esc_html__( 'Properties', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'enable_search',
			[
				'label'              => esc_html__( 'Enable Property Search Filtering', RHEA_TEXT_DOMAIN ),
				'description'        => esc_html__( 'Enabling this will filter the property cards on search results page.', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'condition'          => [
					'single_agent_agency_properties' => [ '', 'no' ],
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'single_agent_agency_properties',
			[
				'label'              => esc_html__( 'Display Properties for Agent/Agency Single Page', RHEA_TEXT_DOMAIN ),
				'description'        => esc_html__( 'Enabling this will display properties only on agent/agency single page based on agent/agency ID.', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'condition'          => [
					'enable_search' => [ '', 'no' ],
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'layout',
			[
				'label'              => esc_html__( 'Layout', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'options'            => [
					'carousel' => esc_html__( 'Carousel', RHEA_TEXT_DOMAIN ),
					'grid'     => esc_html__( 'Grid', RHEA_TEXT_DOMAIN ),
					'list'     => esc_html__( 'List', RHEA_TEXT_DOMAIN ),
				],
				'default'            => 'carousel',
				'frontend_available' => true,
			]
		);

		$this->add_responsive_control(
			'grid_columns',
			[
				'label'              => esc_html__( 'Grid Columns', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'default'            => 3,
				'options'            => [
					1 => 1,
					2 => 2,
					3 => 3,
					4 => 4,
					5 => 5,
					6 => 6,
				],
				'condition'          => [
					'layout' => 'grid',
				],
				'frontend_available' => true,
			]
		);

		$this->add_responsive_control(
			'list_columns',
			[
				'label'     => esc_html__( 'List Columns', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 1,
				'options'   => [
					1 => 1,
					2 => 2,
					3 => 3
				],
				'condition' => [
					'layout' => 'list',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name'               => 'thumbnail',
				'default'            => 'modern-property-child-slider',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'posts_per_page',
			[
				'label'              => esc_html__( 'Properties Per Page', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::NUMBER,
				'min'                => 1,
				'max'                => 60,
				'step'               => 1,
				'default'            => 6,
				'frontend_available' => true,
			]
		);

		$get_agents = array();
		$agent_ids  = get_posts( array(
			'fields'         => 'ids',
			'posts_per_page' => -1,
			'post_type'      => 'agent'
		) );

		if ( ! empty( $agent_ids ) ) {
			foreach ( $agent_ids as $id ) {
				$get_agents["$id"] = get_the_title( $id );
			}
		}
		$this->add_control(
			'select_agent',
			[
				'label'              => esc_html__( 'Agent Based Properties', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT2,
				'default'            => '',
				'options'            => $get_agents,
				'label_block'        => true,
				'condition'          => [
					'single_agent_agency_properties' => [ '', 'no' ],
				],
				'frontend_available' => true,
			]
		);

		// Select controls for Custom Taxonomies related to Property
		$property_taxonomies = get_object_taxonomies( 'property', 'objects' );
		if ( ! empty( $property_taxonomies ) && ! is_wp_error( $property_taxonomies ) ) {
			foreach ( $property_taxonomies as $single_tax ) {
				$options = [];
				$terms   = get_terms( $single_tax->name );

				if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
					foreach ( $terms as $term ) {
						$options[ $term->slug ] = $term->name;
					}
				}

				$this->add_control(
					$single_tax->name,
					[
						'label'       => $single_tax->label,
						'type'        => \Elementor\Controls_Manager::SELECT2,
						'multiple'    => true,
						'label_block' => true,
						'options'     => $options,
					]
				);
			}
		}

		// Sorting Controls
		$this->add_control(
			'orderby',
			[
				'label'              => esc_html__( 'Order By', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'options'            => [
					'date'       => esc_html__( 'Date', RHEA_TEXT_DOMAIN ),
					'price'      => esc_html__( 'Price', RHEA_TEXT_DOMAIN ),
					'title'      => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
					'menu_order' => esc_html__( 'Menu Order', RHEA_TEXT_DOMAIN ),
					'rand'       => esc_html__( 'Random', RHEA_TEXT_DOMAIN ),
				],
				'default'            => 'date',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'order',
			[
				'label'              => esc_html__( 'Order', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'options'            => [
					'asc'  => esc_html__( 'Ascending', RHEA_TEXT_DOMAIN ),
					'desc' => esc_html__( 'Descending', RHEA_TEXT_DOMAIN ),
				],
				'default'            => 'desc',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'show_filters',
			[
				'label'        => esc_html__( 'Frontend Filters', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => [
					'layout' => 'carousel',
				],
			]
		);

		$this->add_control(
			'grid_row',
			[
				'label'              => esc_html__( 'Carousel Grid Layout ', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'frontend_available' => true,
				'condition'          => [
					'layout' => 'carousel',
				],
			]
		);

		$this->add_control(
			'featured_properties_condition',
			[
				'label'   => esc_html__( 'Featured Properties', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'normal'             => esc_html__( 'Normal', RHEA_TEXT_DOMAIN ),
					'show_featured_only' => esc_html__( 'Show Featured Only', RHEA_TEXT_DOMAIN ),
					'hide_featured'      => esc_html__( 'Hide Featured', RHEA_TEXT_DOMAIN ),
				],
				'default' => 'normal',
			]
		);

		$this->add_control(
			'skip_sticky_properties',
			[
				'label'              => esc_html__( 'Skip Sticky Properties', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'offset',
			[
				'label'              => esc_html__( 'Offset or Skip From Start', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::NUMBER,
				'default'            => '0',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'show_views_count',
			[
				'label'              => esc_html__( 'Show Views Count', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'show_address',
			[
				'label'              => esc_html__( 'Show Address', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'show_old_price',
			[
				'label'              => esc_html__( 'Show Old Price', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'frontend_available' => true,
                'separator' => 'before'
			]
		);

		$this->add_control(
			'show_price_prefix',
			[
				'label'   => esc_html__( 'Show Price Prefix', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'yes' => [
						'title' => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
						'icon'  => 'eicon-check',
					],
					'no' => [
						'title' => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
						'icon'  => 'eicon-close',
					],
				],
				'default' => 'no',
				'selectors_dictionary' => [
					'yes' => 'display:inline;',
					'no'  => 'display:none;',
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .ere-price-prefix' => '{{VALUE}}',
				],
			]
		);

		$this->add_control(
			'show_price_postfix',
			[
				'label'   => esc_html__( 'Show Price Postfix', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'yes' => [
						'title' => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
						'icon'  => 'eicon-check',
					],
					'no' => [
						'title' => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
						'icon'  => 'eicon-close',
					],
				],
				'default' => 'no',
				'selectors_dictionary' => [
					'yes' => 'display:inline;',
					'no'  => 'display:none;',
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .ere-price-postfix' => '{{VALUE}}',
				],
			]
		);

		$this->add_control(
			'show_price_slash',
			[
				'label'   => esc_html__( 'Show Price Slash', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'yes' => [
						'title' => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
						'icon'  => 'eicon-check',
					],
					'no' => [
						'title' => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
						'icon'  => 'eicon-close',
					],
				],
				'default' => 'no',
				'selectors_dictionary' => [
					'yes' => 'display:inline;',
					'no'  => 'display:none;',
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .ere-price-slash' => '{{VALUE}}',
				],
				'separator' => 'after'
			]
		);

		$this->add_control(
			'show_agent',
			[
				'label'              => esc_html__( 'Show Agent/Owner', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'show_verification_badge',
			[
				'label'              => esc_html__( 'Show Agent Verification Badge', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'show_property_type',
			[
				'label'              => esc_html__( 'Show Property Type', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'condition'          => [
					'layout' => [ 'list' ]
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'show_property_status',
			[
				'label'              => esc_html__( 'Show Property Status', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'condition'          => [
					'layout' => [ 'list' ]
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'show_pagination',
			[
				'label'              => esc_html__( 'Show Pagination', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'condition'          => [
					'layout' => [ 'grid', 'list' ],
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'pagination_type',
			[
				'label'              => esc_html__( 'Pagination Type', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'options'            => [
					'default'         => esc_html__( 'Default', RHEA_TEXT_DOMAIN ),
					'load_more'       => esc_html__( 'Load More', RHEA_TEXT_DOMAIN ),
					'infinite_scroll' => esc_html__( 'Infinite Scroll', RHEA_TEXT_DOMAIN ),
				],
				'default'            => 'default',
				'condition'          => [
					'layout'          => [ 'grid', 'list' ],
					'show_pagination' => 'yes',
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'ajax_pagination',
			[
				'label'              => esc_html__( 'Ajax Pagination', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'condition'          => [
					'layout'          => [ 'grid', 'list' ],
					'show_pagination' => 'yes',
					'pagination_type' => 'default',
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'properties_limit',
			[
				'label'              => esc_html__( 'Maximum Properties Limit in Infinite Scroll', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::NUMBER,
				'default'            => '30',
				'frontend_available' => true,
				'condition'          => [
					'layout'          => [ 'grid', 'list' ],
					'show_pagination' => 'yes',
					'pagination_type' => 'infinite_scroll',
				],
			]
		);

		$this->add_control(
			'ere_enable_fav_properties',
			[
				'label'              => esc_html__( 'Show Add To Favourite Button', RHEA_TEXT_DOMAIN ),
				'description'        => wp_kses( __( '<strong>Important:</strong> Make sure to select <strong>Show</strong> in Customizer <strong>Dashboard > My Favorites (Add to Favorites Button on Property Detail Page)</strong> settings. ', RHEA_TEXT_DOMAIN ), $allowed_html ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'ere_enable_compare_properties',
			[
				'label'              => esc_html__( 'Show Add To Compare Button  ', RHEA_TEXT_DOMAIN ),
				'description'        => wp_kses( __( '<strong>Important:</strong> Make sure to select <strong>Enable</strong> in Customizer <strong>Floating Features (Compare Properties)</strong> settings. ', RHEA_TEXT_DOMAIN ), $allowed_html ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'ere_properties_labels',
			[
				'label' => esc_html__( 'Labels', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'property_added_date_label',
			[
				'label'              => esc_html__( 'Property Added Date', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::TEXT,
				'default'            => esc_html__( 'Added:', RHEA_TEXT_DOMAIN ),
				'frontend_available' => true,
			]
		);

		$this->action_buttons_labels();

		$this->inline_action_buttons_labels();

		$this->end_controls_section();

		$this->start_controls_section(
			'meta_settings',
			[
				'label' => esc_html__( 'Meta Settings', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'meta_layout',
			[
				'label'     => esc_html__( 'Layout', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => '1',
				'options'   => [
					'1' => esc_html__( 'Default', RHEA_TEXT_DOMAIN ),
					'2' => esc_html__( 'Simple', RHEA_TEXT_DOMAIN ),
				],
				'condition' => [
					'layout' => [ 'grid', 'list' ],
				],
			]
		);

		$get_meta = array(
			'bedrooms'   => esc_html__( 'Bedrooms', RHEA_TEXT_DOMAIN ),
			'bathrooms'  => esc_html__( 'Bathrooms', RHEA_TEXT_DOMAIN ),
			'area'       => esc_html__( 'Area', RHEA_TEXT_DOMAIN ),
			'garage'     => esc_html__( 'Garages/Parking', RHEA_TEXT_DOMAIN ),
			'year-built' => esc_html__( 'Year Built', RHEA_TEXT_DOMAIN ),
			'lot-size'   => esc_html__( 'Lot Size', RHEA_TEXT_DOMAIN ),
		);

		$meta_defaults = array(
			array(
				'rhea_property_meta_display' => 'bedrooms',
				'rhea_meta_repeater_label'   => esc_html__( 'Bedrooms', RHEA_TEXT_DOMAIN ),
			),
			array(
				'rhea_property_meta_display' => 'bathrooms',
				'rhea_meta_repeater_label'   => esc_html__( 'Bathrooms', RHEA_TEXT_DOMAIN ),
			),
			array(
				'rhea_property_meta_display' => 'area',
				'rhea_meta_repeater_label'   => esc_html__( 'Area', RHEA_TEXT_DOMAIN ),
			),
		);

		if ( $this->is_rvr_enabled ) {
			$get_meta['guests']   = esc_html__( 'Guests Capacity', RHEA_TEXT_DOMAIN );
			$get_meta['min-stay'] = esc_html__( 'Min Stay', RHEA_TEXT_DOMAIN );

			$meta_defaults[] = array(
				'rhea_property_meta_display' => 'guests',
				'rhea_meta_repeater_label'   => esc_html__( 'Guests', RHEA_TEXT_DOMAIN ),
			);
		}

		$get_meta = apply_filters( 'rhea_custom_fields_meta_icons', $get_meta );

		$meta_repeater = new \Elementor\Repeater();

		$meta_repeater->add_control(
			'rhea_property_meta_display',
			[
				'label'              => esc_html__( 'Select Meta', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'default'            => '',
				'options'            => $get_meta,
				'frontend_available' => true,
			]
		);

		$meta_repeater->add_control(
			'rhea_meta_repeater_label',
			[
				'label'   => esc_html__( 'Meta Label', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Add Label', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'rhea_add_meta_select',
			[
				'label'              => esc_html__( 'Add Meta', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::REPEATER,
				'fields'             => $meta_repeater->get_controls(),
				'default'            => $meta_defaults,
				'title_field'        => ' {{{ rhea_meta_repeater_label }}}',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'top_bar_settings',
			[
				'label'     => esc_html__( 'Top Bar', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_CONTENT,
				'condition' => [
					'layout' => [ 'grid', 'list' ],
				],
			]
		);

		$this->add_control(
			'enable_top_bar',
			[
				'label'        => esc_html__( 'Enable Top Bar', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
			]
		);

		$this->add_control(
			'enable_statistics',
			[
				'label'        => esc_html__( 'Enable Statistics', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'before',
				'condition'    => [
					'enable_top_bar' => 'yes',
				],
			]
		);

		$this->add_control(
			'detailed_statistics',
			[
				'label'        => esc_html__( 'Detailed Statistics', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => [
					'enable_top_bar'    => 'yes',
					'enable_statistics' => 'yes',
				],
			]
		);

		$this->add_control(
			'statistics_label',
			[
				'label'     => esc_html__( 'Custom Label', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'Results', RHEA_TEXT_DOMAIN ),
				'condition' => [
					'detailed_statistics' => '',
					'enable_statistics'   => 'yes',
					'enable_top_bar'      => 'yes',
				],
			]
		);

		$this->add_control(
			'enable_frontend_sorting',
			[
				'label'        => esc_html__( 'Enable Sorting', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'before',
				'condition'    => [
					'enable_top_bar' => 'yes',
				],
			]
		);

		$this->add_control(
			'enable_layout_toggle_buttons',
			[
				'label'        => esc_html__( 'Enable Layout Toggle Buttons', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'before',
				'condition'    => [
					'enable_top_bar' => 'yes',
				],
			]
		);

		$this->add_control(
			'toggle_view_options',
			[
				'label'     => esc_html__( 'Grid Layout Settings in Layout Toggle', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'condition' => [
					'layout'                       => 'list',
					'enable_layout_toggle_buttons' => 'yes',
					'enable_top_bar'               => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'grid_columns_in_toggle_view',
			[
				'label'     => esc_html__( 'Grid Columns', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 3,
				'options'   => [
					1 => 1,
					2 => 2,
					3 => 3,
					4 => 4,
					5 => 5,
					6 => 6,
				],
				'condition' => [
					'layout'                       => 'list',
					'enable_layout_toggle_buttons' => 'yes',
					'enable_top_bar'               => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'slider_settings',
			[
				'label'     => esc_html__( 'Carousel Settings', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_CONTENT,
				'condition' => [
					'layout' => 'carousel',
				],
			]
		);

		$this->add_control(
			'autoplay',
			[
				'label'              => esc_html__( 'Autoplay', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'pause_on_hover',
			[
				'label'              => esc_html__( 'Pause on Hover', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'condition'          => [
					'autoplay' => 'yes',
				],
				'render_type'        => 'none',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'pause_on_interaction',
			[
				'label'              => esc_html__( 'Pause on Interaction', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'condition'          => [
					'autoplay' => 'yes',
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'infinite',
			[
				'label'              => esc_html__( 'Infinite Loop', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'condition'          => [ 'grid_row' => '' ],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'autoplay_speed',
			[
				'label'              => esc_html__( 'Autoplay Speed', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::NUMBER,
				'default'            => 4000,
				'condition'          => [
					'autoplay' => 'yes',
				],
				'render_type'        => 'none',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'speed',
			[
				'label'              => esc_html__( 'Animation Speed', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::NUMBER,
				'default'            => 500,
				'render_type'        => 'none',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'navigation_controls',
			[
				'label' => esc_html__( 'Navigation', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_control(
			'show_arrows',
			[
				'label'        => esc_html__( 'Arrows', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'show_dots',
			[
				'label'        => esc_html__( 'Dots', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'filters_section',
			[
				'label'     => esc_html__( 'Filters', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'layout'       => 'carousel',
					'show_filters' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'filters_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .filter-btn',
			]
		);

		$this->add_control(
			'filters_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .filter-btn' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'filters_hover_color',
			[
				'label'     => esc_html__( 'Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .filter-btn:hover, 
					 {{WRAPPER}} .rh-properties-widget-v14 .filter-btn.current' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'filters_bgcolor',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .filter-btn' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'filters_hover_bg_color',
			[
				'label'     => esc_html__( 'Background Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .filter-btn:hover,
					 {{WRAPPER}} .rh-properties-widget-v14 .filter-btn.current' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'filters_padding',
			[
				'label'      => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14 .filter-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'filters_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .filters' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'filters_column_gap',
			[
				'label'     => esc_html__( 'Column Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .filters' => 'column-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'filters_row_gap',
			[
				'label'     => esc_html__( 'Row Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .filters' => 'row-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'top_bar_styles',
			[
				'label'     => esc_html__( 'Top Bar', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'layout' => [ 'grid', 'list' ],
				],
			]
		);

		$this->add_responsive_control(
			'top_bar_padding',
			[
				'label'      => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14-top-bar' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'top_bar_margin',
			[
				'label'      => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14-top-bar' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'top_bar_border_radius',
			[
				'label'      => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14-top-bar' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'top_bar_border',
				'label'    => esc_html__( 'Border', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14-top-bar',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'top_bar_box_shadow',
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14-top-bar',
			]
		);

		$this->add_control(
			'statistics_heading',
			[
				'label'     => esc_html__( 'Statistics', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'enable_statistics' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'statistics_text_typography',
				'label'     => esc_html__( 'Text Typography', RHEA_TEXT_DOMAIN ),
				'global'    => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector'  => '{{WRAPPER}} .rh-properties-widget-v14-stats',
				'condition' => [
					'enable_statistics' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'statistics_numbers_typography',
				'label'     => esc_html__( 'Numbers Typography', RHEA_TEXT_DOMAIN ),
				'global'    => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector'  => '{{WRAPPER}} .rh-properties-widget-v14-stats span',
				'condition' => [
					'enable_statistics' => 'yes',
				],
			]
		);

		$this->add_control(
			'statistics_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-stats' => 'color: {{VALUE}}',
				],
				'condition' => [
					'enable_statistics' => 'yes',
				],
			]
		);

		$this->add_control(
			'sorting_heading',
			[
				'label'     => esc_html__( 'Sorting', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'sorting_label_typography',
				'label'     => esc_html__( 'Label Typography', RHEA_TEXT_DOMAIN ),
				'global'    => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector'  => '{{WRAPPER}} .rh-properties-widget-v14-sort-controls label',
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'sorting_label_color',
			[
				'label'     => esc_html__( 'Label Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-sort-controls label' => 'color: {{VALUE}}',
				],
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'sorting_label_icon_color',
			[
				'label'     => esc_html__( 'Label Icon Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-sort-controls label path' => 'stroke: {{VALUE}}',
				],
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'dropdown_heading',
			[
				'label'     => esc_html__( 'Dropdown', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'sorting_dropdown_typography',
				'label'     => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'    => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector'  => '{{WRAPPER}} .bootstrap-select.rh-ultra-select-light .dropdown-menu li a, {{WRAPPER}} .bootstrap-select > .dropdown-toggle',
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'sorting_dropdown_color',
			[
				'label'     => esc_html__( 'Items Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select .dropdown-menu li a, 
					{{WRAPPER}} .bootstrap-select > .dropdown-toggle' => 'color: {{VALUE}}',
				],
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'sorting_dropdown_selected_item_color',
			[
				'label'     => esc_html__( 'Selected Item Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select .dropdown-menu li.selected .text,
					 {{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown .dropdown-menu li.selected span.check-mark' => 'color: {{VALUE}}',
				],
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'sorting_dropdown_items_hover_bg_color',
			[
				'label'     => esc_html__( 'Items Hover Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select .dropdown-menu li:hover' => 'background: {{VALUE}}',
				],
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'sorting_dropdown_items_hover_text_color',
			[
				'label'     => esc_html__( 'Items Hover Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select .dropdown-menu li:hover a, 
					 {{WRAPPER}} .bootstrap-select .dropdown-menu li.selected:hover .text,
					 {{WRAPPER}} .bootstrap-select .dropdown-menu li.selected:hover span.check-mark' => 'color: {{VALUE}}',
				],
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'caret_icon_color',
			[
				'label'     => esc_html__( 'Caret Icon Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-sort-controls .bootstrap-select>.dropdown-toggle .bs-caret' => 'border-top-color: {{VALUE}}; border-left-color: {{VALUE}}',
				],
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'caret_icon_hover_color',
			[
				'label'     => esc_html__( 'Caret Icon Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-sort-controls .bootstrap-select.open > .dropdown-toggle .bs-caret,
					 {{WRAPPER}} .rh-properties-widget-v14-sort-controls .bootstrap-select>.dropdown-toggle .bs-caret:hover' => 'border-top-color: {{VALUE}}; border-left-color: {{VALUE}}',
				],
				'condition' => [
					'enable_frontend_sorting' => 'yes',
				],
			]
		);

		$this->add_control(
			'layout_buttons_heading',
			[
				'label'     => esc_html__( 'Layout Buttons', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'condition' => [
					'enable_layout_toggle_buttons' => 'yes',
				],
			]
		);

		$this->add_control(
			'layout_button_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-toggle-layout a'          => 'border-color:{{VALUE}};',
					'{{WRAPPER}} .rh-properties-widget-v14-toggle-layout .list path' => 'stroke:{{VALUE}};',
					'{{WRAPPER}} .rh-properties-widget-v14-toggle-layout .grid rect' => 'fill:{{VALUE}};',
				],
				'condition' => [
					'enable_layout_toggle_buttons' => 'yes',
				],
			]
		);

		$this->add_control(
			'layout_button_color_active',
			[
				'label'     => esc_html__( 'Color (Active)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-toggle-layout .list:hover path, {{WRAPPER}} .rh-properties-widget-v14-toggle-layout .list.current path' => 'stroke:{{VALUE}};',
					'{{WRAPPER}} .rh-properties-widget-v14-toggle-layout .grid:hover rect, {{WRAPPER}} .rh-properties-widget-v14-toggle-layout .grid.current rect' => 'fill:{{VALUE}};',
				],
				'condition' => [
					'enable_layout_toggle_buttons' => 'yes',
				],
			]
		);

		$this->add_control(
			'layout_button_bg_color_active',
			[
				'label'     => esc_html__( 'Background Color (Active)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-toggle-layout a.current, 
					{{WRAPPER}} .rh-properties-widget-v14-toggle-layout a:hover' => 'background-color: {{VALUE}}; border-color:{{VALUE}};',
				],
				'condition' => [
					'enable_layout_toggle_buttons' => 'yes',
				],
			]
		);


		$this->add_responsive_control(
			'layout_button_size',
			[
				'label'     => esc_html__( 'Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 18,
						'max' => 96,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-toggle-layout a' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'enable_layout_toggle_buttons' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_card_section',
			[
				'label' => esc_html__( 'Property Card', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'property_card_wrapper_padding',
			[
				'label'     => esc_html__( 'Card Wrapper Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-card-v14-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'property_card_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .swiper-container, 
					 {{WRAPPER}} .rh-properties-widget-v14 .swiper-wrapper,
					 {{WRAPPER}} .rh-properties-widget-v14-grid-layout .rh-property-card-v14'      => 'border-radius: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rh-properties-widget-v14-grid-layout .rh-property-thumb-wrapper' => 'border-radius: {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}} 0 0;',
				],
				'condition' => [
					'layout' => [ 'carousel', 'grid' ],
				],
			]
		);

		$this->add_responsive_control(
			'property_list_card_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-list-layout .rh-property-card-v14'                                                               => 'border-radius: {{SIZE}}{{UNIT}}; overflow:hidden;',
					'[data-elementor-device-mode=mobile] {{WRAPPER}} .rh-properties-widget-v14-list-layout .rh-property-thumb-wrapper,					
					[data-elementor-device-mode=mobile] {{WRAPPER}} .rh-properties-widget-v14-list-layout .rh-property-thumb-wrapper>a' => 'border-radius: {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}} 0 0;',
					'{{WRAPPER}} .rh-properties-widget-v14-list-layout .rh-property-thumb-wrapper,
					{{WRAPPER}} .rh-properties-widget-v14-list-layout .rh-property-thumb-wrapper>a'                                     => 'border-radius: {{SIZE}}{{UNIT}} 0 0 {{SIZE}}{{UNIT}};',
					'.rtl {{WRAPPER}} .rh-properties-widget-v14-list-layout .rh-property-thumb-wrapper,
					.rtl {{WRAPPER}} .rh-properties-widget-v14-list-layout .rh-property-thumb-wrapper>a'                                => 'border-radius: 0 {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}} 0;',
				],
				'condition' => [
					'layout' => [ 'list' ],
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'property_card_box_shadow',
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .rh-property-card-v14',
			]
		);

		$this->add_responsive_control(
			'property_card_content_heading',
			[
				'label' => esc_html__( 'Content Wrapper', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_responsive_control(
			'property_card_content_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-content-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'property_card_content_bg_color',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-grid-layout .rh-property-card-v14 .rh-property-content-wrapper,
					 {{WRAPPER}} .rh-properties-widget-v14-list-layout .rh-property-card-v14' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'no_result_padding',
			[
				'label'     => esc_html__( 'No Result Text Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .no-results' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'property_agent_and_price_container_heading',
			[
				'label' => esc_html__( 'Agent and Price Container Border', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'property_agent_and_price_container_border',
				'label'    => esc_html__( 'Agent and Price Container Border', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .rh-property-more-info',
			]
		);

		$this->end_controls_section();

		$this->properties_tags_controls();

		$this->start_controls_section(
			'property_title_section',
			[
				'label' => esc_html__( 'Property Title', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .property-title',
			]
		);

		$this->add_control(
			'property_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .property-title a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_title_hover_color',
			[
				'label'     => esc_html__( 'Color Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .property-title a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'property_title_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .property-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_thumbnail_section',
			[
				'label' => esc_html__( 'Property Thumbnail', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_views_count_typography',
				'label'    => esc_html__( 'Views Count Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .rh-property-thumb-wrapper .property-views',
			]
		);

		$this->add_control(
			'property_views_count_text_color',
			[
				'label'     => esc_html__( 'Views Count Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-thumb-wrapper .property-views' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'property_views_count_icon_color',
			[
				'label'     => esc_html__( 'Views Count Icon Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-thumb-wrapper .property-views svg,
					 {{WRAPPER}} .rh-properties-widget-v14 .rh-property-thumb-wrapper .property-views svg path' => 'fill: {{VALUE}}; color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'property_views_count_icon_size',
			[
				'label'     => esc_html__( 'Views Count Icon Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-thumb-wrapper .property-views svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'property_views_count_top_position',
			[
				'label'     => esc_html__( 'Views Count Top Position', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-thumb-wrapper .property-views' => 'top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'property_views_count_left_position',
			[
				'label'     => esc_html__( 'Views Count Left Position', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-thumb-wrapper .property-views' => 'left: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();


		$this->start_controls_section(
			'property_address_section',
			[
				'label' => esc_html__( 'Property Address', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_address_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .rh-property-address',
			]
		);

		$this->add_control(
			'property_address_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-address' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_address_icon_color',
			[
				'label'     => esc_html__( 'Icon Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-address-wrapper > .pin-icon' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_address_icon_size',
			[
				'label'     => esc_html__( 'Icon Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-address-wrapper > .pin-icon' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'property_address_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-address' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_added_date_section',
			[
				'label' => esc_html__( 'Property Added Date', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_added_date_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .rh-property-added-date',
			]
		);

		$this->add_control(
			'property_added_date_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-added-date' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'property_added_date_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-property-added-date' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_meta_section',
			[
				'label' => esc_html__( 'Property Meta', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_meta_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .figure,
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .rhea-meta-labels,
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .label, 		
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .rhea-meta-icons-labels, 
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .figure,
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .rhea-meta-labels
				',
			]
		);

		$this->add_control(
			'property_meta_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .figure,
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .rhea-meta-labels,
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .label, 		
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .rhea-meta-icons-labels, 
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .figure,				
				{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta .rhea-meta-labels				
					' => 'color: {{VALUE}}
				',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_meta_postfix_typography',
				'label'    => esc_html__( 'Postfix Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .label',
			]
		);

		$this->add_control(
			'property_meta_postfix_color',
			[
				'label'     => esc_html__( 'Postfix Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_meta_box .label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_meta_svg_color',
			[
				'label'     => esc_html__( 'SVG Icon Fill', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
					{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra svg,
					{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card__meta .rhea_guests,
					{{WRAPPER}} .rh-properties-widget-v14 .rhea_ultra_prop_card__meta svg' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_meta_svg_stroke',
			[
				'label'     => esc_html__( 'SVG Icon Stroke', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
					{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra svg path,
					{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra svg rect,
					{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_guests path,
					{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_guests rect,
					{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta svg path,
					{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra svg rect' => 'stroke: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'property_meta_icon_size',
			[
				'label'     => esc_html__( 'Meta Icon Size (px)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra svg'                             => 'width: {{SIZE}}{{UNIT}} !important;',
					'{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra .rhea_ultra_prop_card__meta svg' => 'width: {{SIZE}}{{UNIT}}; height:auto;',
				],
			]
		);

		$this->add_responsive_control(
			'property_meta_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh_prop_card_meta_wrap_ultra' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_price_section',
			[
				'label' => esc_html__( 'Property Price', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_price_typography',
				'label'    => esc_html__( 'Price Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '
				{{WRAPPER}} .rh-properties-widget-v14 .ere-price-display,
				{{WRAPPER}} .rh-properties-widget-v14 .property-current-price
				',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_old_price_typography',
				'label'    => esc_html__( 'Old Price Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .property-old-price',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_price_prefix_typography',
				'label'    => esc_html__( 'Prefix Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .ere-price-prefix',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_price_postfix_typography',
				'label'    => esc_html__( 'Postfix Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '
				 {{WRAPPER}} .rh-properties-widget-v14 .ere-price-slash, 
				 {{WRAPPER}} .rh-properties-widget-v14 .ere-price-postfix
				 ',
			]
		);

		$this->add_control(
			'property_price_color',
			[
				'label'     => esc_html__( 'Price Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
				{{WRAPPER}} .rh-properties-widget-v14 .ere-price-display,
				{{WRAPPER}} .rh-properties-widget-v14 .property-current-price
					 ' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_old_price_color',
			[
				'label'     => esc_html__( 'Old Price Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .property-old-price' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_price_prefix_color',
			[
				'label'     => esc_html__( 'Prefix Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .ere-price-prefix' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_price_postfix_color',
			[
				'label'     => esc_html__( 'Postfix Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .ere-price-postfix, {{WRAPPER}} .rh-properties-widget-v14 .ere-price-slash' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_agent_section',
			[
				'label' => esc_html__( 'Property Agent', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_agent_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-property-agent .agent-title',
			]
		);

		$this->add_control(
			'property_agent_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-property-agent .agent-title' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'property_agent_image_border_radius',
			[
				'label'     => esc_html__( 'Image Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-property-agent .agent-image img' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'property_agent_vb_heading',
			[
				'label' => esc_html__( 'Verification Badge', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_control(
			'property_agent_vb_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-property-agent svg .verified-sign' => 'stroke: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_agent_vb_bg_color',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-property-agent svg .verification-badge' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_type_and_status_section',
			[
				'label' => esc_html__( 'Property Type and Status', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_type_and_status_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14-list-layout .property-type-and-status-tags a',
			]
		);

		$this->add_control(
			'property_type_and_status_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-list-layout .property-type-and-status-tags a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_type_and_status_hover_color',
			[
				'label'     => esc_html__( 'Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-list-layout .property-type-and-status-tags a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_pagination_section',
			[
				'label'     => esc_html__( 'Carousel Navigations', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'layout' => 'carousel',
				],
			]
		);

		$this->add_control(
			'pagination_bg_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled span' => '--swiper-pagination-bullet-inactive-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'pagination_active_bullet_bg_color',
			[
				'label'     => esc_html__( 'Active Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled span.swiper-pagination-bullet-active,
					 {{WRAPPER}} .swiper-pagination-styled span:hover'            => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .swiper-pagination-styled .select-active-bullet' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'pagination_container_section',
			[
				'label' => esc_html__( 'Container Options', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_responsive_control(
			'pagination_container_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-slider-navigations' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_container_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14-slider-navigations' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'pagination_items_options_section',
			[
				'label' => esc_html__( 'Items Options', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_responsive_control(
			'property_items_gap',
			[
				'label'     => esc_html__( 'Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled .swiper-pagination-inner' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'property_items_width',
			[
				'label'     => esc_html__( 'Width', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled .swiper-pagination-bullet' => 'width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .swiper-pagination-styled .select-active-bullet'     => 'width: calc({{SIZE}}{{UNIT}} + 6px);',
				],
			]
		);

		$this->add_responsive_control(
			'property_items_height',
			[
				'label'     => esc_html__( 'Height', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled .swiper-pagination-bullet' => 'height: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .swiper-pagination-styled .select-active-bullet'     => 'height: calc({{SIZE}}{{UNIT}} + 6px);',
				],
			]
		);

		$this->add_responsive_control(
			'property_items_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled .swiper-pagination-bullet, {{WRAPPER}} .swiper-pagination-styled .select-active-bullet' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'arrows_options_section',
			[
				'label' => esc_html__( 'Arrows Options', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_control(
			'arrows_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .swiper-button-next svg path, {{WRAPPER}} .swiper-button-prev svg path' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'arrows_hover_color',
			[
				'label'     => esc_html__( 'Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .swiper-button-next:hover svg path, {{WRAPPER}} .swiper-button-prev:hover svg path' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->action_button_controls_v2();

		$this->start_controls_section(
			'pagination',
			[
				'label'     => esc_html__( 'Pagination', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'layout'          => [ 'grid', 'list' ],
					'show_pagination' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_margin',
			[
				'label'      => esc_html__( 'Container Margin', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14 .pagination' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_padding',
			[
				'label'      => esc_html__( 'Container Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14 .pagination' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_items_padding',
			[
				'label'      => esc_html__( 'Items Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14 .pagination a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_items_spacing',
			[
				'label'     => esc_html__( 'Items Spacing', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .pagination' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_items_border_radius',
			[
				'label'      => esc_html__( 'Items Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14 .pagination a' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'pagination_items_text_color',
			[
				'label'     => esc_html__( 'Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .pagination a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'pagination_items_text_hover_color',
			[
				'label'     => esc_html__( 'Text Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .pagination a:hover, {{WRAPPER}} .rh-properties-widget-v14 .pagination a.current' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'pagination_items_bg_color',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .pagination a' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'pagination_items_bg_hover_color',
			[
				'label'     => esc_html__( 'Background Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .pagination a:hover, {{WRAPPER}} .rh-properties-widget-v14 .pagination a.current' => 'background: {{VALUE}}'
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'pagination_items_box_shadow',
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .pagination a',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'pagination_items_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .pagination a',
			]
		);

		$this->add_control(
			'load_more_section',
			[
				'label'     => esc_html__( 'Load More Button', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'load_more_container_margin',
			[
				'label'      => esc_html__( 'Container Margin', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14 .rh-load-more-properties' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'load_more_button_padding',
			[
				'label'      => esc_html__( 'Button Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14 .load-more-properties' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'load_more_button_border_radius',
			[
				'label'      => esc_html__( 'Button Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .rh-properties-widget-v14 .load-more-properties' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'load_more_button_text_color',
			[
				'label'     => esc_html__( 'Button Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .load-more-properties' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'load_more_button_text_hover_color',
			[
				'label'     => esc_html__( 'Button Text Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .load-more-properties:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'load_more_button_bg_color',
			[
				'label'     => esc_html__( 'Button Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .load-more-properties' => 'background: {{VALUE}}; border-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'load_more_button_bg_hover_color',
			[
				'label'     => esc_html__( 'Button Background Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .load-more-properties:hover' => 'background: {{VALUE}}; border-color: {{VALUE}};'
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'load_more_button_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .load-more-properties',
			]
		);

		$this->add_control(
			'no_more_properties_section',
			[
				'label'     => esc_html__( 'No More Properties Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'no_more_properties_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v14 .no-more-properties',
			]
		);

		$this->add_control(
			'no_more_properties_text_color',
			[
				'label'     => esc_html__( 'Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v14 .no-more-properties' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		global $settings, $widget_id, $properties_args, $properties_query, $key, $paged, $column_classes;

		$settings   = $this->get_settings_for_display();
		$widget_id  = $this->get_id();
		$key        = 'properties-layout-' . $widget_id;
		$meta_query = array();

		if ( $settings['layout'] ) {
			$layout = $settings['layout'];
		} else {
			$layout = 'carousel';
		}

		$paged = 1;
		if ( get_query_var( 'paged' ) ) {
			$paged = get_query_var( 'paged' );
		} else if ( get_query_var( 'page' ) ) { // if is static front page
			$paged = get_query_var( 'page' );
		}

		if ( $settings['offset'] ) {
			$offset = $settings['offset'] + ( $paged - 1 ) * $settings['posts_per_page'];
		} else {
			$offset = '';
		}

		$properties_args = array(
			'post_type'      => 'property',
			'post_status'    => 'publish',
			'posts_per_page' => $settings['posts_per_page'],
			'order'          => $settings['order'],
			'offset'         => $offset,
			'paged'          => $paged,
		);

		// Sticky property filtering – use meta_query instead of meta_key
		if ( $settings['skip_sticky_properties'] !== 'yes' ) {
			$meta_query[] = [
				'key'     => 'REAL_HOMES_sticky',
				'compare' => 'EXISTS',
			];
		}

		// Featured properties condition
		if ( ! empty( $settings['featured_properties_condition'] ) ) {
			$condition = $settings['featured_properties_condition'];

			if ( 'show_featured_only' === $condition ) {
				$meta_query[] = [
					'key'     => 'REAL_HOMES_featured',
					'value'   => 1,
					'compare' => '=',
					'type'    => 'NUMERIC',
				];
			} else if ( 'hide_featured' === $condition ) {
				$meta_query[] = [
					'key'     => 'REAL_HOMES_featured',
					'value'   => 1,
					'compare' => '!=',
					'type'    => 'NUMERIC',
				];
			}
		}

		$select_agent = $settings['select_agent'];
		if ( 'yes' === $settings['single_agent_agency_properties'] || ! empty( $select_agent ) ) {
			$agent_id = get_the_ID();
			if ( ! empty( $select_agent ) ) {
				$agent_id = $select_agent;
			}
			$meta_query[] = array(
				'key'     => 'REAL_HOMES_agents',
				'value'   => $agent_id,
				'compare' => '=',
			);
		}

		// Add meta_query if not empty
		if ( ! empty( $meta_query ) ) {
			$properties_args['meta_query'] = $meta_query;
		}

		// Sorting
		if ( 'price' === $settings['orderby'] ) {
			$properties_args['orderby']  = 'meta_value_num';
			$properties_args['meta_key'] = 'REAL_HOMES_property_price';
		} else {
			// for date, title, menu_order and rand
			$properties_args['orderby'] = $settings['orderby'];
		}

		if ( in_array( $layout, array( 'grid', 'list' ) ) ) {
			// Filter based on custom taxonomies
			$property_taxonomies = get_object_taxonomies( 'property', 'objects' );
			if ( ! empty( $property_taxonomies ) && ! is_wp_error( $property_taxonomies ) ) {
				foreach ( $property_taxonomies as $single_tax ) {
					$setting_key = $single_tax->name;
					if ( ! empty( $settings[ $setting_key ] ) ) {
						$properties_args['tax_query'][] = [
							'taxonomy' => $setting_key,
							'field'    => 'slug',
							'terms'    => $settings[ $setting_key ],
						];
					}
				}

				if ( isset( $properties_args['tax_query'] ) && count( $properties_args['tax_query'] ) > 1 ) {
					$properties_args['tax_query']['relation'] = 'AND';
				}
			}
		}

		if ( 'yes' === $settings['enable_search'] ) {
			$properties_args                   = apply_filters( 'real_homes_search_parameters', $properties_args );
			$properties_args['posts_per_page'] = $settings['posts_per_page'];
		}

		$properties_args = $this->sort_properties( $properties_args );
		$properties_args = apply_filters( 'rh_properties_widget_v14_args', $properties_args );

		// Stuff to add for grid and list layouts
		if ( 'carousel' !== $layout ) {
			rhea_prepare_map_data( $properties_args );

			if ( is_tax() ) {
				global $wp_query;
				$properties_args = array_merge( $wp_query->query_vars, $properties_args );
			}
		}

		$properties_query = new WP_Query( $properties_args );

		if ( $properties_query->have_posts() ) {

			if ( in_array( $layout, array( 'grid', 'list' ) ) ) {
				$column_classes = $this->get_columns( $layout );
			}

			if ( isset( $_GET[ $key ] ) && in_array( $_GET[ $key ], array( 'grid', 'list' ) ) ) {
				$layout = sanitize_text_field( $_GET[ $key ] );

				if ( 'grid' === $layout ) {
					$default_column = 3;

					if ( $settings['grid_columns_in_toggle_view'] ) {
						$default_column = $settings['grid_columns_in_toggle_view'];
					}
				} else {
					$default_column = 1;
				}

				$column_classes = $this->get_columns( $layout, $default_column );
			}

			if ( 'carousel' === $layout ) {
				$this->carousel();

			} else {
				$this->listings( $layout );
			}

		} else if ( function_exists( 'realhomes_print_no_result' ) ) {
			realhomes_print_no_result();
		}

		if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
			?>
            <script> rheaSelectPicker( '.inspiry_select_picker_trigger' );</script>
			<?php
		}
	}

	/**
	 * Outputs the filters for the property widget.
	 *
	 * This method generates the filter buttons based on property taxonomies such as
	 * status, type, city, and feature. It loops through available taxonomies, fetches their
	 * terms from the widget settings, and creates the corresponding filter buttons.
	 *
	 * @since 2.4.0
	 *
	 * @return void
	 */
	public function filters() {
		global $settings;

		$property_taxonomies = array(
			'property-status',
			'property-type',
			'property-city',
			'property-feature',
		);

		$output = '<div id="filters" class="filters">';

		$output .= sprintf( '<span class="filter-btn current" data-properties-filter="all">%s</span>', esc_html__( 'All', RHEA_TEXT_DOMAIN ) );

		if ( ! empty( $property_taxonomies ) ) {
			foreach ( $property_taxonomies as $property_taxonomy ) {
				$setting_id = $property_taxonomy;

				if ( ! empty( $settings[ $setting_id ] ) ) {
					$terms = $settings[ $setting_id ];

					foreach ( $terms as $term ) {
						$output .= sprintf( '<span class="filter-btn" data-properties-filter="%1$s" data-properties-taxonomy="%3$s"><span>%2$s</span></span>', esc_attr( $term ), esc_html( str_replace( '-', ' ', $term ) ), esc_attr( $property_taxonomy ) );
					}
				}
			}
		}

		$output .= '</div>';

		echo $output;
	}

	/**
	 * Outputs a carousel of properties for the widget.
	 *
	 * This method generates the HTML for a property carousel using a Swiper slider. It displays
	 * properties as slides and includes optional navigational arrows, pagination dots, and filters.
	 * The method loops through the queried properties and calls the card display function for each.
	 *
	 * @since 2.4.0
	 *
	 * @return void
	 */
	public function carousel() {
		global $settings, $widget_id, $properties_query;

		$is_rtl = is_rtl() ? 'dir="rtl"' : '';
		?>
        <section id="rh-properties-widget-v14-<?php echo esc_attr( $widget_id ); ?>" class="rh-properties-widget-v14" data-thumb-size="<?php echo esc_attr( $settings['thumbnail_size'] ); ?>">
			<?php
			if ( 'yes' === $settings['show_filters'] ) {
				$this->filters();
			}
			?>
            <div class="rh-properties-widget-v14-slider-wrapper">
                <div class="rh-properties-widget-v14-inner">
                    <div id="rh-properties-widget-v14-slider-<?php echo esc_attr( $widget_id ); ?>" class="rh-properties-widget-v14-slider swiper swiper-container" <?php echo esc_attr( $is_rtl ); ?>>
                        <div class="swiper-wrapper">
							<?php
							while ( $properties_query->have_posts() ) {
								$properties_query->the_post();

								$property_id      = get_the_ID();
								$args             = array();
								$args['is_rvr']   = $this->is_rvr_enabled;
								$args['slide']    = 'true';
								$args['settings'] = $settings;

								rhea_properties_widget_v14_card( $property_id, $args );
							}

							wp_reset_postdata();
							?>
                        </div>
                    </div>
					<?php
					$show_arrows = ( 'yes' === $settings['show_arrows'] );
					$show_dots   = ( 'yes' === $settings['show_dots'] );

					if ( $show_arrows || $show_dots ) {
						?>
                        <div class="rh-properties-widget-v14-slider-navigations">
							<?php
							if ( $show_arrows ) {
								?>
                                <div id="swiper-button-prev-<?php echo esc_attr( $widget_id ); ?>" class="swiper-button-prev">
                                    <svg width="24" height="16" viewBox="0 0 24 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M1.0822 7.4174C0.667263 7.83233 0.667263 8.50507 1.0822 8.92L7.8439 15.6817C8.25884 16.0966 8.93157 16.0966 9.34651 15.6817C9.76144 15.2668 9.76144 14.594 9.34651 14.1791L3.3361 8.1687L9.34651 2.15829C9.76144 1.74336 9.76144 1.07062 9.34651 0.655692C8.93157 0.24076 8.25884 0.24076 7.8439 0.655692L1.0822 7.4174ZM23.7918 7.1062H1.8335V9.2312H23.7918V7.1062Z" />
                                    </svg>
                                </div>
								<?php
							}

							if ( $show_dots ) {
								?>
                                <div id="swiper-pagination-<?php echo esc_attr( $widget_id ); ?>" class="swiper-pagination swiper-pagination-styled">
                                    <div class="swiper-pagination-inner"></div>
                                </div>
								<?php
							}

							if ( $show_arrows ) {
								?>
                                <div id="swiper-button-next-<?php echo esc_attr( $widget_id ); ?>" class="swiper-button-next">
                                    <svg width="24" height="16" viewBox="0 0 24 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M22.918 8.92C23.333 8.50507 23.333 7.83233 22.918 7.4174L16.1563 0.655692C15.7414 0.24076 15.0687 0.24076 14.6537 0.655692C14.2388 1.07062 14.2388 1.74336 14.6537 2.15829L20.6641 8.1687L14.6537 14.1791C14.2388 14.594 14.2388 15.2668 14.6537 15.6817C15.0687 16.0966 15.7414 16.0966 16.1563 15.6817L22.918 8.92ZM0.208414 9.2312L22.1667 9.2312L22.1667 7.1062L0.208414 7.1062L0.208414 9.2312Z" />
                                    </svg>
                                </div>
								<?php
							} ?>
                        </div>
						<?php
					}
					?>
                </div>
            </div>
        </section><!-- .rh-properties-widget-v14 -->
		<?php
	}

	/**
	 * Outputs the given layout based on grid or list view of properties for the widget.
	 *
	 * This method generates the HTML for displaying properties based on the given layout. It includes a top bar
	 * for view toggling, a property loop that displays each property using the card template, and pagination.
	 * Depending on the settings.
	 *
	 * @since 2.4.0
	 *
	 * @return void
	 */
	public function listings( $layout ) {
		global $settings, $widget_id, $properties_query, $column_classes, $paged, $key;
		?>
        <div id="rh-properties-widget-v14-<?php echo esc_attr( $widget_id ); ?>" data-filterable-listing-widget-id="<?php echo esc_attr( $widget_id ); ?>" class="rh-properties-widget-v14 rh-properties-widget-v14-<?php echo esc_attr( $layout ); ?>-layout rhea-listing-properties-wrapper">
			<?php
			if ( 'yes' === $settings['enable_top_bar'] ) {
				$page_id = get_the_ID();
				?>
                <div class="rh-properties-widget-v14-top-bar">
                    <div class="rh-properties-widget-v14-top-bar-col rh-properties-widget-v14-top-bar-col-left rh-properties-widget-v14-stats-wrapper">
						<?php
						if ( 'yes' === $settings['enable_statistics'] ) {
							if ( 'yes' === $settings['detailed_statistics'] ) {
								$found_properties = $properties_query->found_posts;
								$per_page         = $properties_query->query_vars['posts_per_page'];
								$state_first      = ( $per_page * $paged ) - $per_page + 1;
								$state_last       = min( $found_properties, $per_page * $paged );
								?>
                                <p class="rh-properties-widget-v14-stats" data-page="<?php echo intval( $paged ); ?>" data-max="<?php echo intval( $properties_query->max_num_pages ); ?>" data-total-properties="<?php echo intval( $found_properties ); ?>" data-page-id="<?php echo intval( $page_id ); ?>">
									<?php
									if ( $found_properties > 0 && ( $found_properties >= $per_page || -1 !== $per_page ) ) {
										?>
                                        <span><?php echo intval( $state_first ); ?></span>
										<?php esc_html_e( 'to', RHEA_TEXT_DOMAIN ); ?>
                                        <span><?php echo intval( $state_last ); ?></span>
										<?php esc_html_e( 'out of ', RHEA_TEXT_DOMAIN ); ?>
                                        <span><?php echo intval( $found_properties ); ?></span>
										<?php esc_html_e( 'properties', RHEA_TEXT_DOMAIN );
									}
									?>
                                </p>
								<?php
							} else {
								?>
                                <p class="rh-properties-widget-v14-stats properties-result-count">
                                    <span><?php echo esc_html( $properties_query->found_posts ); ?></span>
									<?php
									if ( $settings['statistics_label'] ) {
										echo esc_html( $settings['statistics_label'] );
									} else {
										esc_html_e( 'Results', RHEA_TEXT_DOMAIN );
									}
									?>
                                </p>
								<?php
							}
						}
						?>
                    </div>
                    <div class="rh-properties-widget-v14-top-bar-col rh-properties-widget-v14-top-bar-col-right">
						<?php
						if ( 'yes' === $settings['enable_frontend_sorting'] ) {
							?>
                            <div class="rh-properties-widget-v14-sort-controls">
                                <label for="sort-properties">
                                    <svg width="21" height="16" viewBox="0 0 21 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M19 1.54199H1.99997" stroke="#1B1B1B" stroke-width="2.125" stroke-linecap="round" />
                                        <path d="M14.75 7.91699H1.99997" stroke="#1B1B1B" stroke-width="2.125" stroke-linecap="round" />
                                        <path d="M14.75 14.292H1.99997" stroke="#1B1B1B" stroke-width="2.125" stroke-linecap="round" />
                                    </svg>
                                    <span><?php esc_html_e( 'Sort By:', RHEA_TEXT_DOMAIN ); ?></span>
                                </label>
								<?php
								$sort_by = '';
								if ( ! empty( $_GET['sortby'] ) ) {
									$sort_by = sanitize_text_field( $_GET['sortby'] );
								}

								rhea_generate_sort_dropdown_options( array(
									'selected' => $sort_by
								) );
								?>
                            </div>
							<?php
						}

						if ( 'yes' === $settings['enable_layout_toggle_buttons'] ) {
							?>
                            <div class="rh-properties-widget-v14-toggle-layout">
								<?php
								$layout   = '';
								$page_url = '';

								if ( is_tax() ) {
									$term_link = get_term_link( get_queried_object()->term_id );
									if ( ! is_wp_error( $term_link ) ) {
										$page_url = $term_link;
									}
								} else {
									$page_url = home_url( add_query_arg( null, null ) );
								}

								if ( isset( $_GET[ $key ] ) && in_array( $_GET[ $key ], array( 'grid', 'list' ) ) ) {
									$layout = sanitize_text_field( $_GET[ $key ] );
								} else {
									if ( $settings['layout'] && in_array( $settings['layout'], array( 'grid', 'list' ) ) ) {
										$layout = $settings['layout'];
									}
								}
								?>
                                <a class="list <?php echo ( 'list' === $layout ) ? 'current' : ''; ?>" href="<?php echo esc_url( add_query_arg( array( $key => 'list', ), $page_url ) ); ?>">
                                    <svg width="27" height="17" viewBox="0 0 27 17" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M1.91647 1.4165H6.47745" stroke="#783DF6" stroke-width="2.125" stroke-linecap="round" />
                                        <path d="M9.8981 1.4165H25.2914" stroke="#783DF6" stroke-width="2.125" stroke-linecap="round" />
                                        <path d="M1.91647 8.5H6.47745" stroke="#783DF6" stroke-width="2.125" stroke-linecap="round" />
                                        <path d="M9.8981 8.5H25.2914" stroke="#783DF6" stroke-width="2.125" stroke-linecap="round" />
                                        <path d="M1.91647 15.583H6.16647" stroke="#783DF6" stroke-width="2.125" stroke-linecap="round" />
                                        <path d="M9.70815 15.583H25.2915" stroke="#783DF6" stroke-width="2.125" stroke-linecap="round" />
                                    </svg>
                                </a>
                                <a class="grid <?php echo ( 'grid' === $layout ) ? 'current' : ''; ?>" href="<?php echo esc_url( add_query_arg( array( $key => 'grid', ), $page_url ) ); ?>">
                                    <svg width="16" height="17" viewBox="0 0 16 17" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <rect x="0.232819" y="0.651855" width="5.64706" height="5.64706" rx="0.708333" fill="white" />
                                        <rect x="0.232574" y="10.5342" width="5.64706" height="5.64706" rx="0.708333" fill="white" />
                                        <rect x="9.40921" y="0.651855" width="5.64706" height="5.64706" rx="0.708333" fill="white" />
                                        <rect x="9.40921" y="10.5342" width="5.64706" height="5.64706" rx="0.708333" fill="white" />
                                    </svg>
                                </a>
                            </div>
							<?php
						}
						?>
                    </div>
                </div>
				<?php
			}
			?>
            <div id="rhea-filterable-properties-container" class="rh-properties-container-inner <?php echo esc_attr( $column_classes ); ?>">
				<?php
				while ( $properties_query->have_posts() ) {
					$properties_query->the_post();

					$property_id      = get_the_ID();
					$args             = array();
					$args['is_rvr']   = $this->is_rvr_enabled;
					$args['layout']   = $layout;
					$args['settings'] = $settings;

					rhea_properties_widget_v14_card( $property_id, $args );
				}

				wp_reset_postdata();
				?>
            </div>
			<?php
			if ( 'yes' == $settings['show_pagination'] ) {
				$pagination_type = $settings['pagination_type'];

				if ( 'yes' === $settings['ajax_pagination'] || 'default' !== $pagination_type ) {
					?>
                    <div class="rh-properties-widget-v14-svg-loader">
						<?php include RHEA_ASSETS_DIR . '/icons/loading-bars.svg'; ?>
                    </div>
					<?php
				}

				if ( 'default' === $pagination_type ) {
					RHEA_ajax_pagination( $properties_query->max_num_pages, $properties_query, 'rh-properties-widget-v14-pagination' );
				} else if ( in_array( $pagination_type, array( 'load_more', 'infinite_scroll' ) ) ) {
					?>
                    <div id="rh-load-more-properties" class="rh-load-more-properties" data-posts="<?php echo esc_attr( $properties_query->found_posts ); ?>" data-page="<?php echo esc_attr( $properties_query->max_num_pages ); ?>">
						<?php
						if ( 'load_more' === $pagination_type ) {
							?>
                            <a href="#" class="rh-btn rh-btn-primary load-more-properties"><?php esc_html_e( 'Load More', RHEA_TEXT_DOMAIN ); ?></a>
							<?php
						}
						?>
                        <p class="no-more-properties hidden"><?php esc_html_e( 'No more properties to load!', RHEA_TEXT_DOMAIN ); ?></p>
                    </div>
					<?php
				}
			}
			?>
        </div><!-- .rh-properties-widget-v14 -->
		<?php
	}

	/**
	 * Retrieve the CSS column classes based on the specified layout.
	 *
	 * @since 2.4.0
	 *
	 * @param string $layout         The layout identifier.
	 * @param string $default_column The column identifier.
	 *
	 * @return string The CSS classes for the grid or list columns.
	 */
	public function get_columns( $layout, $default_column = '' ) {
		$settings          = $this->get_settings_for_display();
		$setting_id        = "{$layout}_columns";
		$class_prefix      = sprintf( 'rh-properties-%s', str_replace( '_', '-', esc_html( $setting_id ) ) );
		$container_classes = array( $class_prefix );

		// Define the devices list.
		$devices = array(
			"widescreen",
			"laptop",
			"tablet_extra",
			"tablet",
			"mobile_extra",
			"mobile"
		);

		if ( ! empty( $settings[ $setting_id ] ) ) {
			$default_column = $settings[ $setting_id ];
		}

		// Default column class.
		$container_classes[] = $class_prefix . '-' . esc_html( $default_column );

		// Iterate through each device and add device-specific column class.
		foreach ( $devices as $device ) {
			$id = $setting_id . '_' . $device;
			if ( ! empty( $settings[ $id ] ) ) {
				$container_classes[] = $class_prefix . '-' . esc_html( $device ) . '-' . esc_html( $settings[ $id ] );
			}
		}

		// Combine the container classes into a space-separated string and return
		return join( ' ', $container_classes );
	}

	/**
	 * Modify the properties query based on sorting criteria.
	 *
	 * This function adjusts the properties query based on the 'sortby' parameter
	 * in the URL, enabling sorting by title, price, or date in ascending or
	 * descending order.
	 *
	 * @since 2.4.0
	 *
	 * @param array $properties_args The existing arguments for the properties query.
	 *
	 * @return array The modified properties query arguments.
	 */
	public function sort_properties( $properties_args ) {
		// Check if sorting criteria is provided in the URL.
		if ( empty( $_GET['sortby'] ) ) {
			return $properties_args;
		}

		$sort_by = sanitize_text_field( $_GET['sortby'] );
		if ( $sort_by == 'title-asc' ) {
			$properties_args['orderby'] = 'title';
			$properties_args['order']   = 'ASC';
		} else if ( $sort_by == 'title-desc' ) {
			$properties_args['orderby'] = 'title';
			$properties_args['order']   = 'DESC';
		} else if ( $sort_by == 'price-asc' ) {
			$properties_args['orderby']  = 'meta_value_num';
			$properties_args['meta_key'] = 'REAL_HOMES_property_price';
			$properties_args['order']    = 'ASC';
		} else if ( $sort_by == 'price-desc' ) {
			$properties_args['orderby']  = 'meta_value_num';
			$properties_args['meta_key'] = 'REAL_HOMES_property_price';
			$properties_args['order']    = 'DESC';
		} else if ( $sort_by == 'date-asc' ) {
			$properties_args['orderby'] = 'date';
			$properties_args['order']   = 'ASC';
		} else if ( $sort_by == 'date-desc' ) {
			$properties_args['orderby'] = 'date';
			$properties_args['order']   = 'DESC';
		}

		return $properties_args;
	}
}
