<?php
/**
 * RealHomes Demo Importer Class
 *
 * This class handles all demo import functionality including:
 * - Demo configuration
 * - Plugin registration
 * - Post-import setup
 *
 * @since      2.1.2
 * @package    realhomes-demo-import
 * @subpackage realhomes-demo-import/demos
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RealHomes_Demo_Importer {

	/**
	 * Path to demos directory
	 *
	 * @var string
	 */
	private $demos_path;

	/**
	 * URL to demos directory
	 *
	 * @var string
	 */
	private $demos_url;

	/**
	 * Available demos configuration
	 *
	 * @var array
	 */
	private $demos = [];

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->demos_path = trailingslashit( plugin_dir_path( __FILE__ ) );
		$this->demos_url  = plugin_dir_url( __FILE__ );

		$this->init_hooks();
		$this->setup_demos();
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		add_filter( 'ocdi/import_files', [ $this, 'get_import_files' ] );
		add_filter( 'ocdi/register_plugins', [ $this, 'register_plugins' ] );
		add_action( 'ocdi/after_import', [ $this, 'after_import_setup' ] );
		add_action( 'import_start', [ $this, 'enable_svg_permission' ] );
	}

	/**
	 * Setup available demos
	 */
	private function setup_demos() {
		$this->demos = [
			// Ultra - Elementor Demos
			[
				'import_file_name'             => 'Fusion - Elementor',
				'local_import_file'            => $this->demos_path . 'fusion/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'fusion/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'fusion/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'fusion/demo.jpg',
				'preview_url'                  => 'https://ultra.realhomes.io/fusion',
				'categories'                   => [ 'New', 'Elementor', 'Ultra' ],
				'is_new'                       => true,
			],
			[
				'import_file_name'             => 'Ultra - Elementor',
				'local_import_file'            => $this->demos_path . 'elementor-ultra/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'elementor-ultra/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'elementor-ultra/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'elementor-ultra/demo.jpg',
				'preview_url'                  => 'https://ultra.realhomes.io/',
				'categories'                   => [ 'New', 'Elementor', 'Ultra' ],
				'is_new'                       => true,
			],
			[
				'import_file_name'             => 'Hotel - Elementor',
				'local_import_file'            => $this->demos_path . 'hotel-ultra/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'hotel-ultra/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'hotel-ultra/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'hotel-ultra/demo.jpg',
				'preview_url'                  => 'https://ultra.realhomes.io/hotel',
				'categories'                   => [ 'New', 'Elementor', 'Ultra', 'Vacation Rentals' ],
				'is_new'                       => true,
			],
			[
				'import_file_name'             => 'Agency - Elementor',
				'local_import_file'            => $this->demos_path . 'agency-ultra/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'agency-ultra/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'agency-ultra/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'agency-ultra/demo.jpg',
				'preview_url'                  => 'https://ultra.realhomes.io/agency/',
				'categories'                   => [ 'New', 'Elementor', 'Ultra' ],
				'is_new'                       => true,
			],
			[
				'import_file_name'             => 'Vacation Rentals Ultra - Elementor',
				'local_import_file'            => $this->demos_path . 'vacation-rentals-ultra/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'vacation-rentals-ultra/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'vacation-rentals-ultra/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'vacation-rentals-ultra/demo.jpg',
				'preview_url'                  => 'https://ultra.realhomes.io/vacation-rentals/',
				'categories'                   => [ 'New', 'Elementor', 'Ultra', 'Vacation Rentals' ],
				'is_new'                       => true,
			],
			[
				'import_file_name'             => 'Condominium - Elementor',
				'local_import_file'            => $this->demos_path . 'elementor-condominium/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'elementor-condominium/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'elementor-condominium/customizer.dat',
				'local_import_slider_file'     => $this->demos_path . 'elementor-condominium/realhomes-condominium-slider.zip',
				'import_preview_image_url'     => $this->demos_url . 'elementor-condominium/demo.jpg',
				'preview_url'                  => 'https://ultra.realhomes.io/condominium/',
				'categories'                   => [ 'New', 'Elementor', 'Ultra', 'Slider Revolution' ],
				'is_new'                       => true,
			],

			// Modern - Elementor Demos
			[
				'import_file_name'             => 'Modern - Elementor',
				'local_import_file'            => $this->demos_path . 'elementor-modern/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'elementor-modern/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'elementor-modern/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'elementor-modern/demo.jpg',
				'preview_url'                  => 'https://modern.realhomes.io/',
				'categories'                   => [ 'Elementor', 'Modern' ],
			],
			[
				'import_file_name'             => 'Modern 02 - Elementor',
				'local_import_file'            => $this->demos_path . 'elementor-modern-02/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'elementor-modern-02/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'elementor-modern-02/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'elementor-modern-02/demo.jpg',
				'preview_url'                  => 'https://sample.realhomes.io/modern02/',
				'categories'                   => [ 'Elementor', 'Modern' ],
			],
			[
				'import_file_name'             => 'Modern 03 - Elementor',
				'local_import_file'            => $this->demos_path . 'elementor-modern-03/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'elementor-modern-03/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'elementor-modern-03/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'elementor-modern-03/demo.jpg',
				'preview_url'                  => 'https://sample.realhomes.io/modern03/',
				'categories'                   => [ 'Elementor', 'Modern' ],
			],
			[
				'import_file_name'             => 'Modern 04 - Elementor',
				'local_import_file'            => $this->demos_path . 'elementor-modern-04/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'elementor-modern-04/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'elementor-modern-04/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'elementor-modern-04/demo.jpg',
				'preview_url'                  => 'https://sample.realhomes.io/modern04/',
				'categories'                   => [ 'Elementor', 'Modern' ],
			],
			[
				'import_file_name'             => 'Modern 05 - Elementor',
				'local_import_file'            => $this->demos_path . 'elementor-modern-05/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'elementor-modern-05/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'elementor-modern-05/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'elementor-modern-05/demo.jpg',
				'preview_url'                  => 'https://sample.realhomes.io/demo05/',
				'categories'                   => [ 'Elementor', 'Modern' ],
			],

			// Single Property Demos
			[
				'import_file_name'             => 'Single Property - Elementor',
				'local_import_file'            => $this->demos_path . 'single-property/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'single-property/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'single-property/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'single-property/demo.jpg',
				'preview_url'                  => 'https://demo.realhomes.io/single-property/',
				'categories'                   => [ 'Elementor', 'Modern' ],
			],
			[
				'import_file_name'             => 'Single Property 02 - Elementor',
				'local_import_file'            => $this->demos_path . 'single-property-02/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'single-property-02/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'single-property-02/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'single-property-02/demo.jpg',
				'preview_url'                  => 'https://sample.realhomes.io/single-property-02/',
				'categories'                   => [ 'Elementor', 'Modern' ],
			],
			[
				'import_file_name'             => 'Single Agent - Elementor',
				'local_import_file'            => $this->demos_path . 'single-agent/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'single-agent/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'single-agent/customizer.dat',
				'local_import_slider_file'     => $this->demos_path . 'single-agent/realhomes-agent-slider.zip',
				'import_preview_image_url'     => $this->demos_url . 'single-agent/demo.jpg',
				'preview_url'                  => 'https://demo.realhomes.io/single-agent/',
				'categories'                   => [ 'Elementor', 'Modern', 'Slider Revolution' ],
			],

			// Vacation Rentals Demo
			[
				'import_file_name'             => 'Vacation Rentals - Elementor',
				'local_import_file'            => $this->demos_path . 'vacation-rentals/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'vacation-rentals/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'vacation-rentals/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'vacation-rentals/demo.jpg',
				'preview_url'                  => 'https://vacation-rentals.realhomes.io/',
				'categories'                   => [ 'Elementor', 'Modern', 'Vacation Rentals' ],
			],

			// Standard Demos
			[
				'import_file_name'             => 'Modern',
				'local_import_file'            => $this->demos_path . 'modern/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'modern/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'modern/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'modern/demo.jpg',
				'preview_url'                  => 'https://modern.realhomes.io/',
				'categories'                   => [ 'Modern' ],
			],
			[
				'import_file_name'             => 'Classic',
				'local_import_file'            => $this->demos_path . 'classic/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'classic/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'classic/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'classic/demo.jpg',
				'preview_url'                  => 'https://classic.realhomes.io/',
				'categories'                   => [ 'Classic' ],
			],
			[
				'import_file_name'             => 'Classic - Elementor',
				'local_import_file'            => $this->demos_path . 'elementor-classic/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'elementor-classic/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'elementor-classic/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'elementor-classic/demo.jpg',
				'preview_url'                  => 'https://di.realhomes.io/classic-elementor/',
				'categories'                   => [ 'Elementor', 'Classic' ],
			],

			// Language Demos
			[
				'import_file_name'             => 'Hebrew (עִברִית) Ultra - Elementor',
				'local_import_file'            => $this->demos_path . 'hebrew-ultra/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'hebrew-ultra/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'hebrew-ultra/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'hebrew-ultra/demo.jpg',
				'preview_url'                  => 'https://sample.realhomes.io/arabic',
				'categories'                   => [ 'Elementor', 'Ultra', 'Language' ],
			],
			[
				'import_file_name'             => 'Español Modern - Elementor',
				'local_import_file'            => $this->demos_path . 'spanish-elementor-modern/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'spanish-elementor-modern/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'spanish-elementor-modern/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'spanish-elementor-modern/demo.jpg',
				'preview_url'                  => 'https://demo.realhomes.io/spanish/',
				'categories'                   => [ 'Elementor', 'Modern', 'Language' ],
			],
			[
				'import_file_name'             => 'Arabic (العربية) Modern - Elementor',
				'local_import_file'            => $this->demos_path . 'arabic-elementor-modern/contents.xml',
				'local_import_widget_file'     => $this->demos_path . 'arabic-elementor-modern/widgets.wie',
				'local_import_customizer_file' => $this->demos_path . 'arabic-elementor-modern/customizer.dat',
				'import_preview_image_url'     => $this->demos_url . 'arabic-elementor-modern/demo.jpg',
				'preview_url'                  => 'https://sample.realhomes.io/arabic',
				'categories'                   => [ 'Elementor', 'Modern', 'Language' ],
			],
		];
	}

	/**
	 * Get import files for OCDI
	 *
	 * @return array
	 */
	public function get_import_files() {
		return $this->demos;
	}

	/**
	 * Register required plugins
	 *
	 * @param array $plugins Existing plugins
	 *
	 * @return array
	 */
	public function register_plugins( $plugins ) {
		$theme_plugins = [];

		// Check if RealHomes is active
		if ( ! class_exists( 'RealHomes_Helper' ) ) {
			return $plugins;
		}

		// Core plugin - always required
		$theme_plugins[] = [
			'name'        => 'Easy Real Estate',
			'description' => esc_html__( 'Provides real estate functionality for RealHomes theme.', RHDI_TEXT_DOMAIN ),
			'slug'        => 'easy-real-estate',
			'source'      => 'https://inspiry-plugins.s3.us-east-2.amazonaws.com/easy-real-estate.zip',
			'version'     => RealHomes_Helper::get_plugin_version( 'easy-real-estate' ),
			'required'    => true,
			'preselected' => true,
		];

		// Get demo name for conditional plugin loading
		$demo_name = $this->get_selected_demo_name();

		if ( $demo_name ) {
			$this->add_conditional_plugins( $theme_plugins, $demo_name );
		} else {
			$this->add_ajax_plugins( $theme_plugins );
		}

		// Recommended plugin
		$theme_plugins[] = [
			'name'        => 'Quick and Easy FAQs',
			'description' => esc_html__( 'Provides FAQs functionality.', RHDI_TEXT_DOMAIN ),
			'slug'        => 'quick-and-easy-faqs',
			'required'    => false,
		];

		return array_merge( $plugins, $theme_plugins );
	}

	/**
	 * Get selected demo name
	 *
	 * @return string|false
	 */
	private function get_selected_demo_name() {
		if ( isset( $_GET['step'], $_GET['import'] ) && $_GET['step'] === 'import' ) {
			return $this->get_demo_name( $_GET['import'] );
		}

		return false;
	}

	/**
	 * Add conditional plugins based on demo
	 *
	 * @param array  $theme_plugins
	 * @param string $demo_name
	 */
	private function add_conditional_plugins( &$theme_plugins, $demo_name ) {
		// Standard demos don't need Elementor
		if ( in_array( $demo_name, [ 'Modern', 'Classic' ] ) ) {
			return;
		}

		// Add Elementor plugins
		$theme_plugins[] = [
			'name'        => esc_html__( 'Elementor Page Builder', RHDI_TEXT_DOMAIN ),
			'description' => esc_html__( 'The page builder supported by RealHomes theme.', RHDI_TEXT_DOMAIN ),
			'slug'        => 'elementor',
			'required'    => true,
			'preselected' => true,
		];

		$theme_plugins[] = [
			'name'        => 'RealHomes Elementor Addon',
			'description' => esc_html__( 'Provides RealHomes based Elementor widgets.', RHDI_TEXT_DOMAIN ),
			'slug'        => 'realhomes-elementor-addon',
			'source'      => 'https://inspiry-plugins.s3.us-east-2.amazonaws.com/realhomes-elementor-addon.zip',
			'version'     => RealHomes_Helper::get_plugin_version( 'realhomes-elementor-addon' ),
			'required'    => true,
			'preselected' => true,
		];

		// Slider Revolution for specific demos
		if ( in_array( $demo_name, [ 'Condominium - Elementor', 'Single Agent - Elementor' ] ) ) {
			$theme_plugins[] = [
				'name'        => 'Slider Revolution',
				'description' => esc_html__( 'Slider Revolution plugin.', RHDI_TEXT_DOMAIN ),
				'slug'        => 'revslider',
				'version'     => RealHomes_Helper::get_plugin_version( 'revslider' ),
				'source'      => 'https://inspiry-plugins.s3.us-east-2.amazonaws.com/revslider.zip',
				'required'    => true,
				'preselected' => true,
			];
		}

		// Vacation Rentals for specific demos
		$vacation_demos = [ 'Hotel - Elementor', 'Vacation Rentals - Elementor', 'Vacation Rentals Ultra - Elementor' ];
		if ( in_array( $demo_name, $vacation_demos ) ) {
			$theme_plugins[] = [
				'name'        => 'RealHomes Vacation Rentals',
				'description' => esc_html__( 'Provides vacation rentals functionality for RealHomes.', RHDI_TEXT_DOMAIN ),
				'slug'        => 'realhomes-vacation-rentals',
				'source'      => 'https://inspiry-plugins.s3.us-east-2.amazonaws.com/realhomes-vacation-rentals.zip',
				'version'     => RealHomes_Helper::get_plugin_version( 'realhomes-vacation-rentals' ),
				'required'    => true,
				'preselected' => true,
			];
		}
	}

	/**
	 * Handle AJAX plugin installation requests
	 *
	 * @param array $theme_plugins
	 */
	private function add_ajax_plugins( &$theme_plugins ) {
		if ( ! isset( $_POST['slug'] ) ) {
			return;
		}

		$slug           = $_POST['slug'];
		$plugin_configs = $this->get_plugin_configurations();

		if ( isset( $plugin_configs[ $slug ] ) ) {
			$theme_plugins[] = $plugin_configs[ $slug ];
		}
	}

	/**
	 * Get plugin configurations
	 *
	 * @return array
	 */
	private function get_plugin_configurations() {
		return [
			'elementor'                  => [
				'name'        => esc_html__( 'Elementor Page Builder', RHDI_TEXT_DOMAIN ),
				'description' => esc_html__( 'The page builder supported by RealHomes theme.', RHDI_TEXT_DOMAIN ),
				'slug'        => 'elementor',
				'required'    => true,
				'preselected' => true,
			],
			'realhomes-elementor-addon'  => [
				'name'        => 'RealHomes Elementor Addon',
				'description' => esc_html__( 'Provides RealHomes based Elementor widgets.', RHDI_TEXT_DOMAIN ),
				'slug'        => 'realhomes-elementor-addon',
				'source'      => 'https://inspiry-plugins.s3.us-east-2.amazonaws.com/realhomes-elementor-addon.zip',
				'version'     => RealHomes_Helper::get_plugin_version( 'realhomes-elementor-addon' ),
				'required'    => true,
				'preselected' => true,
			],
			'revslider'                  => [
				'name'        => 'Slider Revolution',
				'description' => esc_html__( 'Slider Revolution plugin.', RHDI_TEXT_DOMAIN ),
				'slug'        => 'revslider',
				'version'     => RealHomes_Helper::get_plugin_version( 'revslider' ),
				'source'      => 'https://inspiry-plugins.s3.us-east-2.amazonaws.com/revslider.zip',
				'required'    => true,
				'preselected' => true,
			],
			'realhomes-vacation-rentals' => [
				'name'        => 'RealHomes Vacation Rentals',
				'description' => esc_html__( 'Provides vacation rentals functionality for RealHomes.', RHDI_TEXT_DOMAIN ),
				'slug'        => 'realhomes-vacation-rentals',
				'source'      => 'https://inspiry-plugins.s3.us-east-2.amazonaws.com/realhomes-vacation-rentals.zip',
				'version'     => RealHomes_Helper::get_plugin_version( 'realhomes-vacation-rentals' ),
				'required'    => true,
				'preselected' => true,
			],
		];
	}

	/**
	 * After import setup
	 *
	 * @param array $selected_import
	 */
	public function after_import_setup( $selected_import ) {
		$import_demo_name = $selected_import['import_file_name'];

		// Set design variation
		$this->set_design_variation( $import_demo_name );

		// Setup menus
		$this->setup_menus();

		// Setup pages
		$this->setup_pages( $import_demo_name );

		// Import sliders if needed
		$this->import_sliders( $import_demo_name, $selected_import );

		// Finalize settings
		$this->finalize_settings( $import_demo_name );

		// Set permalink structure to "Post name"
		$this->set_permalink_structure();
	}

	/**
	 * Set design variation based on demo
	 *
	 * @param string $demo_name
	 */
	private function set_design_variation( $demo_name ) {
		$classic_demos = [ 'Classic', 'Classic - Elementor' ];
		$modern_demos  = [
			'Modern',
			'Modern - Elementor',
			'Modern 02 - Elementor',
			'Modern 03 - Elementor',
			'Modern 04 - Elementor',
			'Modern 05 - Elementor',
			'Single Agent - Elementor',
			'Single Property - Elementor',
			'Single Property 02 - Elementor',
			'Vacation Rentals - Elementor',
			'Español Modern - Elementor',
			'Arabic (العربية) Modern - Elementor'
		];

		if ( in_array( $demo_name, $classic_demos ) ) {
			$design_variation = 'classic';
		} elseif ( in_array( $demo_name, $modern_demos ) ) {
			$design_variation = 'modern';
		} else {
			$design_variation = 'ultra';
		}

		update_option( 'inspiry_design_variation', $design_variation );
	}

	/**
	 * Setup navigation menus
	 */
	private function setup_menus() {
		$menu_locations = [ 'main-menu', 'responsive-menu' ];
		$menu_title     = 'Main Menu';

		$menu = get_term_by( 'name', $menu_title, 'nav_menu' );
		if ( ! $menu ) {
			return;
		}

		$set_locations = [];
		foreach ( $menu_locations as $location ) {
			$set_locations[ $location ] = $menu->term_id;
		}

		set_theme_mod( 'nav_menu_locations', $set_locations );
	}

	/**
	 * Setup homepage and blog pages
	 *
	 * @param string $demo_name
	 */
	private function setup_pages( $demo_name ) {
		$page_config = $this->get_page_configuration( $demo_name );

		// Set show on front
		if ( $page_config['home'] || $page_config['blog'] ) {
			update_option( 'show_on_front', 'page' );
		}

		// Set homepage
		if ( $page_config['home'] ) {
			update_option( 'page_on_front', $page_config['home']->ID );
		}

		// Set blog page
		if ( $page_config['blog'] ) {
			update_option( 'page_for_posts', $page_config['blog']->ID );
			update_option( 'posts_per_page', 4 );
		}

		// Handle vacation rentals settings
		$this->handle_vacation_rentals( $demo_name );
	}

	/**
	 * Get page configuration for demo
	 *
	 * @param string $demo_name
	 *
	 * @return array
	 */
	private function get_page_configuration( $demo_name ) {
		$config = [
			'home' => null,
			'blog' => null
		];

		switch ( $demo_name ) {
			case 'Condominium - Elementor':
				$config['home'] = $this->get_page_by_title( 'Condominium' );
				break;

			case 'Hebrew (עִברִית) Ultra - Elementor':
				$config['home'] = $this->get_page_by_title( 'בַּיִת' );
				$config['blog'] = $this->get_page_by_title( 'בלוג' );
				break;

			case 'Arabic (العربية) Modern - Elementor':
				$config['home'] = $this->get_page_by_title( 'الرئيسية' );
				$config['blog'] = $this->get_page_by_title( 'اتصال' );
				break;

			case 'Español Modern - Elementor':
				$config['home'] = $this->get_page_by_title( 'Inicio' );
				$config['blog'] = $this->get_page_by_title( 'Blog' );
				break;

			case 'Hotel - Elementor':
			case 'Vacation Rentals - Elementor':
			case 'Vacation Rentals Ultra - Elementor':
			case 'Ultra - Elementor':
			case 'Single Property - Elementor':
			case 'Single Property 02 - Elementor':
			case 'Single Agent - Elementor':
			case 'Modern - Elementor':
			case 'Modern 02 - Elementor':
			case 'Modern 03 - Elementor':
			case 'Modern 04 - Elementor':
			case 'Modern 05 - Elementor':
			case 'Fusion - Elementor':
				$config['home'] = $this->get_page_by_title( 'Home' );
				$config['blog'] = $this->get_page_by_title( 'Blog' );
				break;

			default:
				$config['home'] = $this->get_page_by_title( 'Home' );
				$config['blog'] = $this->get_page_by_title( 'News' );
				break;
		}

		return $config;
	}

	/**
	 * Handle vacation rentals settings
	 *
	 * @param string $demo_name
	 */
	private function handle_vacation_rentals( $demo_name ) {
		$vacation_demos = [ 'Hotel - Elementor', 'Vacation Rentals - Elementor', 'Vacation Rentals Ultra - Elementor' ];

		if ( ! in_array( $demo_name, $vacation_demos ) ) {
			return;
		}

		if ( ! class_exists( 'Realhomes_Vacation_Rentals' ) ) {
			return;
		}

		$rvr_settings = get_option( 'rvr_settings' );
		$rvr_enabled  = isset( $rvr_settings['rvr_activation'] ) ? $rvr_settings['rvr_activation'] : false;

		if ( ! $rvr_enabled ) {
			$rvr_settings['rvr_activation'] = 1;
			update_option( 'rvr_settings', $rvr_settings );
		}
	}

	/**
	 * Import sliders for specific demos
	 *
	 * @param string $demo_name
	 * @param array  $selected_import
	 */
	private function import_sliders( $demo_name, $selected_import ) {
		$slider_demos = [ 'Single Agent - Elementor', 'Condominium - Elementor' ];

		if ( ! in_array( $demo_name, $slider_demos ) ) {
			return;
		}

		if ( ! class_exists( 'RevSliderSliderImport' ) ) {
			return;
		}

		if ( ! isset( $selected_import['local_import_slider_file'] ) ) {
			return;
		}

		$rev_slider_zip = $selected_import['local_import_slider_file'];

		if ( ! file_exists( $rev_slider_zip ) ) {
			return;
		}

		$rev_slider_importer = new RevSliderSliderImport();
		$is_template         = false;
		$single_slide        = ( $demo_name === 'Single Agent - Elementor' );
		$update_animation    = true;
		$update_navigation   = true;
		$install             = true;

		$slider_import_result = $rev_slider_importer->import_slider(
			$update_animation,
			$rev_slider_zip,
			$is_template,
			$single_slide,
			$update_navigation,
			$install
		);

		if ( ! $slider_import_result['success'] && function_exists( 'inspiry_log' ) ) {
			inspiry_log( $slider_import_result );
		}
	}

	/**
	 * Finalize import settings
	 *
	 * @param string $demo_name
	 */
	private function finalize_settings( $demo_name ) {
		// Configure custom settings
		$this->configure_custom_settings();

		// Set fonts based on demo
		$this->set_fonts( $demo_name );

		// Configure Elementor settings
		$this->configure_elementor();
	}

	/**
	 * Configure custom settings after import
	 */
	private function configure_custom_settings(): void {
		// No need of migration after latest demo import
		update_option( 'inspiry_home_settings_migration', 'true' );

		// Set scroll to top position
		update_option( 'inspiry_stp_position_from_bottom', '15px' );

		// Set defaults for the site social links
		update_option( 'theme_facebook_link', '#' );
		update_option( 'theme_twitter_link', '#' );
		update_option( 'theme_linkedin_link', '#' );
		update_option( 'theme_instagram_link', '#' );
		update_option( 'theme_youtube_link', '#' );
	}

	/**
	 * Set fonts based on demo
	 *
	 * @param string $demo_name
	 */
	private function set_fonts( $demo_name ) {
		$custom_font_demos = [ 'Hotel - Elementor', 'Single Property - Elementor', 'Condominium - Elementor' ];
		$fusion_demos      = [ 'Fusion - Elementor' ];

		if ( in_array( $demo_name, $custom_font_demos ) ) {
			// Keep custom fonts for these demos
			return;
		} elseif ( in_array( $demo_name, $fusion_demos ) ) {
			update_option( 'inspiry_heading_font', 'Lora' );
			update_option( 'inspiry_secondary_font', 'Roboto' );
			update_option( 'inspiry_body_font', 'Roboto' );
		} else {
			update_option( 'inspiry_heading_font', 'Default' );
			update_option( 'inspiry_secondary_font', 'Default' );
			update_option( 'inspiry_body_font', 'Default' );
		}
	}

	/**
	 * Configure Elementor settings
	 */
	private function configure_elementor() {
		// Disable Elementor typography and color schemes
		update_option( 'elementor_disable_typography_schemes', 'yes' );
		update_option( 'elementor_disable_color_schemes', 'yes' );
		update_option( 'elementor_google_font', '0' );

		// Set container width if not set
		$container_width = get_option( 'elementor_container_width' );
		if ( empty( $container_width ) ) {
			update_option( 'elementor_container_width', 1240 );
		}
	}

	/**
	 * Set permalink structure to "Post name"
	 */
	private function set_permalink_structure() {
		update_option( 'permalink_structure', '/%postname%/' );

		global $wp_rewrite;
		$wp_rewrite->init();
		$wp_rewrite->flush_rules( true );
	}

	/**
	 * Enable SVG file upload during import
	 */
	public function enable_svg_permission() {
		add_filter( 'upload_mimes', [ $this, 'add_svg_mime_type' ] );
	}

	/**
	 * Add SVG mime type
	 *
	 * @param array $mimes
	 *
	 * @return array
	 */
	public function add_svg_mime_type( $mimes ) {
		$mimes['svg'] = 'image/svg+xml';

		return $mimes;
	}

	/**
	 * Get page by title (replacement for deprecated get_page_by_title)
	 *
	 * @param string $page_title
	 *
	 * @return WP_Post|null
	 */
	public function get_page_by_title( $page_title ) {
		$args = [
			'post_type'              => 'page',
			'title'                  => $page_title,
			'post_status'            => 'publish',
			'posts_per_page'         => 1,
			'no_found_rows'          => true,
			'ignore_sticky_posts'    => true,
			'update_post_term_cache' => false,
			'update_post_meta_cache' => false,
			'orderby'                => 'post_date ID',
			'order'                  => 'ASC',
		];

		$query = new WP_Query( $args );

		return ! empty( $query->post ) ? $query->post : null;
	}

	/**
	 * Get demo name by index
	 *
	 * @param string $index
	 *
	 * @return string|false
	 */
	public function get_demo_name( $index ) {
		if ( isset( $this->demos[ $index ] ) ) {
			return $this->demos[ $index ]['import_file_name'];
		}

		return false;
	}

	/**
	 * Get all demos
	 *
	 * @return array
	 */
	public function get_demos() {
		return $this->demos;
	}

	/**
	 * Get demos path
	 *
	 * @return string
	 */
	public function get_demos_path() {
		return $this->demos_path;
	}

	/**
	 * Get demos URL
	 *
	 * @return string
	 */
	public function get_demos_url() {
		return $this->demos_url;
	}

}

// Initialize the class
if ( ! function_exists( 'realhomes_demo_importer_init' ) ) {
	/**
	 * Initialize RealHomes Demo Importer
	 */
	function realhomes_demo_importer_init() {
		new RealHomes_Demo_Importer();
	}
}

// Hook initialization to plugins_loaded to ensure all dependencies are loaded
add_action( 'plugins_loaded', 'realhomes_demo_importer_init' );