<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://inspirythemes.com/
 * @since      1.0.0
 *
 * @package    realhomes-currency-switcher
 * @subpackage realhomes-currency-switcher/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 */
class Realhomes_Currency_Switcher {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Realhomes_Currency_Switcher_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Stores the plugin settings.
	 *
	 * This variable holds an array of the plugin's settings.
	 *
	 * @since    1.1.0
	 * @access   protected
	 * @var      array    $settings  Plugin settings.
	 */
	protected $settings;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'REALHOMES_CURRENCY_SWITCHER_VERSION' ) ) {
			$this->version = REALHOMES_CURRENCY_SWITCHER_VERSION;
		} else {
			$this->version = '1.0.6';
		}

		if ( defined( 'REALHOMES_CURRENCY_SWITCHER_NAME' ) ) {
			$this->plugin_name = REALHOMES_CURRENCY_SWITCHER_NAME;
		} else {
			$this->plugin_name = 'realhomes-currency-switcher';
		}

		$this->settings = get_option( 'rcs_settings' );

		add_action( 'admin_notices', array( $this, 'currency_switcher_app_id_notice' ) );

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Realhomes_Currency_Switcher_Loader. Orchestrates the hooks of the plugin.
	 * - Realhomes_Currency_Switcher_i18n. Defines internationalization functionality.
	 * - Realhomes_Currency_Switcher_Admin. Defines all hooks for the admin area.
	 * - Realhomes_Currency_Switcher_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-realhomes-currency-switcher-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-realhomes-currency-switcher-i18n.php';

		/**
		 * The class responsible for defining all price formats.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/currencies-formats.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-realhomes-currency-switcher-admin.php';

		/**
		 * The class responsible for defining the plugin settings page.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-realhomes-currency-switcher-settings.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-realhomes-currency-switcher-public.php';

		$this->loader = new Realhomes_Currency_Switcher_Loader();

	}

	/**
	 * Displays a warning notice on the admin screen if the Currency Switcher App ID is missing.
	 *
	 * This function checks if the Currency Switcher is enabled and verifies whether the App ID is provided.
	 * If the App ID is missing, a dismissible warning notice is shown to the admin, prompting them to complete the Currency Switcher configuration.
	 *
	 * @since 1.1.0
	 *
	 * @return void
	 */
	public function currency_switcher_app_id_notice() {
		$rcs_settings = $this->settings;

		// Check if the Currency Switcher is enabled and if the App ID is missing
		if ( ! empty( $rcs_settings['enable_currency_switcher'] ) && empty( $rcs_settings['app_id'] ) ) {

			// Display a dismissible warning notice about the missing Currency Switcher App ID
			?>
			<div class="notice notice-warning is-dismissible">
				<p>
					<?php
					printf(
						esc_html__( '%1$sCurrency Switcher App ID%2$s is missing in %1$sRealHomes > Currency Settings%2$s. Please provide the App ID to ensure the plugin functions correctly on the site.', RHCS_TEXT_DOMAIN ),
						'<strong>',
						'</strong>'
					);
					?>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Realhomes_Currency_Switcher_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {
		$plugin_i18n = new Realhomes_Currency_Switcher_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {
		$plugin_settings = new Realhomes_Currency_Switcher_Settings();

		$this->loader->add_action( 'admin_init', $plugin_settings, 'register_settings' );
		$this->loader->add_action( 'admin_menu', $plugin_settings, 'settings_page_menu', 11 );

		$plugin_admin = new Realhomes_Currency_Switcher_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'realhomes_update_currencies', $plugin_admin, 'update_currencies_rates' );
		$this->loader->add_action( 'wp_ajax_nopriv_switch_currency', $plugin_admin, 'switch_current_currency' );
		$this->loader->add_action( 'wp_ajax_switch_currency', $plugin_admin, 'switch_current_currency' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {
		$plugin_public = new Realhomes_Currency_Switcher_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Realhomes_Currency_Switcher_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}
}
