( function ( $ ) {
    'use strict';
    /**
     * All the code for the public-facing JavaScript source
     */
    $( window ).load( function () {

        // Declare localized data variables
        let notificationStrings = rhCrmStrings.notification;
        let confirmationStrings = rhCrmStrings.confirmation;


        /**
         * Multiple items selection & deletion ajax request handler
         * @type {jQuery|HTMLElement|*}
         */
        const $action_buttons_wrap    = $( '#rh-crm-selected-items-actions-warp' );
        const $delete_button          = $( '#rh-crm-delete-selected-items' );
        const $item_selectors         = $( "input[name='rh-crm-item-select[]']" );
        const $all_items_selector     = $( '#rh-crm-item-select-all' );
        const $selected_items_counter = $( '#rh-crm-selected-items-counter' );

        // When the 'select-all' checkbox is checked or unchecked
        $all_items_selector.on( 'change', function () {
            const isChecked = $( this ).is( ':checked' );
            // Check or uncheck all child checkboxes based on the parent checkbox
            $item_selectors.prop( 'checked', isChecked ).trigger( 'change' );
        } );

        // Toggle delete button visibility and parent selector behaviour
        $item_selectors.on( "change", function () {
            const $checkedItemsSelectors = $( "input[name='rh-crm-item-select[]']:checked" );
            const anyChecked             = $checkedItemsSelectors.length > 0;
            $selected_items_counter.text( $checkedItemsSelectors.length );
            $action_buttons_wrap.toggleClass( "hidden", ! anyChecked );

            const allChecked = $item_selectors.length === $checkedItemsSelectors.length;
            // Update the parent checkbox state
            $all_items_selector.prop( 'checked', allChecked );
        } );

        // Handle delete button click
        $delete_button.on( "click", async function ( event ) {
            event.preventDefault();

            const $item_type    = $( this ).data( 'type' );
            const selectedItems = $( "input[name='rh-crm-item-select[]']:checked" )
            .map( function () {
                return $( this ).val();
            } )
            .get();

            let deleteConfirmation = await getConfirmation( confirmationStrings['deleteBulkLabel'], confirmationStrings['deleteBulkDesc'] );

            if ( ! deleteConfirmation ) {
                // showNotification( notificationStrings['cancelLabel'], notificationStrings['deleteCancelDesc'].replace( '{item_type}', $item_type ), 'info' );
                return; // Exit if user cancels
            }

            // Send AJAX request
            $.ajax( {
                url     : rhCrmData.ajaxurl, // AJAX URL
                type    : 'POST',
                data    : {
                    action   : 'realhomes_crm_delete_items', // Action name
                    security : rhCrmData.nonce, // Nonce
                    data     : {
                        item_type : $item_type, // Get item type
                        item_ids  : selectedItems // Get item IDs
                    }
                },
                success : function ( response ) {

                    if ( response.success ) {
                        // showNotification( notificationStrings['successLabel'], response.data.message || notificationStrings['deletedDesc'], 'success' );
                        location.reload();
                    } else {
                        showNotification( notificationStrings['errorLabel'], response.data.message || notificationStrings['errorDesc'], 'error' );
                    }
                },
                error   : function ( xhr, status, error ) {
                    showNotification( notificationStrings['errorLabel'], error || notificationStrings['errorDesc'], 'error' );
                }
            } );
        } );

        /**
         * Number items selection controller
         */
        $( "#rh-crm-items-per-page" ).on( "change", function () {
            const selectedValue = parseInt( $( this ).val() );

            // Save to cookie (30 days expiry)
            const expiryDate = new Date();
            expiryDate.setDate( expiryDate.getDate() + 30 );
            document.cookie = 'rh_crm_items_per_page=' + selectedValue + '; expires=' + expiryDate.toUTCString() + '; path=/; SameSite=Lax';

            // Update URL
            const urlParams = new URLSearchParams( window.location.search );
            urlParams.set( 'items', selectedValue );
            urlParams.delete( 'itemsPage' ); // Reset pagination

            window.location.search = decodeURIComponent( urlParams.toString() );
        } );

        /**
         * Item actions display controller
         */
        $( document ).on( 'click', '.rh-crm-item-actions-controller', function ( e ) {
            e.stopPropagation(); // Prevent click event from propagating to the document

            let allActionLists = $( '.realhomes-crm-item-actions' );
            let actionsList    = $( this ).siblings( '.realhomes-crm-item-actions' );

            allActionLists.not( actionsList ).addClass( 'hidden hide-it' );

            actionsList.removeClass( 'hidden' );
            actionsList.toggleClass( 'show-it' );
            actionsList.toggleClass( 'hide-it' );

            if ( actionsList.hasClass( 'hide-it' ) ) {
                setTimeout( function () {
                    actionsList.addClass( 'hidden' );
                }, 80 );
            }
        } );

        /**
         * A listing item view ajax request
         */
        $( document ).on( 'click', '.rh-crm-view-item', function ( e ) {
            e.preventDefault();

            const $button      = $( this ); // Reference to the clicked button
            const $item_type   = $button.data( 'type' );
            const $item_id     = $button.data( 'id' );
            const $item_action = $button.data( 'action' );

            // Send AJAX request
            $.ajax( {
                url     : rhCrmData.ajaxurl, // AJAX URL
                type    : 'POST',
                data    : {
                    action   : 'realhomes_crm_get_item_markup', // Action name
                    security : rhCrmData.nonce, // Nonce
                    data     : {
                        item_type   : $item_type, // Get item type
                        item_action : $item_action, // Get item action
                        item_id     : $item_id // Get item ID
                    }
                },
                success : function ( response ) {
                    setDrawer( true, response.data.item_markup );
                },
                error   : function ( xhr, status, error ) {
                    showNotification( notificationStrings['errorLabel'], error || notificationStrings['errorDesc'], 'error' );
                }
            } );
        } );

        /**
         * A listing item add/edit in drawer ajax request
         */
        $( document ).on( 'click', '.rh-crm-add-edit-item', function ( e ) {
            e.preventDefault();

            const $button      = $( this ); // Reference to the clicked button
            const $item_type   = $button.data( 'type' );
            const $item_id     = $button.data( 'id' );
            const $item_action = $button.data( 'action' );

            btnAjaxStateToggle( $button ); // Enable button ajax running state

            // Send AJAX request
            $.ajax( {
                url     : rhCrmData.ajaxurl, // AJAX URL
                type    : 'POST',
                data    : {
                    action   : 'realhomes_crm_add_edit_item_markup', // Action name
                    security : rhCrmData.nonce, // Nonce
                    data     : {
                        item_type   : $item_type, // Get item type
                        item_action : $item_action, // Get item action
                        item_id     : $item_id // Get item ID
                    }
                },
                success : function ( response ) {
                    btnAjaxStateToggle( $button ); // Disable button ajax running state
                    setDrawer( true, response.data.item_markup );

                    // Additional JS code for specific item types and actions
                    if ( 'add' === $item_action || 'edit' === $item_action ) {
                        if ( 'task' === $item_type || 'deal' === $item_type ) {
                            datePickerInit();
                        }
                    }
                },
                error   : function ( xhr, status, error ) {
                    showNotification( notificationStrings['errorLabel'], error || notificationStrings['errorDesc'], 'error' );
                }
            } );

        } );

        /**
         * Listing item add/update ajax request
         */
        $( document ).on( 'submit', '#rh-crm-item-add-update-form', function ( e ) {
            e.preventDefault();

            const $button = $( this ).find( 'button' );
            btnAjaxStateToggle( $button ); // Enable button ajax running state

            // Send AJAX request
            $.ajax( {
                url     : rhCrmData.ajaxurl, // AJAX URL
                type    : 'POST',
                data    : {
                    action   : 'realhomes_crm_add_update_item', // Action name
                    security : rhCrmData.nonce, // Nonce
                    data     : $( this ).serialize()
                },
                success : function ( response ) {
                    location.reload();
                },
                error   : function ( xhr, status, error ) {
                    btnAjaxStateToggle( $button ); // Disable button ajax running state
                    showNotification( notificationStrings['errorLabel'], error || notificationStrings['errorDesc'], 'error' );
                }
            } );
        } );

        /**
         * A listing item delete ajax request
         */
        $( document ).on( 'click', '.rh-crm-delete-item', async function ( e ) {
            e.preventDefault();

            const $button          = $( this ); // Reference to the clicked button
            const $item            = $( this ).parents( '.rh-crm-item-wrapper' ); // Reference to the clicked button
            const $item_type       = $button.data( 'type' );
            const $item_id         = $button.data( 'id' );
            const $item_type_label = $item_type.charAt( 0 ).toUpperCase() + $item_type.slice( 1 );

            let confirmed = await getConfirmation( confirmationStrings['deleteLabel'].replace( '{item_type}', $item_type_label ), confirmationStrings['deleteDesc'].replace( '{item_type}', $item_type ) );

            if ( ! confirmed ) {
                // showNotification( notificationStrings['cancelLabel'], notificationStrings['deleteCancelDesc'].replace( '{item_type}', $item_type ), 'info' );
                return; // Exit if user cancels
            }

            // Send AJAX request
            $.ajax( {
                url     : rhCrmData.ajaxurl, // AJAX URL
                type    : 'POST',
                data    : {
                    action   : 'realhomes_crm_delete_item', // Action name
                    security : rhCrmData.nonce, // Nonce
                    data     : {
                        item_type : $item_type, // Get item type
                        item_id   : $item_id // Get item ID
                    }
                },
                success : function ( response ) {

                    if ( response.success ) {
                        showNotification( notificationStrings['successLabel'], response.data.message || notificationStrings['deletedDesc'], 'success' );
                        // $item.slideUp();

                        $item.slideUp( 400, function () {
                            $( this ).remove();

                            if ( 'note' !== $item_type ) {

                                // Check remaining items on current page
                                const remainingItems = $( '.rh-crm-item-wrapper:visible' ).length;

                                // Get current page from URL
                                const urlParams   = new URLSearchParams( window.location.search );
                                const currentPage = parseInt( urlParams.get( 'itemsPage' ) ) || 1;

                                // If no items left on this page, and we're not on page 1
                                if ( remainingItems === 0 && currentPage > 1 ) {
                                    // Redirect to previous page
                                    urlParams.set( 'itemsPage', currentPage - 1 );
                                    window.location.search = urlParams.toString();
                                } else if ( remainingItems === 0 && currentPage === 1 ) {
                                    // If on page 1 and no items, just reload to show "no items" message
                                    location.reload();
                                }
                            }
                        } );

                        if ( 'note' === $item_type ) {
                            // Update total notes counter
                            let $notes_tab_link = $( '#rh-crm-notes-tab-link span' );
                            let totalNotes      = parseInt( $notes_tab_link.text() );
                            totalNotes--;
                            $notes_tab_link.text( totalNotes );
                        }
                    } else {
                        showNotification( notificationStrings['errorLabel'], response.data.message || notificationStrings['errorDesc'], 'error' );
                    }
                },
                error   : function ( xhr, status, error ) {
                    showNotification( notificationStrings['errorLabel'], error || notificationStrings['errorDesc'], 'error' );
                }
            } );
        } );

        /**
         * Item view drawer tabs
         */
        $( document ).on( 'click', '.rh-crm-tab-link', function () {
            // Remove active class from all tab links
            $( '.rh-crm-tab-link' ).removeClass( 'active' );

            // Hide all tab contents
            $( '.rh-crm-tab-content' ).hide();

            // Add active class to the clicked tab
            $( this ).addClass( 'active' );

            // Show the associated tab content
            let tabToShow = $( this ).data( 'tab' );
            $( '#' + tabToShow ).fadeIn();
        } );

        /**
         * Item's note adding ajax request.
         */
        $( document ).on( 'submit', '#rh-crm-add-item-note', async function ( e ) {
            e.preventDefault();

            let $form           = $( this );
            let $button         = $form.find( 'button' );
            let $notes_tab_link = $( '#rh-crm-notes-tab-link span' );
            let $notes_wrap     = $( '#realhomes-crm-item-notes-list' ); // Notes list container
            let $note_textarea  = $form.find( 'textarea' );
            let $no_items_found = $notes_wrap.find( '.rh-crm-no-item-found' );


            if ( '' === $note_textarea.val() ) {
                showNotification( notificationStrings['errorLabel'], notificationStrings['emptyNoteDesc'], 'error' );
                return;
            }

            btnAjaxStateToggle( $button ); // Enable button ajax running state

            // Send AJAX request
            $.ajax( {
                url     : rhCrmData.ajaxurl, // AJAX URL
                type    : 'POST',
                data    : {
                    action   : 'realhomes_crm_add_item_note', // Action name
                    security : rhCrmData.nonce, // Nonce
                    data     : $( this ).serialize()
                },
                success : function ( response ) {
                    btnAjaxStateToggle( $button ); // Disable button ajax running state
                    if ( response.success ) {
                        showNotification( notificationStrings['successLabel'], response.data.message || notificationStrings['addedDesc'], 'success' );

                        // Put the added note markup at top of the notes list
                        $notes_wrap.prepend( response.data.note_markup );
                        $( '#note-' + response.data.note_id ).slideDown();
                        $note_textarea.val( '' );

                        // Remove no items found if exists
                        if ( $no_items_found ) {
                            $no_items_found.slideUp();
                        }

                        // Update total notes counter
                        let totalNotes = parseInt( $notes_tab_link.text() );
                        totalNotes++;
                        $notes_tab_link.text( totalNotes );
                    } else {
                        showNotification( notificationStrings['errorLabel'], response.data.message || notificationStrings['errorDesc'], 'error' );
                    }
                },
                error   : function ( xhr, status, error ) {
                    showNotification( notificationStrings['errorLabel'], error || notificationStrings['errorDesc'], 'error' );
                }
            } );
        } );

        /**
         * Send matched inquiry listing to the attached lead email
         */
        $( document ).on( 'submit', '#rh-crm-send-matched-listing', function ( e ) {
            e.preventDefault();

            let $submitBtn = $( $( this ).find( 'button' ) );

            $submitBtn.addClass( 'rh-crm-btn-ajax-running' );

            // Send AJAX request
            $.ajax( {
                url     : rhCrmData.ajaxurl, // AJAX URL
                type    : 'POST',
                data    : {
                    action   : 'realhomes_crm_send_matched_listings', // Action name
                    security : rhCrmData.nonce, // Nonce
                    data     : $( this ).serialize()
                },
                success : function ( response ) {
                    if ( response.success ) {
                        showNotification( notificationStrings['successLabel'], response.data.message, 'success' );
                    } else {
                        showNotification( notificationStrings['errorLabel'], response.data.message, 'error' );
                    }
                },
                error   : function ( xhr, status, error ) {
                    showNotification( notificationStrings['errorLabel'], error || notificationStrings['errorDesc'], 'error' );
                }
            } );
        } );

        /**
         * Drawer close handler
         */
        $( document ).on( 'click', '.rh-crm-close-drawer', function ( e ) {
            e.preventDefault();
            setDrawer( false );
        } );

        /**
         * Close the drawer when clicked on the backdrop
         */
        $( document ).on( 'click', '.realhomes-crm-drawer-wrapper', function ( e ) {
            // If clicked inside .realhomes-crm-drawer, do nothing
            if ( $( e.target ).closest( '.realhomes-crm-drawer' ).length ) {
                return;
            }

            // Otherwise, close the drawer
            e.preventDefault();
            setDrawer( false );
        } );

        /**
         * Common action where items/sections will be hided when other areas are clicked.
         */
        $( document ).on( 'click', function () {
            // Hide all action lists when clicking anywhere else
            let allActionLists = $( '.realhomes-crm-item-actions' );
            allActionLists.addClass( 'hidden hide-it' );
        } );

        /**
         * Export items ajax request
         */
        $( '#rh-crm-export-items' ).on( 'click', function ( e ) {
            e.preventDefault();

            const $button    = $( this );
            const exportType = $button.data( 'type' );

            btnAjaxStateToggle( $button ); // Enable button ajax running state

            $.ajax( {
                url     : rhCrmData.ajaxurl, // AJAX URL
                type    : 'POST',
                data    : {
                    action   : 'realhomes_crm_export_items', // Action name
                    security : rhCrmData.nonce, // Nonce
                    data     : {
                        export_type : exportType // Get export type
                    }
                },
                success : function ( response ) {

                    if ( response.success !== undefined && ! response.success ) {
                        showNotification( notificationStrings['errorLabel'], response.data.message || notificationStrings['errorDesc'], 'error' );
                    } else {

                        // The response is a Blob (binary data)
                        const blob = new Blob( [response], { type : 'text/csv' } );

                        // Create a download link
                        const url = window.URL.createObjectURL( blob );
                        const a   = document.createElement( 'a' );
                        a.href    = url;

                        // Get the current date and time
                        const now           = new Date();
                        const formattedDate = now.toISOString().slice( 0, 10 ); // YYYY-MM-DD format
                        const formattedTime = now.toTimeString().slice( 0, 8 ).replace( /:/g, '-' ); // HH-MM-SS format

                        // Combine the date and time into the filename
                        const timestamp = `${formattedDate}-${formattedTime}`;
                        a.download      = `${exportType}-${timestamp}.csv`;


                        document.body.appendChild( a );
                        a.click();

                        // Cleanup
                        window.URL.revokeObjectURL( url );
                        a.remove();
                    }

                    btnAjaxStateToggle( $button ); // Disable button ajax running state
                },
                error   : function ( xhr, status, error ) {
                    showNotification( notificationStrings['errorLabel'], error || notificationStrings['errorDesc'], 'error' );
                }
            } );
        } );

        /**
         * All import process goes here
         */

        // New CSV file uploader
        $( '#rh-crm-upload-csv-form' ).on( 'submit', function ( e ) {
            e.preventDefault();

            const $button   = $( '#rh-crm-upload-csv-button' );
            const fileInput = $( '#rh-crm-csv-file' )[0];

            if ( fileInput.files.length === 0 ) {
                showNotification( notificationStrings['csvFile']['uploadFileMissingLabel'], notificationStrings['csvFile']['uploadFileMissingDesc'], 'info' );
                return;
            }

            btnAjaxStateToggle( $button ); // Enable button ajax running state

            const formData = new FormData();
            formData.append( 'csv_file', fileInput.files[0] );
            formData.append( 'action', 'realhomes_crm_upload_csv_file' );
            formData.append( 'security', rhCrmData.nonce );

            // Send AJAX request
            $.ajax( {
                url         : rhCrmData.ajaxurl,
                type        : 'POST',
                data        : formData,
                processData : false,
                contentType : false,
                success     : function ( response ) {

                    if ( response.success ) {
                        showNotification( notificationStrings['successLabel'], response.data.message, 'success' );
                        $( '#rh-crm-upload-csv-form' ).resetForm();

                        // Get the current URL as a string
                        let currentURL = window.location.href;

                        // Replace or append the 'fetch_file' parameter
                        if ( currentURL.includes( 'fetch_file=' ) ) {
                            // Update the existing parameter
                            currentURL = currentURL.replace( /fetch_file=[^&]*/, `fetch_file=${response.data.file_name}` );
                        } else {
                            // Append the parameter if it doesn't exist
                            currentURL += ( currentURL.includes( '?' ) ? '&' : '?' ) + `fetch_file=${response.data.file_name}`;
                        }

                        // Redirect to the updated URL
                        window.location.href = currentURL;

                    } else {
                        showNotification( notificationStrings['errorLabel'], response.data.message || notificationStrings['errorDesc'], 'error' );
                    }

                    btnAjaxStateToggle( $button ); // Disable button ajax running state
                },
                error       : function ( xhr, status, error ) {
                    showNotification( notificationStrings['errorDesc'], error, 'error' );
                }
            } );
        } );

        // Delete an uploaded CSV file
        $( '.rh-crm-delete-csv-file' ).on( 'click', async function () {

            const fileName   = $( this ).data( 'file' );
            let confirmation = await getConfirmation( confirmationStrings['csvFile']['deleteFileLabel'].replace( '{fileName}', fileName ), confirmationStrings['csvFile']['deleteFileDesc'] );

            if ( confirmation ) {
                $.ajax( {
                    url     : rhCrmData.ajaxurl,
                    type    : 'POST',
                    data    : {
                        action    : 'realhomes_crm_delete_csv_file',
                        security  : rhCrmData.nonce,
                        file_name : fileName
                    },
                    success : function ( response ) {

                        if ( response.success ) {
                            showNotification( notificationStrings['deleteLabel'], response.data.message, 'success' );
                            location.reload();
                        } else {
                            showNotification( notificationStrings['errorLabel'], response.data.message, 'error' );
                        }
                    },
                    error   : function ( error ) {
                        alert( 'Error while deleting the file.' );
                    }
                } );
            } else {
                showNotification( notificationStrings['cancelLabel'], notificationStrings['csvFile']['deleteCancelDesc'], 'info' );
            }
        } );

        // Fetch data of a CSV file URL builder
        $( '.fetch-button' ).on( 'click', function () {
            const fileName   = $( this ).data( 'file' );
            const currentURL = new URL( window.location.href );
            currentURL.searchParams.set( 'fetch_file', fileName );
            window.location.href = currentURL.toString();
        } );

        // Import mapped data
        $( '#rh-crm-import-csv-file' ).on( 'click', function ( e ) {
            e.preventDefault();

            const $button  = $( this );
            const fileName = $( '#rh-crm-csv-file-name' ).val(); // Get the file name from the hidden input

            if ( ! fileName ) {
                showNotification( notificationStrings['errorLabel'], notificationStrings['csvFile']['importFileMissingDesc'], 'error' );
                return;
            }

            btnAjaxStateToggle( $button ); // Enable button ajax running state
            const formData = $( '#rh-crm-fields-mapping-form' ).serialize(); // Serialize the form data

            $.ajax( {
                url     : rhCrmData.ajaxurl, // AJAX URL
                type    : 'POST',
                data    : {
                    action   : 'realhomes_crm_import_csv_file', // Action name
                    security : rhCrmData.nonce, // Nonce
                    data     : formData
                },
                success : function ( response ) {
                    if ( response.success ) {
                        $( '#rh-crm-fields-mapping-wrapper' ).slideUp();
                        $( '#rh-crm-imported-result' ).text( response.data.import_results.inserted );
                        $( '#rh-crm-duplicate-result' ).text( response.data.import_results.duplicates );
                        $( '#rh-crm-skipped-result' ).text( response.data.import_results.errors );
                        $( '#rh-crm-import-result' ).slideDown();
                    } else {
                        btnAjaxStateToggle( $button ); // Disable button ajax running state
                        console.log( response );
                    }
                },
                error   : function () {
                    alert( 'An error occurred while importing.' );
                }
            } );
        } );


        /**
         * Leads stats
         */
        if ( $( '#rh-crm-leads-stats' ).length ) {
            $.ajax( {
                url      : rhCrmData.ajaxurl,
                type     : "GET",
                data     : {
                    action   : "realhomes_crm_get_leads_stats_data",
                    security : rhCrmData.nonce
                },
                dataType : "json",
                success  : function ( response ) {

                    if ( response.success ) {
                        let statsContainer = $( "#rh-crm-leads-stats" );
                        let statHTML       = '';

                        $.each( response.data, function ( label, values ) {
                            let count = values[0];   // First value: leads count
                            let ratio = values[1];   // Second value: ratio with arrow

                            statHTML += `
                                <div class="flex flex-col items-center p-4">
                                    <h4 class="mb-2 text-gray-600" >${label}</h4>
                                    <span class="text-xl font-medium text-gray-900">${count}</span>
                                    ${ratio}
                                </div>
                            `;
                        } );

                        statsContainer.append( statHTML );
                    }
                },
                error    : function ( xhr, status, error ) {
                    console.error( "Error fetching chart data:", error );
                }
            } );
        }

        /**
         * Deals stats and chart
         */
        if ( $( '#rh-crm-deals-stats' ).length ) {
            $.ajax( {
                url      : rhCrmData.ajaxurl,
                type     : "GET",
                data     : {
                    action   : "realhomes_crm_get_deals_stats_data",
                    security : rhCrmData.nonce
                },
                dataType : "json",
                success  : function ( response ) {

                    if ( response.success ) {

                        let ctx = document.getElementById( "rh-crm-deals-chart" ).getContext( "2d" );

                        const allZero = response.data.data.every( value => value === 0 );

                        const chartData = {
                            labels   : allZero ? [""] : response.data.labels,
                            datasets : [{
                                data            : allZero ? [1] : response.data.data,
                                backgroundColor : allZero ? ["#CCCCCC"] : response.data.colors
                            }]
                        };

                        new Chart( ctx, {
                            type    : "doughnut",
                            data    : chartData,
                            options : {
                                responsive          : true,
                                maintainAspectRatio : false,
                                cutout              : "60%",
                                plugins             : {
                                    legend  : {
                                        display : false
                                    },
                                    tooltip : {
                                        enabled  : false,  // Disable default tooltip completely
                                        external : function ( context ) {
                                            // Only run custom tooltip if not allZero
                                            if ( allZero ) {
                                                return;
                                            }

                                            // Get or create tooltip element
                                            let tooltipEl = document.getElementById( 'chartjs-tooltip' );

                                            if ( ! tooltipEl ) {
                                                tooltipEl                     = document.createElement( 'div' );
                                                tooltipEl.id                  = 'chartjs-tooltip';
                                                tooltipEl.style.background    = 'rgba(0, 0, 0, 0.9)';
                                                tooltipEl.style.borderRadius  = '6px';
                                                tooltipEl.style.color         = 'white';
                                                tooltipEl.style.opacity       = 0;
                                                tooltipEl.style.pointerEvents = 'none';
                                                tooltipEl.style.position      = 'absolute';
                                                tooltipEl.style.transition    = 'all .15s ease';
                                                tooltipEl.style.padding       = '10px 14px';
                                                tooltipEl.style.fontSize      = '13px';
                                                tooltipEl.style.zIndex        = '9999';
                                                tooltipEl.style.whiteSpace    = 'nowrap';
                                                tooltipEl.style.boxShadow     = '0 4px 12px rgba(0, 0, 0, 0.15)';
                                                document.body.appendChild( tooltipEl );

                                                // Create arrow element
                                                const arrow             = document.createElement( 'div' );
                                                arrow.id                = 'chartjs-tooltip-arrow';
                                                arrow.style.position    = 'absolute';
                                                arrow.style.width       = '0';
                                                arrow.style.height      = '0';
                                                arrow.style.borderStyle = 'solid';
                                                arrow.style.borderWidth = '6px 6px 6px 0';
                                                arrow.style.borderColor = 'transparent rgba(0, 0, 0, 0.9) transparent transparent';
                                                arrow.style.left        = '-6px';
                                                arrow.style.top         = '50%';
                                                arrow.style.transform   = 'translateY(-50%)';
                                                tooltipEl.appendChild( arrow );
                                            }

                                            // Hide if no tooltip
                                            const tooltipModel = context.tooltip;
                                            if ( tooltipModel.opacity === 0 ) {
                                                tooltipEl.style.opacity = 0;
                                                return;
                                            }

                                            // Get the color of the hovered dataset
                                            let borderColor = '#3b82f6'; // Default color
                                            if ( tooltipModel.dataPoints && tooltipModel.dataPoints.length > 0 ) {
                                                const dataPoint = tooltipModel.dataPoints[0];
                                                const dataset   = context.chart.data.datasets[dataPoint.datasetIndex];
                                                if ( dataset.backgroundColor && Array.isArray( dataset.backgroundColor ) ) {
                                                    borderColor = dataset.backgroundColor[dataPoint.dataIndex];
                                                } else if ( dataset.backgroundColor ) {
                                                    borderColor = dataset.backgroundColor;
                                                }
                                            }

                                            // Set Text
                                            if ( tooltipModel.body ) {
                                                const titleLines = tooltipModel.title || [];
                                                const bodyLines  = tooltipModel.body.map( b => b.lines );

                                                let innerHtml = '<div style="border-top: 3px solid ' + borderColor + '; padding-top: 8px;">';

                                                titleLines.forEach( function ( title ) {
                                                    innerHtml += '<div style="font-weight: 600; margin-bottom: 6px; font-size: 13px;">' + title + '</div>';
                                                } );

                                                bodyLines.forEach( function ( body, i ) {
                                                    innerHtml += '<div style="font-size: 12px; opacity: 0.95;">' + body + '</div>';
                                                } );

                                                innerHtml += '</div>';
                                                tooltipEl.innerHTML = innerHtml;

                                                // Re-add arrow after innerHTML update
                                                const arrow             = document.createElement( 'div' );
                                                arrow.id                = 'chartjs-tooltip-arrow';
                                                arrow.style.position    = 'absolute';
                                                arrow.style.width       = '0';
                                                arrow.style.height      = '0';
                                                arrow.style.borderStyle = 'solid';
                                                arrow.style.borderWidth = '6px 6px 6px 0';
                                                arrow.style.borderColor = 'transparent rgba(0, 0, 0, 0.9) transparent transparent';
                                                arrow.style.left        = '-6px';
                                                arrow.style.top         = '50%';
                                                arrow.style.transform   = 'translateY(-50%)';
                                                tooltipEl.appendChild( arrow );
                                            }

                                            const position = context.chart.canvas.getBoundingClientRect();

                                            // Position tooltip to the right of cursor
                                            const tooltipX = position.left + window.scrollX + tooltipModel.caretX + 15; // 15px offset to the right
                                            const tooltipY = position.top + window.scrollY + tooltipModel.caretY;

                                            // Display, position, and set styles for font
                                            tooltipEl.style.opacity   = 1;
                                            tooltipEl.style.left      = tooltipX + 'px';
                                            tooltipEl.style.top       = tooltipY + 'px';
                                            tooltipEl.style.transform = 'translateY(-50%)'; // Center vertically relative to cursor
                                        }
                                    },
                                    title   : {
                                        display : false,
                                        text    : 'Deals by Status',
                                        padding : {
                                            top    : 10,
                                            bottom : 30
                                        }
                                    }
                                }
                            }
                        } );

                        // Append Stats Based on Labels and Data
                        let statsContainer = $( "#rh-crm-deals-stats" );
                        let statHTML       = '';
                        let total          = 0; // Get the corresponding total
                        // statsContainer.empty(); // Clear previous content

                        response.data.labels.forEach( ( label, index ) => {
                            let value = response.data.data[index]; // Get the corresponding value
                            let color = response.data.colors[index]; // Get the corresponding color

                            statHTML += `
                                <div class="flex flex-col justify-center items-center p-6">
                                    <h4 class="text-md mb-5" style="color: ${color};">${label}</h4>
                                    <span class="text-xl font-[500] text-gray-900">${value}</span>
                                </div>
                            `;

                            total += value;

                        } );

                        statHTML += `
                                <div class="flex flex-col justify-center items-center p-6">
                                    <h4 class="text-md text-gray-900 mb-5">Total</h4>
                                    <span class="text-xl font-[500] text-gray-900">${total}</span>
                                </div>
                            `;

                        statsContainer.append( statHTML );

                    }
                },
                error    : function ( xhr, status, error ) {
                    console.error( "Error fetching chart data:", error );
                }
            } );
        }

    } );

    /**
     * Control the drawer display and place drawer contents
     * @param status
     * @param content
     */
    function setDrawer( status, content = '' ) {

        if ( status ) {
            // Get the drawer template
            const template = wp.template( 'realhomes-crm-drawer' );

            // Render the drawer with dynamic data
            const drawerHtml = template( {
                content : content
            } );

            // Append drawer to the body
            const $drawer = $( drawerHtml );
            $( '#dashboard' ).append( $drawer );
        }

        let $drawer_wrap     = $( '.realhomes-crm-drawer-wrapper' );
        let $drawer_contents = $( '.realhomes-crm-drawer' );

        if ( status ) {
            openDrawer()
        } else {
            closeDrawer();
        }

        function openDrawer() {
            // Set time out use to fix CSS animation issue, as the drawer is dynamically added to the DOM.
            setTimeout( function () {
                showBackdrop();
                $drawer_contents.addClass( 'show-it' );
            }, 10 );
        }

        function closeDrawer() {
            $drawer_contents.removeClass( 'show-it' );
            hideBackdrop();

            // Set time out use before removal to allow CSS animation to complete.
            setTimeout( function () {
                $drawer_wrap.remove();
            }, 300 )
        }
    }

    /**
     * Function to confirm an ajax action
     *
     * @returns {Promise<true/false>}
     */
    function getConfirmation( message, details ) {

        // Get the notification template
        const template = wp.template( 'realhomes-crm-ajax-confirmation' );

        // Render the notification with dynamic data
        const modalHtml = template( {
            message : message,
            details : details
        } );

        // Append modal to the body
        const $modal = $( modalHtml );
        $( 'body' ).append( $modal );

        const modalWrap     = $( '.realhomes-crm-ajax-modal-wrapper' );
        const modal         = $( '.realhomes-crm-ajax-modal' );
        const cancelButton  = $( '#rh-crm-ajax-modal-cancel' );
        const confirmButton = $( '#rh-crm-ajax-modal-confirm' );


        return new Promise( ( resolve ) => {

            showModal(); // Show modal

            // Event listener for Cancel button
            cancelButton.on( 'click', () => {
                hideModal(); // Hide modal
                resolve( false ); // Return false for cancel
            } );

            // Event listener for Confirm button
            confirmButton.on( 'click', () => {
                hideModal(); // Hide modal
                resolve( true ); // Return true for confirm
            } );
        } );

        function showModal() {
            // Set time out use to fix CSS animation issue, as the modal is dynamically added to the DOM.
            setTimeout( function () {
                showBackdrop();
                modal.addClass( 'show-it' );
            }, 10 );
        }

        function hideModal() {
            modal.removeClass( 'show-it' );
            hideBackdrop();

            // Set time out use before removal to allow CSS animation to complete.
            setTimeout( function () {
                modalWrap.remove();
            }, 300 );
        }
    }

    function showBackdrop() {
        let $backdrop = $( '.realhomes-crm-ajax-modal-backdrop' );
        $backdrop.addClass( 'show-it' );

        // Prevent background scrolling
        $( 'body' ).css( 'overflow', 'hidden' );
    }

    function hideBackdrop() {
        let $backdrop = $( '.realhomes-crm-ajax-modal-backdrop' );
        $backdrop.removeClass( 'show-it' );

        // Re-enable background scrolling
        $( 'body' ).css( 'overflow', '' );
    }

    /**
     * Function to show notification
     */
    function showNotification( message, details, type ) {

        $( '.realhomes-crm-ajax-notification' ).remove(); // Remove if any existing notification is displayed.

        // Get the notification template
        const template = wp.template( 'realhomes-crm-ajax-notification' );

        // Render the notification with dynamic data
        const notificationHtml = template( {
            message : message,
            details : details,
            type    : type
        } );

        // Append notification to the body
        const $notification = $( notificationHtml );
        $( 'body' ).append( $notification );

        // Set time out use to fix CSS animation issue, as the notification is dynamically added to the DOM.
        setTimeout( function () {
            $notification.addClass( 'show-it' );
        }, 10 );

        // Close notification on button click
        $notification.find( '.js-close-notification' ).on( 'click', function () {
            $notification.removeClass( 'show-it' );
            setTimeout( function () {
                $notification.remove();
            }, 300 );
        } );

        // Auto-hide after 5 seconds
        setTimeout( function () {
            $notification.removeClass( 'show-it' );
            setTimeout( function () {
                $notification.remove();
            }, 300 );
        }, 5000 );
    }

    /**
     * Function to toggle a button state for the Ajax call running/done
     */
    function btnAjaxStateToggle( $button ) {
        $button.toggleClass( 'rh-crm-btn-ajax-running' );
        $button.find( 'svg' ).toggleClass( 'hidden' );
    }

    /**
     * Date Picker Initialization
     */
    function datePickerInit() {
        $( '.rh-crm-date-picker' ).datepicker( {
            dateFormat  : 'yy-mm-dd', // Format: YYYY-MM-DD
            changeMonth : true,
            changeYear  : true,
            showAnim    : 'fadeIn',
            minDate     : 0
        } );
    }


} )( jQuery );