<?php
/**
 * This file contains the functions for the Realhomes CRM plugin.
 *
 * @since 1.0.0
 */


/**
 * Sets up the query arguments for property listings based on given property info.
 *
 * @param array $query_info Property information to build the query arguments.
 *
 * @return array $query_args Array of WP_Query arguments.
 */
add_filter( 'realhomes_crm_match_listing_args', function ( array $query_info ): array {
	// Initialize taxonomy query array with AND relation.
	$tax_query = array( 'relation' => 'AND' );

	// Add taxonomy query conditions if available in $query_info.
	if ( ! empty( $query_info['property-type'] ) ) {
		$tax_query[] = array(
			'taxonomy' => 'property-type',
			'field'    => 'slug',
			'terms'    => $query_info['property-type'],
		);
	}

	if ( ! empty( $query_info['property-city'] ) ) {
		$tax_query[] = array(
			'taxonomy' => 'property-city',
			'field'    => 'slug',
			'terms'    => $query_info['property-city'],
		);
	}

	if ( ! empty( $query_info['property-status'] ) ) {
		$tax_query[] = array(
			'taxonomy' => 'property-status',
			'field'    => 'slug',
			'terms'    => $query_info['property-status'],
		);
	}

	// Initialize meta query array with AND relation.
	$meta_query = array( 'relation' => 'AND' );

	// Add meta queries for beds, baths, garage, and area size.
	$meta_keys = array(
		'min-beds'   => array( 'key' => 'REAL_HOMES_property_bedrooms', 'compare' => '>=' ),
		'max-beds'   => array( 'key' => 'REAL_HOMES_property_bedrooms', 'compare' => '<=' ),
		'min-baths'  => array( 'key' => 'REAL_HOMES_property_bathrooms', 'compare' => '>=' ),
		'max-baths'  => array( 'key' => 'REAL_HOMES_property_bathrooms', 'compare' => '<=' ),
		'min-garage' => array( 'key' => 'REAL_HOMES_property_garage', 'compare' => '>=' ),
		'max-garage' => array( 'key' => 'REAL_HOMES_property_garage', 'compare' => '<=' ),
		'min-area'   => array( 'key' => 'REAL_HOMES_property_size', 'compare' => '>=' ),
		'max-area'   => array( 'key' => 'REAL_HOMES_property_size', 'compare' => '<=' ),
	);

	foreach ( $meta_keys as $key => $meta_info ) {
		if ( ! empty( $query_info[ $key ] ) ) {
			$meta_query[] = array(
				'key'     => $meta_info['key'],
				'value'   => intval( $query_info[ $key ] ),
				'type'    => 'NUMERIC',
				'compare' => $meta_info['compare']
			);
		}
	}

	// Construct the final query arguments.
	return array(
		'post_type'      => 'property',
		'posts_per_page' => 50,
		'meta_query'     => $meta_query,
		'tax_query'      => $tax_query,
	);
} );

/**
 * Put required JS templates in the footer to support plugin dynamic templates & actions.
 */
add_action( 'wp_footer', function () {
	if ( empty( $_GET['module'] ) || 'properties-crm' !== $_GET['module'] ) {
		return;
	}
	?>
	<!-- Notification Template -->
	<script type="text/html" id="tmpl-realhomes-crm-ajax-notification">
		<div class="realhomes-crm-ajax-notification z-20">
			<div class="p-4">
				<div class="flex items-start">
					<div class="shrink-0 rh-crm-icon-wrapper">
						<# if ( 'success' === data.type ) { #>
						<!-- Success Icon (check mark) -->
						<?php Realhomes_Crm_SVG::the_svg( 'success', 'size-8 text-green-400 mt-0.5' ); ?>
						<# } else if ( 'error' === data.type ) { #>
						<!-- Error Icon (cross) -->
						<?php Realhomes_Crm_SVG::the_svg( 'error', 'size-8 text-red-400 mt-0.5' ); ?>
						<# } else if ( 'info' === data.type ) { #>
						<!-- Info Icon -->
						<?php Realhomes_Crm_SVG::the_svg( 'info', 'size-8 text-blue-400 mt-0.5' ); ?>
						<# } else if ( 'warning' === data.type ) { #>
						<!-- Warning Icon -->
						<?php Realhomes_Crm_SVG::the_svg( 'warning', 'size-8 text-yellow-400 mt-0.5' ); ?>
						<# } #>
					</div>
					<div class="ml-3 w-0 flex-1 pt-0.5">
						<p class="text-md font-medium text-gray-900">{{{ data.message }}}</p>
						<p class="mt-1 text-md text-gray-500">{{{ data.details }}}</p>
					</div>
					<div class="ml-4 flex shrink-0">
						<button type="button" class="inline-flex rounded-md bg-white text-gray-400 hover:text-gray-500 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 js-close-notification">
							<?php Realhomes_Crm_SVG::the_svg( 'close', 'size-8' ); ?>
						</button>
					</div>
				</div>
			</div>
		</div>
	</script>

	<!-- Drawer Template -->
	<script type="text/html" id="tmpl-realhomes-crm-drawer">
		<div class="realhomes-crm-drawer-wrapper relative z-10" aria-labelledby="slide-over-title" role="dialog" aria-modal="true">

			<!-- Background backdrop, show/hide based on slide-over state. -->
			<div class="realhomes-crm-ajax-modal-backdrop"></div>

			<div class="fixed inset-0 overflow-hidden">
				<div class="pointer-events-none fixed inset-y-0 right-6 flex max-w-full">
					<!-- CRM Drawer -->
					<div class="realhomes-crm-drawer pointer-events-auto w-screen max-w-6xl mt-36 mb-12 bg-white rounded-md">
						<!-- CRM Item Contents Here-->
						{{{data.content}}}
					</div>
				</div>
			</div>
		</div>
	</script>

	<!-- Confirmation Modal Template -->
	<script type="text/html" id="tmpl-realhomes-crm-ajax-confirmation">
		<div class="realhomes-crm-ajax-modal-wrapper relative z-10" aria-labelledby="modal-title" role="dialog" aria-modal="true">

			<!-- Background backdrop, show/hide based on slide-over state. -->
			<div class="realhomes-crm-ajax-modal-backdrop"></div>

			<div class="fixed inset-0 z-10 w-screen h-screen">
				<div class="flex min-h-full items-end justify-center p-4 text-center sm:items-start sm:mt-96 sm:p-0">

					<div class="realhomes-crm-ajax-modal relative transform overflow-hidden rounded-lg bg-white px-4 pb-4 pt-5 text-left shadow-xl transition-all sm:my-8 sm:w-full sm:max-w-2xl sm:p-8">
						<div>
							<div class="mx-auto flex size-12 items-center justify-center rounded-full bg-yellow-100">
								<?php Realhomes_Crm_SVG::the_svg( 'warning', 'size-8 text-yellow-400' ); ?>
							</div>
							<div class="mt-3 text-center sm:mt-5">
								<h3 class="text-base text-gray-900" id="modal-title">{{{ data.message }}}</h3>
								<div class="mt-2">
									<p class="text-sm text-gray-500">{{{ data.details }}}</p>
								</div>
							</div>
						</div>
						<div class="mt-5 sm:mt-8 sm:grid sm:grid-flow-row-dense sm:grid-cols-2 sm:gap-3">
							<button id="rh-crm-ajax-modal-confirm" class="inline-flex w-full justify-center rounded-md bg-[var(--dashboard-primary-color)] px-3 py-3 text-sm text-white shadow-sm active:bg-slate-700 hover:bg-slate-700 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600 sm:col-start-2"><?php esc_html_e( 'Confirm',
									REALHOMES_CRM_TEXT_DOMAIN ); ?></button>
							<button id="rh-crm-ajax-modal-cancel" class="inline-flex w-full justify-center rounded-md bg-white px-3 py-3 text-sm text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 hover:bg-gray-50 sm:col-start-1 sm:mt-0"><?php esc_html_e( 'Cancel',
									REALHOMES_CRM_TEXT_DOMAIN ); ?></button>
						</div>
					</div>
				</div>
			</div>
		</div>
	</script>
	<?php
} );

