<?php
/**
 * The file that defines all the actions for the tasks management.
 *
 * @link       https://realhomes.io/
 * @since      1.0.0
 *
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 */

/**
 * The tasks management class.
 *
 * This is used to define all the actions and data management of tasks.
 *
 * @since      1.0.0
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 * @author     InspiryThemes <fahid@inspirythemes.com>
 */
class Realhomes_Crm_Tasks {

	/**
	 * Inserts a new task into the tasks table in the database.
	 *
	 * @return int|false The number of rows affected or false on error. On success, returns the ID of the inserted row.
	 */
	public static function insert_custom_task( $data ) {
		global $wpdb;

		// Prepare data for insertion
		$data = array(
			'title'       => sanitize_text_field( $data['title'] ),
			'description' => sanitize_textarea_field( $data['description'] ),
			'priority'    => sanitize_text_field( $data['priority'] ),
			'type'        => sanitize_text_field( $data['type'] ),
			'status'      => sanitize_text_field( $data['status'] ),
			'due_date'    => sanitize_text_field( $data['due-date'] ),
			'created_at'  => current_time( 'mysql' ),
			'updated_at'  => current_time( 'mysql' ),
		);

		// Define the data format
		$format = array( '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s' );

		// Insert the data into the tasks table
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_tasks; // Replace with your actual table name
		$inserted   = $wpdb->insert( $table_name, $data, $format );

		// Check for successful insertion
		if ( $inserted === false ) {
			// Handle the error if insert failed
			error_log( 'Database insert task failed: ' . $wpdb->last_error );
		}

		return $inserted;
	}

	/**
	 * Updates an existing task in the database with new values.
	 *
	 * @return int|false Number of rows affected, or false on failure.
	 */
	public static function update_task( $data ) {
		// Sanitize and validate inputs
		$task_id     = intval( $data['item_id'] );
		$title       = sanitize_text_field( $data['title'] );
		$description = sanitize_textarea_field( $data['description'] );
		$status      = sanitize_text_field( $data['status'] );
		$type        = sanitize_text_field( $data['type'] );
		$due_date    = sanitize_text_field( $data['due-date'] );
		$priority    = sanitize_text_field( $data['priority'] );
		$updated_at  = current_time( 'mysql' );

		global $wpdb;

		// Prepare data for update
		$data = [
			'title'       => $title,
			'description' => $description,
			'status'      => $status,
			'type'        => $type,
			'due_date'    => $due_date,
			'priority'    => $priority,
			'updated_at'  => $updated_at,
		];

		// Define where clause
		$where = [ 'id' => $task_id ];

		// Update task in the database
		$result = $wpdb->update(
			$wpdb->prefix . Realhomes_Crm_Helper::$table_tasks, // Assuming tasks are stored in 'crm_tasks' table
			$data,
			$where,
			array( '%s', '%s', '%s', '%s', '%s', '%s', '%s' ),
			array( '%d' )
		);

		// Check if the update was successful
		if ( $result === false ) {
			// Handle the error if update failed
			error_log( 'Database "task" update failed: ' . $wpdb->last_error );
		}

		return $result;
	}

	/**
	 * Delete a single task based on the given task ID
	 *
	 * @param $task_id
	 *
	 * @return void
	 */
	public static function delete_task( $task_id ) {
		global $wpdb;

		// Make sure the task ID is valid and not empty
		if ( ! empty( $task_id ) && intval( $task_id ) > 0 ) {
			// Table name
			$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_tasks;

			// Prepare and execute delete query
			$result = $wpdb->delete( $table_name, array( 'id' => intval( $task_id ) ) );

			if ( $result === false ) {
				// Handle the error if delete failed
				error_log( 'Task delete failed: ' . $wpdb->last_error );
			}

			return $result;
		}
	}

	/**
	 * Function to fetch a single task from the database based on the given task ID.
	 *
	 * @param int $task_id The ID of the task to retrieve.
	 *
	 * @return array|false|object|stdClass The task data, or false if not found.
	 */
	public static function fetch_task( $task_id ) {
		global $wpdb;

		// Table name for tasks
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_tasks;

		// Validate the task ID
		$task_id = absint( $task_id );

		if ( ! $task_id ) {
			return false; // Return false if the task ID is invalid
		}

		// Prepare the SQL query to fetch the task by ID
		$sql = $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d", $task_id );

		// Execute the query and fetch the single task
		$task = $wpdb->get_row( $sql );

		// Return the task data, or false if no task is found
		return $task ?? false;
	}

	/**
	 * Function to fetch all tasks from the database.
	 *
	 * @param array $args   Query arguments to filter tasks.
	 * @param array $fields Fields to select in the query.
	 *
	 * @return array|object|stdClass[]|null The tasks data, or null if no tasks are found.
	 */
	public static function fetch_tasks( $args = array(), $fields = array() ) {
		global $wpdb;

		// Table name for tasks
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_tasks;

		// Default query arguments
		$defaults = array(
			'number'  => 10,            // Number of tasks to retrieve (default: 50)
			'offset'  => 0,             // Offset for pagination (default: 0)
			'orderby' => 'created_at',    // Order by column (default: due_date)
			'order'   => 'DESC',        // Order direction (default: DESC)
		);

		// Merge the default arguments with the provided ones
		$args = wp_parse_args( $args, $defaults );

		// Determine fields to select: if specific fields are provided, use those; otherwise, select all (*)
		$select_fields = '*';
		if ( ! empty( $fields ) && is_array( $fields ) ) {
			// Sanitize each field to prevent SQL injection
			$select_fields = implode( ', ', array_map( 'esc_sql', $fields ) );
		}

		// Start building the SQL query with selected fields
		$sql = "SELECT {$select_fields} FROM $table_name WHERE 1=1";

		// Add filtering by status, if provided
		if ( ! empty( $args['status'] ) ) {
			$sql .= $wpdb->prepare( " AND status = %s", $args['status'] );
		}

		// Add filtering by type, if provided
		if ( ! empty( $args['type'] ) ) {
			$sql .= $wpdb->prepare( " AND type = %s", $args['type'] );
		}

		// Add a search query if provided (search by task title or description)
		if ( ! empty( $args['keyword'] ) ) {
			$keyword = '%' . $wpdb->esc_like( $args['keyword'] ) . '%';
			$sql     .= $wpdb->prepare( " AND (title LIKE %s OR description LIKE %s OR status LIKE %s OR type LIKE %s OR priority LIKE %s)", $keyword, $keyword, $keyword, $keyword, $keyword );
		}

		// Add ordering and pagination
		$sql .= $wpdb->prepare( " ORDER BY {$args['orderby']} {$args['order']} LIMIT %d OFFSET %d", $args['number'], $args['offset'] );

		// Execute the query and return the results
		return $wpdb->get_results( $sql );
	}

	/**
	 * Handle exporting tasks as a CSV file
	 *
	 * @return void
	 */
	public static function export_tasks() {
		// Retrieve all tasks data
		$tasks = self::fetch_tasks();

		// Set CSV headers
		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=tasks' . '.csv' );

		// Open PHP output stream as the file
		$output = fopen( 'php://output', 'w' );

		// Define CSV column headers
		fputcsv( $output, array(
			'Title',
			'Description',
			'Priority',
			'Status',
			'Type',
			'Due Date',
		) );

		// Add each lead's data as a new row in the CSV
		foreach ( $tasks as $task ) {
			fputcsv( $output, array(
				$task->title,
				$task->description,
				$task->priority,
				$task->status,
				$task->type,
				$task->due_date,
			) );
		}

		// Close the output stream
		fclose( $output );

		exit;
	}

	/**
	 * Import tasks from the uploaded CSV file.
	 *
	 * @param $file_handle
	 * @param $headers
	 * @param $fields_mapping
	 *
	 * @return int[]
	 */
	public static function import_tasks( $file_handle, $headers, $fields_mapping ) {
		// Array to hold import results
		$import_results = [
			'inserted'   => 0,
			'duplicates' => 0,
			'errors'     => 0
		];


		// Loop through each row
		while ( ( $csv_data = fgetcsv( $file_handle ) ) !== false ) {
			$task_data = [];
			foreach ( $fields_mapping as $field_mapping => $csv_field ) {
				$index                       = array_search( $csv_field, $headers );
				$task_data[ $field_mapping ] = $index !== false ? $csv_data[ $index ] : '';
			}
			// Check for duplicates based on title
			global $wpdb;
			$existing_task = $wpdb->get_var( $wpdb->prepare(
				"SELECT COUNT(*) FROM " . $wpdb->prefix . Realhomes_Crm_Helper::$table_tasks . " WHERE title = %s",
				$task_data['title']
			) );

			if ( $existing_task ) {
				$import_results['duplicates'] ++;
				continue; // Skip duplicate entries
			}

			// Insert task into the database
			$result = $wpdb->insert(
				$wpdb->prefix . Realhomes_Crm_Helper::$table_tasks, // Deals table name
				[
					'title'       => sanitize_text_field( $task_data['title'] ),
					'description' => sanitize_textarea_field( $task_data['description'] ),
					'priority'    => sanitize_text_field( $task_data['priority'] ),
					'status'      => sanitize_text_field( $task_data['status'] ),
					'type'        => sanitize_text_field( $task_data['type'] ),
					'due_date'    => sanitize_text_field( $task_data['due_date'] ),
				],
				[ '%s', '%s', '%s', '%s', '%s', '%s' ]
			);

			// Track import results
			if ( $result ) {
				$import_results['inserted'] ++;
			} else {
				$import_results['errors'] ++;
			}
		}

		return $import_results;
	}

}