<?php
/**
 * The file that defines all the actions for the notes management.
 *
 * @link       https://realhomes.io/
 * @since      1.0.0
 *
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 */

/**
 * The notes management class.
 *
 * This is used to define all the actions and data management of notes.
 *
 * @since      1.0.0
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 * @author     InspiryThemes <fahid@inspirythemes.com>
 */
class Realhomes_Crm_Notes {

	/**
	 * Static function to insert a note
	 *
	 * @param $note_data
	 *
	 * @return array
	 */
	public static function insert_note( $note_data ) {
		global $wpdb;

		// Sanitize and prepare data from the $_POST request
		$user_id         = get_current_user_id(); // Get the logged-in user's ID
		$related_item_id = isset( $note_data['related_item_id'] ) ? intval( $note_data['related_item_id'] ) : 0;
		$note            = isset( $note_data['note'] ) ? sanitize_textarea_field( $note_data['note'] ) : '';
		$type            = isset( $note_data['type'] ) ? sanitize_text_field( $note_data['type'] ) : '';

		// Insert the note into the database
		$result = $wpdb->insert(
			$wpdb->prefix . Realhomes_Crm_Helper::$table_notes,
			array(
				'user_id'         => $user_id,
				'related_item_id' => $related_item_id,
				'note'            => $note,
				'type'            => $type,
				'created_at'      => current_time( 'mysql' )
			),
			array(
				'%d',   // user_id is an integer
				'%d',   // related_item_id is an integer
				'%s',   // note is a string
				'%s',   // type is a string
				'%s'    // created_at is a datetime string
			)
		);

		// Check for errors during insertion
		if ( $result === false ) {
			// Handle the error if insert failed
			error_log( 'Database insert note failed: ' . $wpdb->last_error );

			return [ 'success' => false ];
		}

		return [ 'success' => true, 'insert_id' => $wpdb->insert_id ];
	}

	public static function fetch_notes( $related_item_id, $type ) {
		global $wpdb;

		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_notes;

		// Fetch notes based on related_item_id and type
		$query = $wpdb->prepare(
			"SELECT * FROM $table_name WHERE related_item_id = %d AND type = %s ORDER BY created_at DESC",
			$related_item_id,
			$type
		);

		return $wpdb->get_results( $query );
	}

	public static function delete_note( $note_id ) {
		global $wpdb;

		// Make sure the note ID is valid and not empty
		if ( ! empty( $note_id ) && intval( $note_id ) > 0 ) {

			// Prepare and execute the delete query
			$result = $wpdb->delete( $wpdb->prefix . Realhomes_Crm_Helper::$table_notes, array( 'id' => intval( $note_id ) ) );

			if ( $result === false ) {
				// Handle the error if delete failed
				error_log( 'Note delete failed: ' . $wpdb->last_error );
			}

			return $result;
		}
	}

	/**
	 * Fetch a note by ID and generate its HTML markup.
	 *
	 * @param int $note_id The ID of the note.
	 *
	 * @return string The HTML markup for the note.
	 */
	public static function fetch_note_ajax( int $note_id ): string {
		global $wpdb;

		// Fetch the note from the database.
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_notes;
		$note       = $wpdb->get_row(
			$wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d", $note_id )
		);

		// If no note is found, return an empty string.
		if ( ! $note ) {
			return '';
		}

		// Generate the HTML markup for the note.
		ob_start();
		?>
        <div id="note-<?php echo esc_attr( $note_id ); ?>" class="rh-crm-item-wrapper border mt-10 rounded text-sm hidden">
            <div class="flex justify-between p-6 border-b">
                <div class="flex items-center">
					<?php Realhomes_Crm_SVG::the_svg( 'clock', 'size-6 mr-2 fill-[var(--dashboard-primary-color)]' ); ?>
                    <span class="text-gray-700">
                        <?php echo Realhomes_Crm_Helper::human_readable_date( $note->created_at ); ?>
                    </span>
                </div>
                <div>
                    <a data-type="note" data-id="<?php echo esc_attr( $note->id ); ?>" class="rh-crm-delete-item flex items-center hover:text-red-400 cursor-pointer">
						<?php Realhomes_Crm_SVG::the_svg( 'delete', 'size-6' ); ?>
						<?php esc_html_e( 'Delete', REALHOMES_CRM_TEXT_DOMAIN ); ?>
                    </a>
                </div>
            </div>
            <div class="p-6 leading-8">
				<?php echo esc_html( $note->note ); ?>
            </div>
        </div>
		<?php
		$note_markup = ob_get_clean();
		ob_end_clean();

		return $note_markup;
	}

}