<?php
/**
 * The file that defines all the actions for the leads management.
 *
 * @link       https://realhomes.io/
 * @since      1.0.0
 *
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 */

/**
 * The leads management class.
 *
 * This is used to define all the actions and data management of leads.
 *
 * @since      1.0.0
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 * @author     InspiryThemes <fahid@inspirythemes.com>
 */
class Realhomes_Crm_Leads {

	/**
	 * Insert a lead into the database that came from a front-form submission
	 *
	 * @return int
	 */
	public function insert_lead(): int {
		global $wpdb;

		// Sanitize and prepare the data
		$name_prefix = $_POST['prefix'] ?? '';
		$name        = $_POST['name'] ?? '';
		$fl_name     = explode( ' ', $name );
		$home_phone  = $_POST['home_phone'] ?? '';
		$work_phone  = $_POST['work_phone'] ?? '';
		$address     = $_POST['address'] ?? '';
		$city        = $_POST['city'] ?? '';
		$state       = $_POST['state'] ?? '';
		$country     = $_POST['country'] ?? '';
		$zipcode     = $_POST['zipcode'] ?? '';
		$source      = $_POST['source'] ?? '';
		$email       = $_POST['email'] ?? '';
		$message     = $_POST['message'] ?? '';

		// Preparing phone number
		$possible_number_keys = [ 'number', 'phone' ];
		$number               = '';

		foreach ( $possible_number_keys as $key ) {
			if ( isset( $_POST[ $key ] ) ) {
				$number = $_POST[ $key ];
				break; // Stop at the first found value
			}
		}

		// Prepare source ID
		$possible_source_keys = [ 'property_id', 'the_id', 'agent_id' ];
		$source_id            = '';

		foreach ( $possible_source_keys as $key ) {
			if ( isset( $_POST[ $key ] ) ) {
				$source_id = intval( $_POST[ $key ] );
				break; // Stop at the first found value
			}
		}

		// Prepare data for insertion
		$lead_data = array(
			'name_prefix'    => sanitize_text_field( $name_prefix ), // Lead name prefix
			'display_name'   => sanitize_text_field( $name ), // Lead name
			'first_name'     => sanitize_text_field( $fl_name[0] ?? '' ), // Lead first name
			'last_name'      => sanitize_text_field( $fl_name[1] ?? '' ), // Lead last name
			'email'          => sanitize_email( $email ), // Lead email (optional)
			'mobile'         => sanitize_text_field( $number ), // Lead mobile number (optional)
			'home_phone'     => sanitize_text_field( $home_phone ), // Lead home phone (optional)
			'work_phone'     => sanitize_text_field( $work_phone ), // Lead work phone (optional)
			'address'        => sanitize_text_field( $address ), // Lead address (optional)
			'city'           => sanitize_text_field( $city ), // Lead city (optional)
			'state'          => sanitize_text_field( $state ), // Lead state (optional)
			'country'        => sanitize_text_field( $country ), // Lead country (optional)
			'zipcode'        => intval( $zipcode ), // Lead zipcode (optional)
			'message'        => sanitize_text_field( $message ), // Lead message (optional)
			'status'         => sanitize_text_field( Realhomes_Crm_Helper::get_setting( 'lead_statuses', explode: true )[0] ), // Lead status
			'type'           => sanitize_text_field( Realhomes_Crm_Helper::get_setting( 'lead_types', explode: true )[0] ), // Lead type
			'source'         => sanitize_text_field( empty( $source ) ? get_the_title( $source_id ) : $source ), // Lead source (lead form page title)
			'source_url'     => esc_url( get_permalink( $source_id ) ), // Lead status (lead form page url)

			// Following data is fetched from the WP user if a user exists with the lead email
			'lead_author_id' => 0, // Set default admin as lead author.
			'x_url'          => '',
			'facebook_url'   => '',
			'linkedin_url'   => ''
		);

		// If a WP user has a lead email then get its data and put in the lead otherwise set empty values for the expected data keys.
		if ( $lead_user = $this->is_lead_email_user_exists( $email ) ) {
			$lead_data = array_merge( $lead_data, array_filter( $this->get_lead_user_data( $lead_user ) ) );
		}

		// Define the format for each value in the array
		$format = array(
			'%s', // name_prefix (string)
			'%s', // display_name (string)
			'%s', // first_name (string)
			'%s', // last_name (string)
			'%s', // email (string)
			'%s', // mobile (string)
			'%s', // home_phone (string)
			'%s', // work_phone (string)
			'%s', // address (string)
			'%s', // city (string)
			'%s', // state (string)
			'%s', // country (string)
			'%s', // zipcode (string)
			'%s', // type (string)
			'%s', // status (string)
			'%s', // source (string)
			'%s', // source_url (string)
			'%s', // message (string)
		);

		// Insert data into the 'realhomes_crm_leads' table
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_leads;

		$result = $wpdb->insert( $table_name, $lead_data, $format );

		if ( $result === false ) {
			// Handle the error if insert failed
			error_log( 'Auto lead insert failed: ' . $wpdb->last_error );
		}

		// Return the ID of the newly inserted inquiry
		return $wpdb->insert_id;
	}

	/**
	 * Insert a custom lead into the database that came from "Add new lead" form.
	 *
	 * @param $data
	 *
	 * @return array|false[]
	 */
	public static function insert_custom_lead( $data ): array {
		global $wpdb;

		// Sanitize and prepare the data
		$lead_data = [
			'lead_author_id' => get_current_user_id(),
			'name_prefix'    => sanitize_text_field( $data['name_prefix'] ),
			'display_name'   => sanitize_text_field( $data['display_name'] ),
			'first_name'     => sanitize_text_field( $data['first_name'] ),
			'last_name'      => sanitize_text_field( $data['last_name'] ),
			'email'          => sanitize_email( $data['email'] ),
			'mobile'         => sanitize_text_field( $data['mobile'] ),
			'home_phone'     => sanitize_text_field( $data['home_phone'] ),
			'work_phone'     => sanitize_text_field( $data['work_phone'] ),
			'address'        => sanitize_text_field( $data['address'] ),
			'city'           => sanitize_text_field( $data['city'] ),
			'state'          => sanitize_text_field( $data['state'] ),
			'country'        => sanitize_text_field( $data['country'] ),
			'zipcode'        => sanitize_text_field( $data['zipcode'] ),
			'type'           => sanitize_text_field( $data['type'] ),
			'status'         => sanitize_text_field( $data['status'] ),
			'source'         => sanitize_text_field( $data['source'] ),
			'source_url'     => sanitize_text_field( $data['source_url'] ),
			'x_url'          => sanitize_text_field( $data['x_url'] ),
			'facebook_url'   => sanitize_text_field( $data['facebook_url'] ),
			'linkedin_url'   => sanitize_text_field( $data['linkedin_url'] ),
			'private_note'   => sanitize_text_field( $data['private_note'] ),
			'message'        => sanitize_text_field( $data['message'] ),
		];

		// Define the format for each value in the array
		$format = [
			'%d', // lead_author_id (integer)
			'%s', // name_prefix (string)
			'%s', // display_name (string)
			'%s', // first_name (string)
			'%s', // last_name (string)
			'%s', // email (string)
			'%s', // mobile (string)
			'%s', // home_phone (string)
			'%s', // work_phone (string)
			'%s', // address (string)
			'%s', // city (string)
			'%s', // state (string)
			'%s', // country (string)
			'%s', // zipcode (string)
			'%s', // type (string)
			'%s', // status (string)
			'%s', // source (string)
			'%s', // source_url (string)
			'%s', // x_url (string)
			'%s', // facebook_url (string)
			'%s', // linkedin_url (string)
			'%s', // private_note (string)
			'%s', // message (string)
		];

		// Insert data into the 'realhomes_crm_leads' table
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_leads;

		$result = $wpdb->insert( $table_name, $lead_data, $format );

		if ( $result === false ) {
			// Handle the error if insert failed
			error_log( 'Custom lead insert failed: ' . $wpdb->last_error );

			return [ 'success' => false ];
		}

		return [ 'success' => false, 'lead_id' => $wpdb->insert_id ];
	}

	/**
	 * Update a lead into the database.
	 *
	 * @param $data
	 *
	 * @return false[]|true[]
	 */
	public static function update_lead( $data ): array {
		global $wpdb;

		// Table name
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_leads;

		// Sanitize and get all POST data
		$lead_id = intval( $data['item_id'] );
		$data    = array(
			'name_prefix'  => sanitize_text_field( $data['name_prefix'] ),
			'display_name' => sanitize_text_field( $data['display_name'] ),
			'first_name'   => sanitize_text_field( $data['first_name'] ),
			'last_name'    => sanitize_text_field( $data['last_name'] ),
			'email'        => sanitize_email( $data['email'] ),
			'mobile'       => sanitize_text_field( $data['mobile'] ),
			'home_phone'   => sanitize_text_field( $data['home_phone'] ),
			'work_phone'   => sanitize_text_field( $data['work_phone'] ),
			'address'      => sanitize_text_field( $data['address'] ),
			'city'         => sanitize_text_field( $data['city'] ),
			'state'        => sanitize_text_field( $data['state'] ),
			'country'      => sanitize_text_field( $data['country'] ),
			'zipcode'      => sanitize_text_field( $data['zipcode'] ),
			'type'         => sanitize_text_field( $data['type'] ),
			'status'       => sanitize_text_field( $data['status'] ),
			'source'       => sanitize_text_field( $data['source'] ),
			'source_url'   => esc_url( $data['source_url'] ),
			'x_url'        => esc_url( $data['x_url'] ),
			'facebook_url' => esc_url( $data['facebook_url'] ),
			'linkedin_url' => esc_url( $data['linkedin_url'] ),
			'private_note' => sanitize_textarea_field( $data['private_note'] ),
			'message'      => sanitize_textarea_field( $data['message'] ),
		);

		// Where clause to specify the lead ID
		$where = array( 'id' => $lead_id );

		// Update the lead in the database
		$result = $wpdb->update( $table_name, $data, $where );

		if ( $result === false ) {
			// Handle the error if update failed
			error_log( 'Lead update failed: ' . $wpdb->last_error );

			return [ 'success' => false ];
		}

		return [ 'success' => true ];
	}

	/**
	 * Delete a single lead based on the given lead ID
	 *
	 * @param $lead_id
	 *
	 * @return void
	 */
	public static function delete_lead( $lead_id ) {
		global $wpdb;

		// Make sure the lead ID is valid and not empty
		if ( ! empty( $lead_id ) && intval( $lead_id ) > 0 ) {
			// Table name
			$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_leads;

			// Prepare and execute delete query
			$result = $wpdb->delete( $table_name, array( 'id' => intval( $lead_id ) ) );

			if ( $result === false ) {
				// Handle the error if delete failed
				error_log( 'Lead delete failed: ' . $wpdb->last_error );
			}

			return $result;
		}
	}


	/**
	 * Function to fetch single lead from the database based on given lead ID
	 *
	 * @param $lead_id
	 *
	 * @return array|false|object|stdClass
	 */
	public static function fetch_lead( $lead_id ) {
		global $wpdb;

		// Table name for leads
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_leads;

		// Validate the lead ID
		$lead_id = absint( $lead_id );

		if ( ! $lead_id ) {
			return false; // Return false if the lead ID is invalid
		}

		// Prepare the SQL query to fetch the lead by ID
		$sql = $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d", $lead_id );

		// Execute the query and fetch the single lead
		$lead = $wpdb->get_row( $sql );

		// Return the lead data, or false if no lead is found
		return $lead ?? false;
	}

	/**
	 * Function to fetch all leads from the database
	 *
	 * @param $args
	 *
	 * @return array|object|stdClass[]|null
	 */
	public static function fetch_leads( $args = array(), $fields = array() ) {
		global $wpdb;

		// Table name for leads
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_leads;

		// Default query arguments
		$defaults = array(
			'number'  => 10,          // Number of leads to retrieve (default: 10)
			'offset'  => 0,           // Offset for pagination (default: 0)
			'orderby' => 'created_at',// Order by column (default: created_at)
			'order'   => 'DESC',      // Order direction (default: DESC)
		);

		// Merge the default arguments with the provided ones
		if ( is_array( $args ) ) {
			$args = wp_parse_args( $args, $defaults );
		} else {
			$args = $defaults;
		}

		// Determine fields to select: if specific fields are provided, use those; otherwise, select all (*)
		$select_fields = '*';
		if ( ! empty( $fields ) && is_array( $fields ) ) {
			// Sanitize each field to prevent SQL injection
			$select_fields = implode( ', ', array_map( 'esc_sql', $fields ) );
		}

		// Start building the SQL query with selected fields
		$sql = "SELECT {$select_fields} FROM $table_name WHERE 1=1";

		// Add filtering by status, if provided
		if ( ! empty( $args['status'] ) ) {
			$sql .= $wpdb->prepare( " AND status = %s", $args['status'] );
		}

		// Add a search query if provided (search by name or email)
		if ( ! empty( $args['keyword'] ) ) {
			$keyword = '%' . $wpdb->esc_like( $args['keyword'] ) . '%';
			$sql     .= $wpdb->prepare( " AND (display_name LIKE %s OR email LIKE %s OR mobile LIKE %s OR message LIKE %s OR status LIKE %s OR type LIKE %s OR address LIKE %s OR source LIKE %s)", $keyword, $keyword, $keyword, $keyword, $keyword, $keyword, $keyword, $keyword );
		}

		// Add ordering and pagination
		$sql .= $wpdb->prepare( " ORDER BY {$args['orderby']} {$args['order']} LIMIT %d OFFSET %d", $args['number'], $args['offset'] );

		// Execute the query and return the results
		return $wpdb->get_results( $sql );
	}

	/**
	 * Fetch saved searches for a lead based on lead email address
	 *
	 * @param $lead_id
	 *
	 * @return array|false|object|stdClass[]|null
	 */
	public static function fetch_lead_saved_searches( $lead_email ) {
		global $wpdb;

		// Check if a WP user exists with this email
		$user = get_user_by( 'email', $lead_email );
		if ( ! $user || empty( $user->ID ) ) {
			return false; // No user found with this email
		}

		// Fetch saved searches from the old 'realhomes_saved_searched' table
		$saved_searches_table = $wpdb->prefix . Realhomes_Crm_Helper::$table_searches;
		$user_id              = $user->ID;
		$query                = $wpdb->prepare( "
            SELECT * FROM $saved_searches_table WHERE user_id = %d
        ",
			$user_id );

		// Query to get saved searches based on user_id
		return $wpdb->get_results( $query );
	}

	/**
	 * Check if a WordPress user exists with the given email.
	 *
	 * @param string $email The email to check.
	 *
	 * @return WP_User|false WP_User object if the user exists, or false if not found.
	 */
	public function is_lead_email_user_exists( string $email ) {
		// Sanitize the email input
		$email = sanitize_email( $email );

		// Check if the email is valid
		if ( is_email( $email ) ) {
			// Use get_user_by() to check if a WordPress user exists with this email
			$user = get_user_by( 'email', $email );

			if ( $user ) {
				// If user is found, return the WP_User object
				return $user;
			} else {
				// If no user found, return false
				return false;
			}
		} else {
			// If email is invalid, return false
			return false;
		}
	}

	/**
	 * Get user data including meta fields for the provided WordPress user.
	 *
	 * @param WP_User $user The WP_User object containing user information.
	 *
	 * @return array|false Array of user data if the user exists, or false if invalid object.
	 */
	public function get_lead_user_data( $user ) {
		// Ensure we have a valid WP_User object
		if ( $user instanceof WP_User ) {
			$user_id   = $user->ID;
			$user_meta = get_user_meta( $user_id );

			// Prepare an array with the required user data
			$user_data = [
				'lead_author_id' => $user_id,
				'first_name'     => sanitize_text_field( $user_meta['first_name'][0] ),
				'last_name'      => sanitize_text_field( $user_meta['last_name'][0] ),
				'work_phone'     => sanitize_text_field( $user_meta['office_number'][0] ),
				'address'        => sanitize_text_field( $user_meta['inspiry_user_address'][0] ),
				'x_url'          => esc_url( $user_meta['twitter_url'][0] ),
				'facebook_url'   => esc_url( $user_meta['facebook_url'][0] ),
				'linkedin_url'   => esc_url( $user_meta['linkedin_url'][0] )
			];

			return $user_data;
		}

		// Return false if an invalid object is passed
		return false;
	}

	/**
	 * Handle exporting leads as a CSV file
	 *
	 * @return void
	 */
	public static function export_leads() {
		// Retrieve all leads data
		$leads = self::fetch_leads();

		// Set CSV headers
		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=leads' . '.csv' );

		// Open PHP output stream as the file
		$output = fopen( 'php://output', 'w' );

		// Define CSV column headers
		fputcsv( $output, array(
			'Name Prefix',
			'Display Name',
			'First Name',
			'Last Name',
			'Email',
			'Mobile',
			'Home Phone',
			'Work Phone',
			'Address',
			'City',
			'State',
			'Country',
			'Zipcode',
			'Type',
			'Status',
			'Source',
			'Source URL',
			'X URL',
			'Facebook URL',
			'Linkedin URL',
			'Private Note',
			'Message',
		) );

		// Add each lead's data as a new row in the CSV
		foreach ( $leads as $lead ) {
			fputcsv( $output, array(
				$lead->name_prefix,
				$lead->display_name,
				$lead->first_name,
				$lead->last_name,
				$lead->email,
				$lead->mobile,
				$lead->home_phone,
				$lead->work_phone,
				$lead->address,
				$lead->city,
				$lead->state,
				$lead->country,
				$lead->zipcode,
				$lead->type,
				$lead->status,
				$lead->source,
				$lead->source_url,
				$lead->x_url,
				$lead->facebook_url,
				$lead->linkedin_url,
				$lead->private_note,
				$lead->message,
			) );
		}

		// Close the output stream
		fclose( $output );

		exit;
	}

	/**
	 * Import leads from the imported CSV file
	 *
	 * @return array
	 */
	public static function import_leads( $file_handle, $headers, $fields_mapping ) {
		// Array to hold import results
		$import_results = [
			'inserted'   => 0,
			'duplicates' => 0,
			'errors'     => 0
		];

		// Loop through each row
		while ( ( $csv_data = fgetcsv( $file_handle ) ) !== false ) {
			$lead_data = [];
			foreach ( $fields_mapping as $field_mapping => $csv_field ) {
				$index                       = array_search( $csv_field, $headers );
				$lead_data[ $field_mapping ] = $index !== false ? $csv_data[ $index ] : '';
			}

			// Validate and sanitize data
			if ( ! filter_var( $lead_data['email'], FILTER_VALIDATE_EMAIL ) ) {
				$import_results['errors'] ++;
				continue; // Skip invalid rows
			}

			// Check for duplicates
			global $wpdb;
			$existing_lead = $wpdb->get_var( $wpdb->prepare(
				"SELECT COUNT(*) FROM " . $wpdb->prefix . Realhomes_Crm_Helper::$table_leads . " WHERE email = %s",
				$lead_data['email']
			) );

			if ( $existing_lead ) {
				$import_results['duplicates'] ++;
				continue; // Skip duplicate entries
			}

			// Insert lead into the database
			$result = $wpdb->insert(
				$wpdb->prefix . Realhomes_Crm_Helper::$table_leads, // Leads table name
				[
					'name_prefix'  => sanitize_text_field( $lead_data['name_prefix'] ),
					'display_name' => sanitize_text_field( $lead_data['display_name'] ),
					'first_name'   => sanitize_text_field( $lead_data['first_name'] ),
					'last_name'    => sanitize_text_field( $lead_data['last_name'] ),
					'email'        => sanitize_email( $lead_data['email'] ),
					'mobile'       => sanitize_text_field( $lead_data['mobile'] ),
					'home_phone'   => sanitize_text_field( $lead_data['home_phone'] ),
					'work_phone'   => sanitize_text_field( $lead_data['work_phone'] ),
					'address'      => sanitize_text_field( $lead_data['address'] ),
					'city'         => sanitize_text_field( $lead_data['city'] ),
					'state'        => sanitize_text_field( $lead_data['state'] ),
					'country'      => sanitize_text_field( $lead_data['country'] ),
					'zipcode'      => sanitize_text_field( $lead_data['zipcode'] ),
					'type'         => sanitize_text_field( $lead_data['type'] ),
					'status'       => sanitize_text_field( $lead_data['status'] ),
					'source'       => sanitize_text_field( $lead_data['source'] ),
					'source_url'   => esc_url_raw( $lead_data['source_url'] ),
					'x_url'        => esc_url_raw( $lead_data['x_url'] ),
					'facebook_url' => esc_url_raw( $lead_data['facebook_url'] ),
					'linkedin_url' => esc_url_raw( $lead_data['linkedin_url'] ),
					'private_note' => sanitize_textarea_field( $lead_data['private_note'] ),
					'message'      => sanitize_textarea_field( $lead_data['message'] ),
				],
				[
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s',
					'%s'
				]
			);

			// Track import results
			if ( $result ) {
				$import_results['inserted'] ++;
			} else {
				$import_results['errors'] ++;
			}
		}

		return $import_results;
	}

}