<?php
/**
 * The file that defines all the actions for the inquiries management.
 *
 * @link       https://realhomes.io/
 * @since      1.0.0
 *
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 */

/**
 * The leads management class.
 *
 * This is used to define all the actions and data management of inquiries.
 *
 * @since      1.0.0
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 * @author     InspiryThemes <fahid@inspirythemes.com>
 */
class Realhomes_Crm_Inquiries {

	/**
	 * Inserts a new inquiry into the database.
	 *
	 * Validates and sanitizes input data, then inserts the inquiry record.
	 *
	 * @param $lead_id
	 *
	 * @return int|WP_Error
	 */
	public function insert_inquiry( $lead_id, $hook_name ) {
		global $wpdb;

		// Check that required data is provided
		if ( $lead_id <= 0 ) {
			return new WP_Error( 'invalid_data', 'Invalid lead ID.' );
		}

		// Prepare the data for insertion
		$lead_id            = $lead_id ?? 0;
		$property_id        = $_POST['property_id'] ?? 0;
		$inquired_user_id   = $_POST['inquired_user_id'] ?? 0;
		$inquired_user_type = $_POST['inquired_user_type'] ?? '';
		$message            = $_POST['message'] ?? '';
		$private_note       = $_POST['private_note'] ?? '';
		$status             = $_POST['status'] ?? Realhomes_Crm_Helper::get_setting( 'inquiry_statuses', explode: true )[0];
		$type               = $_POST['type'] ?? Realhomes_Crm_Helper::get_setting( 'inquiry_types', explode: true )[0];

		if ( 'rhea_after_inquiry_form_submission' === $hook_name ) {
			$query['min-price']        = isset( $_POST['min_price'] ) ? sanitize_text_field( $_POST['min_price'] ) : 0;
			$query['max-price']        = isset( $_POST['max_price'] ) ? sanitize_text_field( $_POST['max_price'] ) : 0;
			$query['min-beds']         = isset( $_POST['min_beds'] ) ? sanitize_text_field( $_POST['min_beds'] ) : 0;
			$query['max-beds']         = isset( $_POST['max_beds'] ) ? sanitize_text_field( $_POST['max_beds'] ) : 0;
			$query['min-baths']        = isset( $_POST['min_baths'] ) ? sanitize_text_field( $_POST['min_baths'] ) : 0;
			$query['max-baths']        = isset( $_POST['max_baths'] ) ? sanitize_text_field( $_POST['max_baths'] ) : 0;
			$query['min-garage']       = isset( $_POST['min_garage'] ) ? sanitize_text_field( $_POST['min_garage'] ) : 0;
			$query['max-garage']       = isset( $_POST['max_garage'] ) ? sanitize_text_field( $_POST['max_garage'] ) : 0;
			$query['min-area']         = isset( $_POST['min_area'] ) ? sanitize_text_field( $_POST['min_area'] ) : 0;
			$query['max-area']         = isset( $_POST['max_area'] ) ? sanitize_text_field( $_POST['max_area'] ) : 0;
			$query['property-zipcode'] = isset( $_POST['property_zipcode'] ) ? sanitize_text_field( $_POST['property_zipcode'] ) : 0;
			$query['property-address'] = isset( $_POST['property_address'] ) ? sanitize_text_field( $_POST['property_address'] ) : '';

			$query['property-city']   = isset( $_POST['property_city'] ) ? [ sanitize_text_field( $_POST['property_city'] ) ] : '';
			$query['property-status'] = isset( $_POST['property_status'] ) ? [ sanitize_text_field( $_POST['property_status'] ) ] : '';
			$query['property-type']   = isset( $_POST['property_type'] ) ? [ sanitize_text_field( $_POST['property_type'] ) ] : '';
		} else {
			// Check that required data is provided
			if ( $property_id <= 0 ) {
				return new WP_Error( 'invalid_data', 'Invalid property ID.' );
			}

			$query = self::get_property_info( $property_id );
		}


		// Prepare data to insert
		$data = array(
			'lead_id'            => intval( $lead_id ),
			'property_id'        => intval( $property_id ),
			'inquired_user_id'   => intval( $inquired_user_id ),
			'inquired_user_type' => sanitize_text_field( $inquired_user_type ),
			'message'            => sanitize_text_field( $message ),
			'private_note'       => sanitize_text_field( $private_note ),
			'status'             => sanitize_text_field( $status ),
			'type'               => sanitize_text_field( $type ),
			'query'              => maybe_serialize( $query ),
			'created_at'         => current_time( 'mysql' ),  // Set created_at time
			'updated_at'         => current_time( 'mysql' )   // Set updated_at time
		);

		// Define the formats for each field in the $data array
		$format = array(
			'%d',  // lead_id is an integer
			'%d',  // property_id is an integer
			'%d',  // inquired_user_id is an integer
			'%s',  // inquired_user_type is a string
			'%s',  // message is a text
			'%s',  // private_note is a text
			'%s',  // status is a string
			'%s',  // type is a string
			'%s',  // query is a text
			'%s',  // created_at is a datetime string
			'%s'   // updated_at is a datetime string
		);

		// Table name for inquiries
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_inquiries;

		// Insert the inquiry into the database
		$result = $wpdb->insert( $table_name, $data, $format );

		// Check for errors
		if ( false === $result ) {
			return new WP_Error( 'db_insert_error', 'Failed to insert inquiry into the database.' );
		}

		// Return the ID of the newly inserted inquiry
		return $wpdb->insert_id;
	}

	/**
	 * Inserts a custom inquiry into the database.
	 *
	 * Collects and sanitizes inquiry data from POST, prepares it for insertion,
	 * and inserts it into the inquiries table.
	 *
	 * @return int|false The number of rows affected or false on failure.
	 * @global wpdb $wpdb WordPress database object.
	 */
	public static function insert_custom_inquiry( $data ) {
		global $wpdb;

		// Prepare the data for updating
		$inquiry_data = array(
			'lead_id'      => intval( $data['lead-id'] ),
			'message'      => sanitize_textarea_field( $data['message'] ),
			'private_note' => sanitize_textarea_field( $data['private-note'] ),
			'status'       => sanitize_text_field( $data['status'] ),
			'type'         => sanitize_text_field( $data['type'] ),
			'query'        => maybe_serialize( self::prepare_query_data( $data ) ),
			'updated_at'   => current_time( 'mysql' ) // Update timestamp
		);

		// Table name for inquiries
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_inquiries;

		$format = array(
			'%d', // lead_id is an integer
			'%s', // message is a string
			'%s', // private_note is a string
			'%s', // status is a string
			'%s', // type is a string
			'%s', // query is a string
			'%s'  // updated_at is a datetime string
		);

		// Insert the inquiry into the database
		$result = $wpdb->insert( $table_name, $inquiry_data, $format );

		// Check for errors
		if ( $result === false ) {
			// Handle the error if insert failed
			error_log( 'Database insert failed: ' . $wpdb->last_error );
		}

		return $wpdb->insert_id;
	}

	/**
	 * Updates an existing inquiry in the database.
	 *
	 * Validates and sanitizes input data from POST, and updates the inquiry record by ID.
	 *
	 * @return true|WP_Error True on successful update, or WP_Error on failure.
	 * @global wpdb $wpdb WordPress database object.
	 */
	public static function update_inquiry( $data ) {
		global $wpdb;

		// Check if the inquiry ID is set in POST
		if ( isset( $data['item_id'] ) ) {
			$inquiry_id = intval( $data['item_id'] );

			// Prepare the data for updating
			$inquiry_data = array(
				'lead_id'      => intval( $data['lead-id'] ),
				'message'      => sanitize_textarea_field( $data['message'] ),
				'private_note' => sanitize_textarea_field( $data['private-note'] ),
				'status'       => sanitize_text_field( $data['status'] ),
				'type'         => sanitize_text_field( $data['type'] ),
				'query'        => maybe_serialize( self::prepare_query_data( $data ) ),
				'updated_at'   => current_time( 'mysql' ) // Update timestamp
			);

			// Update the inquiry in the database
			$wpdb->update(
				$wpdb->prefix . Realhomes_Crm_Helper::$table_inquiries,
				$inquiry_data,
				array( 'id' => $inquiry_id ),
				array(
					'%d', // lead_id is an integer
					'%s', // message is a string
					'%s', // private_note is a string
					'%s', // status is a string
					'%s', // type is a string
					'%s', // query is a string
					'%s'  // updated_at is a datetime string
				),
				array( '%d' ) // inquiry ID is an integer
			);

			// Optional: Return success or error message
			if ( $wpdb->last_error ) {
				return new WP_Error( 'db_update_error', 'Failed to update inquiry in the database.' );
			}

			return true; // Indicate success
		}

		return new WP_Error( 'invalid_data', 'Inquiry ID is missing.' );
	}


	/**
	 * Deletes an inquiry from the database by its ID.
	 *
	 * Ensures a valid inquiry ID is provided, then deletes the corresponding record.
	 *
	 * @param int   $inquiry_id The ID of the inquiry to delete.
	 *
	 * @return void
	 * @global wpdb $wpdb       WordPress database object.
	 */
	public static function delete_inquiry( $inquiry_id ) {
		global $wpdb;

		// Make sure the inquiry ID is valid and not empty
		if ( ! empty( $inquiry_id ) && intval( $inquiry_id ) > 0 ) {
			// Table name
			$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_inquiries;

			// Prepare and execute delete query
			$result = $wpdb->delete( $table_name, array( 'id' => intval( $inquiry_id ) ) );

			if ( $result === false ) {
				// Handle the error if delete failed
				error_log( 'Inquiry delete failed: ' . $wpdb->last_error );
			}

			return $result;
		}
	}

	/**
	 * Fetches an inquiry by its ID.
	 *
	 * Retrieves a single inquiry record from the database based on the provided ID.
	 *
	 * @param int   $inquiry_id The ID of the inquiry to fetch.
	 *
	 * @return object|WP_Error The inquiry object on success, or WP_Error if not found or ID is invalid.
	 * @global wpdb $wpdb       WordPress database object.
	 */
	public static function fetch_inquiry( $inquiry_id ) {
		global $wpdb;

		// Table name for inquiries
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_inquiries;

		// Check if the inquiry ID is valid
		if ( empty( $inquiry_id ) || ! is_numeric( $inquiry_id ) ) {
			return new WP_Error( 'invalid_inquiry_id', 'Invalid inquiry ID.' );
		}

		// Prepare the SQL query to fetch the inquiry by ID
		$sql = $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d", $inquiry_id );

		// Execute the query and return the result
		$inquiry = $wpdb->get_row( $sql );

		// If no inquiry is found, return an error
		if ( null === $inquiry ) {
			return new WP_Error( 'no_inquiry_found', 'No inquiry found with the provided ID.' );
		}

		return $inquiry;
	}

	/**
	 * Retrieves a list of inquiries from the database with optional filters and pagination.
	 *
	 * Allows filtering by keyword, lead ID, sorting by column, and setting a limit and offset.
	 *
	 * @param array $args Optional. Arguments to filter and paginate inquiries.
	 *
	 * @return array|object|null List of inquiries or null if none found.
	 * @global wpdb $wpdb WordPress database object.
	 */
	public static function fetch_inquiries( $args = array() ) {
		global $wpdb;

		// Table name for inquiries & leads
		$inquiries_table = $wpdb->prefix . Realhomes_Crm_Helper::$table_inquiries;
		$leads_table     = $wpdb->prefix . Realhomes_Crm_Helper::$table_leads;

		// Default query arguments
		$default_args = array(
			'number'  => 10,  // Number of results to return
			'offset'  => 0,   // Starting point
			'lead_id' => 0,   // Optionally filter by lead ID
			'orderby' => 'created_at',  // Column to sort by
			'order'   => 'DESC' // Order direction (ASC or DESC)
		);

		// Merge passed arguments with defaults
		$args = wp_parse_args( $args, $default_args );

		// Start building the query
		$query = "SELECT * FROM $inquiries_table WHERE 1=1";

		if ( ! empty( $args['keyword'] ) ) {
			$keyword = '%' . $wpdb->esc_like( $args['keyword'] ) . '%';

			// Add JOIN to leads table
			$query = "SELECT i.* FROM {$inquiries_table} i 
              LEFT JOIN {$leads_table} l ON i.lead_id = l.id 
              WHERE 1=1";

			$query .= $wpdb->prepare(
				" AND (i.type LIKE %s OR i.status LIKE %s OR l.display_name LIKE %s)",
				$keyword,
				$keyword,
				$keyword
			);
		}

		if ( ! empty( $args['lead_id'] ) ) {
			$query .= $wpdb->prepare( " AND lead_id = %d", $args['lead_id'] );
		}

		// Add ordering and limit to the query
		$query .= $wpdb->prepare( " ORDER BY {$args['orderby']} {$args['order']} LIMIT %d OFFSET %d", $args['number'], $args['offset'] );

		// Execute the query and get the results
		return $wpdb->get_results( $query );
	}


	/**
	 * Retrieves property information by property ID.
	 *
	 * Fetches meta information, city, type, and status taxonomy terms associated with the property.
	 * Returns serialized property details including price, bedrooms, bathrooms, garage, size, address,
	 * cities, types, and statuses.
	 *
	 * @param int $property_id The property ID.
	 *
	 * @return string|false Serialized property information on success, false if property not found.
	 */
	public static function get_property_info( $property_id ) {
		// Fetch property info from the properties table
		$property = get_post( $property_id );
		if ( $property ) {
			$property_meta = get_post_custom( $property_id );
			$property_info = [];

			// Prepare meta information
			$property_info['min-price']        = $property_info['max-price'] = sanitize_text_field( $property_meta['REAL_HOMES_property_price'][0] );
			$property_info['min-beds']         = $property_info['max-beds'] = sanitize_text_field( $property_meta['REAL_HOMES_property_bedrooms'][0] );
			$property_info['min-baths']        = $property_info['max-baths'] = sanitize_text_field( $property_meta['REAL_HOMES_property_bathrooms'][0] );
			$property_info['min-garage']       = $property_info['max-garage'] = sanitize_text_field( $property_meta['REAL_HOMES_property_garage'][0] );
			$property_info['min-area']         = $property_info['max-area'] = sanitize_text_field( $property_meta['REAL_HOMES_property_size'][0] );
			$property_info['property-address'] = sanitize_text_field( $property_meta['REAL_HOMES_property_address'][0] );

			// Prepare property city taxonomy information
			$property_cities = get_the_terms( $property_id, 'property-city' );
			if ( ! empty( $property_cities ) && ! is_wp_error( $property_cities ) ) :
				$cities_list = array();

				// Loop through each term and prepare a list
				foreach ( $property_cities as $city ) {
					$cities_list[] = sanitize_text_field( $city->slug );
				}

				$property_info['property-city'] = $cities_list;
			endif;

			// Prepare type taxonomy information
			$property_types = get_the_terms( $property_id, 'property-type' );
			if ( ! empty( $property_types ) && ! is_wp_error( $property_types ) ) :
				$types_list = array();

				// Loop through each term and prepare a list
				foreach ( $property_types as $type ) {
					$types_list[] = sanitize_text_field( $type->slug );
				}

				$property_info['property-type'] = $types_list;
			endif;

			// Prepare status taxonomy information
			$property_statuses = get_the_terms( $property_id, 'property-status' );
			if ( ! empty( $property_statuses ) && ! is_wp_error( $property_statuses ) ) :
				$statuses_list = array();

				// Loop through each term and prepare a list
				foreach ( $property_statuses as $status ) {
					$statuses_list[] = sanitize_text_field( $status->slug );
				}

				$property_info['property-status'] = $statuses_list;
			endif;

			return $property_info;
		}

		return false;
	}


	/**
	 * Prepares and sanitizes query data from the form input.
	 *
	 * This function processes the `$data` data, extracting specific fields related to property searches.
	 * It sanitizes each field and organizes them into an associative array for further use.
	 *
	 * @return array Sanitized array of query data fields.
	 */
	public static function prepare_query_data( $data ) {
		$fields_data = $data;
		$query_data  = [];

		// Define an array of fields to be extracted from the form data
		$fields = [
			'min-beds',
			'max-beds',
			'min-baths',
			'max-baths',
			'min-garage',
			'max-garage',
			'min-area',
			'max-area',
			'min-price',
			'max-price',
			'property-city',
			'property-type',
			'property-status',
			'property-address',
			'property-zipcode'
		];

		// Iterate through each field, checking if it's set in $fields_data
		foreach ( $fields as $field ) {
			if ( isset( $fields_data[ $field ] ) ) {
				// Handle array fields like property-city, property-type, and property-status
				if ( in_array( $field, array(
					'property-city',
					'property-type',
					'property-status',
				) ) ) {
					$query_data[ $field ] = [ sanitize_text_field( $fields_data[ $field ] ) ];
				} else {
					$query_data[ $field ] = sanitize_text_field( $fields_data[ $field ] );
				}
			}
		}

		return $query_data;
	}


	/**
	 * Exports all inquiries to a CSV file.
	 *
	 * This function retrieves all inquiries from the database and generates a downloadable CSV file with the inquiry details.
	 * The file includes columns for the inquiry ID, message, status, type, and creation date.
	 *
	 * The CSV is sent to the browser with appropriate headers for download as an attachment.
	 */
	public static function export_inquiries() {
		// Retrieve all inquiries data
		$inquiries = self::fetch_inquiries();

		// Set CSV headers
		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=inquiries' . '.csv' );

		// Open PHP output stream as the file
		$output = fopen( 'php://output', 'w' );

		// Initialize column headers
		$static_headers = [ 'ID', 'Contact', 'Email', 'Mobile', 'Property', 'Inquiry Type', 'Inquiry Status' ];

		// Find dynamic fields from the first inquiry
		$dynamic_headers = [];
		if ( ! empty( $inquiries ) && isset( $inquiries[0]->query ) ) {
			$query_data = unserialize( $inquiries[0]->query );
			if ( is_array( $query_data ) ) {
				$dynamic_headers = array_keys( $query_data );
			}
		}

		// Merge static and dynamic headers
		$headers = array_merge( array_merge( $static_headers, array_map( array( 'Realhomes_Crm_Helper', 'format_label' ), $dynamic_headers ) ), [ 'Message', 'Private Note' ] );

		fputcsv( $output, $headers );

		// Add each inquiry's data as a new row in the CSV
		foreach ( $inquiries as $inquiry ) {
			$lead_name = $lead_email = $lead_mobile = '';

			if ( $inquiry->lead_id ) {
				$lead = Realhomes_Crm_Leads::fetch_lead( $inquiry->lead_id );

				if ( $lead ) {
					$lead_name   = $lead->first_name . ' ' . $lead->last_name;
					$lead_email  = $lead->email;
					$lead_mobile = $lead->mobile;
				}
			}

			$static_data = [
				$inquiry->id,
				$lead_name,
				$lead_email,
				$lead_mobile,
				get_the_title( $inquiry->property_id ),
				Realhomes_Crm_Helper::format_label( $inquiry->type ),
				Realhomes_Crm_Helper::format_label( $inquiry->status ),
			];

			// Parse dynamic fields from the query
			$query_data   = unserialize( $inquiry->query );
			$dynamic_data = [];
			if ( is_array( $query_data ) ) {
				foreach ( $dynamic_headers as $field ) {
					$value = $query_data[ $field ] ?? '';

					if ( in_array( $field, [ 'property-city', 'property-type', 'property-status' ] ) ) {
						$value = Realhomes_Crm_Helper::format_label( $value );
					}

					$dynamic_data[] = $value;
				}
			}

			// Merge static and dynamic data
			$row = array_merge( array_merge( $static_data, $dynamic_data ), [ $inquiry->message, $inquiry->private_note, ] );
			fputcsv( $output, $row );
		}

		// Close the output stream
		fclose( $output );

		exit;
	}

	/**
	 * Imports inquiries from a CSV file and inserts them into the database.
	 *
	 * This function handles the import of inquiries from a CSV file uploaded via a form.
	 * It processes each row, sanitizes the data, checks for duplicates, and inserts valid rows into the database.
	 * The results of the import are displayed, including the number of successfully inserted records, duplicates, and errors.
	 *
	 * The CSV file must have columns in the order: ID, Message, Status, Type, and Creation Date.
	 */
	public static function import_inquiries( $file_handle, $headers, $fields_mapping ) {
		// Array to hold import results
		$import_results = [
			'inserted'   => 0,
			'duplicates' => 0,
			'errors'     => 0
		];

		// Loop through each row
		while ( ( $csv_data = fgetcsv( $file_handle ) ) !== false ) {
			$inquiry_data = [];
			foreach ( $fields_mapping as $field_mapping => $csv_field ) {
				$index                          = array_search( $csv_field, $headers );
				$inquiry_data[ $field_mapping ] = $index !== false ? $csv_data[ $index ] : '';
			}

			// Check for duplicates
			global $wpdb;

			// Insert inquiry into the database
			$result = $wpdb->insert(
				$wpdb->prefix . Realhomes_Crm_Helper::$table_inquiries, // Inquiries table name
				[
					'query'        => maybe_serialize( self::prepare_query_data( $inquiry_data ) ),
					'type'         => sanitize_textarea_field( $inquiry_data['inquiry_type'] ),
					'status'       => sanitize_textarea_field( $inquiry_data['inquiry_status'] ),
					'message'      => sanitize_textarea_field( $inquiry_data['message'] ),
					'private_note' => sanitize_textarea_field( $inquiry_data['private_note'] ),
					'created_at'   => current_time( 'mysql' ),
					'updated_at'   => current_time( 'mysql' ),
				],
				[ '%s', '%s', '%s', '%s' ]
			);

			// Track import results
			if ( $result ) {
				$import_results['inserted'] ++;
			} else {
				$import_results['errors'] ++;
			}
		}

		return $import_results;
	}
}