<?php
/**
 * The file that defines all the helper functions for the RealHomes CRM.
 *
 * @link       https://realhomes.io/
 * @since      1.0.0
 *
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 */

/**
 * The helper class for RealHomes CRM.
 *
 * This is used to define all the utility functions and shared data management
 * tasks across the CRM module.
 *
 * @since      1.0.0
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 * @author     InspiryThemes <fahid@inspirythemes.com>
 */
class Realhomes_Crm_Helper {

	public static $table_activities = 'realhomes_crm_activities';

	public static $table_leads = 'realhomes_crm_leads';

	public static $table_inquiries = 'realhomes_crm_inquiries';

	public static $table_deals = 'realhomes_crm_deals';

	public static $table_tasks = 'realhomes_crm_tasks';

	public static $table_notes = 'realhomes_crm_notes';

	public static $table_searches = 'realhomes_saved_searches';

	protected static $crm_settings = null;

	// Load settings once to avoid multiple get_option calls
	public static function load_settings() {
		if ( self::$crm_settings === null ) {
			self::$crm_settings = get_option( 'realhomes_crm_settings', array() );
		}
	}

	public static function setting_default( $key ) {
		$default_values = [
			'empty_values_fallback' => '-',
			'lead_name_prefixes'    => 'Mr, Mrs, Ms, Engr, Sir, Madam, Investor, Broker, Agent, CEO',
			'lead_types'            => 'Prospect, Referral, Partner, Repeat Client, Influencer, Corporate, Investor, Urgent, Property Owner, New Client, Vendor, Service Provider, Other',
			'lead_statuses'         => 'New, Contacted, Qualified, Proposal Sent, Negotiation, Won, Lost, On Hold, Follow Up',
			'inquiry_types'         => 'Purchase, Rent, Lease, Sell, Valuation, Finance, Consultation, Maintenance, Availability, Tour, Application, Custom Offer, Investment, General',
			'inquiry_statuses'      => 'New, Contacted, In Progress, Closed, On Hold, Cancelled, Follow Up, Awaiting Response',
			'deal_stages'           => 'In Progress, On Hold, Won, Lost',
			'deal_stages_colors'    => '#2196F3, #FF9800, #4CAF50, #F44336',
			'deal_probabilities'    => '10%, 25%, 50%, 75%, 90%, 100%',
			'deal_last_steps'       => 'Initial Contact, Site Visit, Phone Call, Follow Up, Proposal Sent, Meeting Held, Negotiation Started, Agreement Reached, Won, Lost',
			'deal_next_steps'       => 'Schedule Meeting, Arrange Site Visit, Send Proposal, Follow Up Call, Negotiate Terms, Awaiting Client Feedback, Send Contract, Close Deal',
			'task_types'            => 'Research, Meeting, Follow Up, Documentation, Development, Testing, Client Support, Project Planning, Training, Marketing, Review, Feedback, Reporting, Other',
			'task_statuses'         => 'Pending, In Progress, Completed, On Hold, Cancelled, Awaiting Review, Deferred, Failed, Scheduled, Not Started',
			'task_priorities'       => 'Low, Medium, High, Urgent, Critical',
			'posts_per_page_list'   => [ 5, 10, 20, 30, 40, 50, 100 ]
		];

		if ( key_exists( $key, $default_values ) ) {
			return $default_values[ $key ];
		}

		return '';
	}

	// Retrieve a specific CRM setting by key with optional default value
	public static function get_setting( $key, $default = '', $explode = false ) {
		self::load_settings();
		$value = isset( self::$crm_settings[ $key ] ) ? self::$crm_settings[ $key ] : $default;

		if ( empty( $value ) ) {
			$value = self::setting_default( $key );
		}

		if ( $explode && is_string( $value ) ) {
			return array_map( 'trim', explode( ',', $value ) );
		}

		return $value;
	}

	public static function format_label( $label ) {
		if ( is_array( $label ) ) {
			$label = implode( ', ', $label );
		}

		return ucwords( str_replace( '-', ' ', str_replace( '_', ' ', $label ) ) );
	}

	public static function get_ratio_arrow( $current, $previous ) {
		if ( 0 == $previous ) {
			$ratio = 100;
			if ( 0 == $current ) {
				$ratio = 0;
			}

			return "<span class='text-green-500'>▲ {$ratio}%</span>";
		}
		$ratio = ( ( $current - $previous ) / $previous ) * 100;
		$arrow = $ratio >= 0 ? '▲' : '▼';
		$color = $ratio >= 0 ? 'text-green-500' : 'text-red-500';

		return "<span class=\"$color\">$arrow " . round( $ratio, 0 ) . "%</span>";
	}

	public static function get_page_type_plurals() {
		return array(
			'lead'    => 'leads',
			'inquiry' => 'inquiries',
			'deal'    => 'deals',
			'task'    => 'tasks'
		);
	}

	/**
	 * Get TimeStep and return date in a human-readable format
	 *
	 * @param $timestamp
	 * @param $timezone
	 *
	 * @return string
	 */
	public static function human_readable_date( $timestamp, $timezone = null ): string {
		// Return empty string for null/empty timestamps
		if ( empty( $timestamp ) || '0000-00-00' === $timestamp ) {
			return '-';
		}

		try {
			// Handle different input types
			if ( is_numeric( $timestamp ) ) {
				// Unix timestamp
				$date = new DateTime( '@' . $timestamp );
			} else {
				// String timestamp
				$date = new DateTime( $timestamp );
			}

			// Set timezone if provided, otherwise use system default
			if ( $timezone ) {
				$date->setTimezone( new DateTimeZone( $timezone ) );
			} else {
				// Use system timezone
				$date->setTimezone( new DateTimeZone( date_default_timezone_get() ) );
			}

			// Get current time in the same timezone
			$now = new DateTime( 'now', $date->getTimezone() );

			// Get the start of "Today", "Yesterday", and "Tomorrow"
			$todayStart     = ( clone $now )->setTime( 0, 0, 0 );
			$tomorrowStart  = ( clone $todayStart )->modify( '+1 day' );
			$yesterdayStart = ( clone $todayStart )->modify( '-1 day' );
			$weekAgoStart   = ( clone $todayStart )->modify( '-7 days' );

			// Check if date is in the future (tomorrow or later)
			if ( $date >= $tomorrowStart ) {
				// For future dates beyond tomorrow, show full date
				$daysDiff = $todayStart->diff( $date )->days;
				if ( $daysDiff === 1 ) {
					return "Tomorrow";
				} else {
					return $date->format( 'M j, Y' );
				}
			} // Today
			elseif ( $date >= $todayStart ) {
				return "Today";
			} // Yesterday
			elseif ( $date >= $yesterdayStart ) {
				return "Yesterday";
			} // Within the past week - show day name
			elseif ( $date >= $weekAgoStart ) {
				return $date->format( 'l' ); // Full day name (Monday, Tuesday, etc.)
			} // Older dates - show full date
			else {
				// For dates in the current year, omit the year
//				$currentYear = $now->format( 'Y' );
//				$dateYear = $date->format( 'Y' );

//				if ( $currentYear === $dateYear ) {
//					return $date->format( 'M j' );
//				} else {
//					return $date->format( 'M j, Y' );
//				}
				return $date->format( 'M j, Y' );
			}
		} catch ( Exception $e ) {
			// Log the error if you have logging enabled
			// error_log( 'human_readable_date error: ' . $e->getMessage() );

			// Return the original timestamp or a fallback message
			return is_string( $timestamp ) ? $timestamp : 'Invalid date';
		}
	}

	/**
	 * An item action buttons to view, edit or delete
	 *
	 * @param       $item_id
	 * @param       $type
	 * @param bool  $last
	 * @param array $enabled
	 *
	 * @return void
	 */
	public static function the_item_actions( $item_id, $type, bool $last = false, array $enabled = [] ) {
		if ( empty( $enabled ) ) {
			$enabled = [ 'view', 'edit', 'delete' ];
		}
		?>
		<div class="realhomes-crm-item-actions-wrapper relative flex-none">

			<!-- Opener Icon -->
			<button type="button" class="rh-crm-item-actions-controller -m-2.5 block p-2.5 text-gray-500 hover:text-gray-900" id="options-menu-2-button" aria-expanded="false" aria-haspopup="true">
				<?php Realhomes_Crm_SVG::the_svg( 'menu-dots', 'size-8' ); ?>
			</button>

			<div class="realhomes-crm-item-actions hidden hide-it absolute right-[80%] z-10 mt-2 w-36 origin-top-right rounded-md bg-white py-2 shadow-lg ring-1 ring-gray-900/5 focus:outline-none" role="menu" aria-orientation="vertical" aria-labelledby="options-menu-2-button" tabindex="-1">

				<?php
				if ( in_array( 'view', $enabled ) ) {
					?>
					<!-- View -->
					<a data-type="<?php echo esc_attr( $type ); ?>" data-action="view" data-id="<?php echo esc_attr( $item_id ); ?>" class="rh-crm-view-item flex items-center px-6 py-3 text-sm/6 text-gray-900 hover:bg-gray-50 hover:text-blue-400 cursor-pointer" role="menuitem" tabindex="-1" id="options-menu-2-item-0">
						<?php
						Realhomes_Crm_SVG::the_svg( 'view', 'size-6 mr-2 text-blue-400' );
						esc_html_e( 'View', REALHOMES_CRM_TEXT_DOMAIN );
						?>
					</a>
					<?php
				}

				if ( in_array( 'edit', $enabled ) ) {
					?>
					<!-- Edit -->
					<a data-type="<?php echo esc_attr( $type ); ?>" data-action="edit" data-id="<?php echo esc_attr( $item_id ); ?>" class="rh-crm-add-edit-item flex items-center px-6 py-3 text-sm/6 text-gray-900 hover:bg-gray-50 hover:text-green-400 cursor-pointer" role="menuitem" tabindex="-1">
						<?php
						Realhomes_Crm_SVG::the_svg( 'edit', 'size-6 mr-2 text-green-400' );
						esc_html_e( 'Edit', REALHOMES_CRM_TEXT_DOMAIN );
						?>
					</a>
					<?php
				}

				if ( in_array( 'delete', $enabled ) ) {
					?>
					<!-- Delete -->
					<a data-type="<?php echo esc_attr( $type ); ?>" data-id="<?php echo esc_attr( $item_id ); ?>" class="rh-crm-delete-item flex items-center px-6 py-3 text-sm/6 text-gray-900 hover:bg-gray-50 hover:text-red-400 cursor-pointer" role="menuitem" tabindex="-1" id="options-menu-2-item-2">
						<?php
						Realhomes_Crm_SVG::the_svg( 'delete', 'size-6 mr-2 text-blue-400' );
						esc_html_e( 'Delete', REALHOMES_CRM_TEXT_DOMAIN ) ?>
					</a>
					<?php
				}
				?>
			</div>

		</div>
		<?php
	}

	/**
	 * Note to display when no items found to display in a list
	 *
	 * @param $page_type
	 * @param $heading
	 * @param $description
	 *
	 * @return void
	 */
	public static function no_item_found_note( $page_type = '', $heading = '', $description = '', $classes = '', $icon_size = 'w-32 h-32' ) {
		?>
		<div class="flex flex-col items-center justify-center bg-gray-50 py-32 <?php echo esc_attr( $classes ); ?>">
			<div class="text-center">
				<div class="mb-6">
					<?php Realhomes_Crm_SVG::the_svg( 'broken_page', $icon_size . ' mx-auto text-gray-500' ); ?>
				</div>
				<?php
				if ( ! empty( $heading ) || ! empty( $description ) ) {
					?>
					<h3 class="text-xl font-[500] text-gray-800"><?php echo esc_html( $heading ); ?></h3>
					<?php
					if ( ! empty( $description ) ) {
						?>
						<p class="mt-2 text-gray-600 text-base">
							<?php echo esc_html( $description ); ?>
						</p>
						<?php
					}
				} else {
					$labels = $import_pages = Realhomes_Crm_Helper::get_page_type_plurals();
					?>

					<h3 class="text-xl font-[500] text-gray-800"><?php echo sprintf( esc_html__( "No %s Found", REALHOMES_CRM_TEXT_DOMAIN ), ucfirst( $labels[ $page_type ] ) ); ?></h3><p class="mt-2 text-gray-600 text-base">
						<?php echo sprintf( esc_html__( 'It seems there are no %s to display here. You can add new %s or import data to get started.', REALHOMES_CRM_TEXT_DOMAIN ), $labels[ $page_type ], $labels[ $page_type ] ); ?>
					</p>
					<div class="mt-12 flex space-x-8 justify-center">
						<button data-type="<?php echo esc_attr( $page_type ); ?>" data-action="add" data-id="" class="rh-crm-add-edit-item rh-crm-btn-primary">
							<?php
							Realhomes_Crm_SVG::the_svg( 'loader', 'size-5 mr-2 hidden' );
							Realhomes_Crm_SVG::the_svg( 'plus', 'size-5 mr-2' );
							esc_html_e( 'Add New', REALHOMES_CRM_TEXT_DOMAIN );
							?>
						</button>

						<a href="<?php echo esc_url( realhomes_get_dashboard_page_url( 'properties-crm' ) . '&submodule=crm/import&import=' . $import_pages[ $page_type ] ) ?>" id="rh-crm-import-items" data-type="<?php echo esc_attr( $page_type ); ?>" class="rh-crm-btn-secondary">
							<?php
							Realhomes_Crm_SVG::the_svg( 'import', 'group-hover:fill-white fill-teal-600 size-5 mr-2' );
							esc_html_e( 'Import', REALHOMES_CRM_TEXT_DOMAIN );
							?>
						</a>
					</div>

					<?php
				}
				?>
			</div>
		</div>
		<?php
	}

	public static function showing_items_info( $offset, $items_per_page, $total_items ) {
		?>
		<p class="rh-crm-pagination-stats flex gap-x-2 text-sm text-gray-700 m-0">
			<span class="font-medium"><?php echo intval( $offset + 1 ); ?></span>
			<?php esc_html_e( 'to', REALHOMES_CRM_TEXT_DOMAIN ); ?>
			<span class="font-medium"><?php echo ( $items_per_page > $total_items || ( $offset + $items_per_page ) > $total_items ) ? $total_items : intval( $offset + $items_per_page ); ?></span>
			<?php esc_html_e( 'out of', REALHOMES_CRM_TEXT_DOMAIN ); ?>
			<span class="font-medium"><?php echo intval( $total_items ); ?></span>
			<?php esc_html_e( 'results', REALHOMES_CRM_TEXT_DOMAIN ); ?>
		</p>
		<?php
	}

	public static function items_pagination( $current_page, $items_per_page, $total_items ) {
		?>
		<div>
			<?php

			// Calculate the total number of pages.
			$total_pages = ceil( $total_items / $items_per_page );

			// Only display pagination if there's more than one page.
			if ( $total_pages > 1 ) {
				// Capture the current query arguments.
				$query_args = $_GET;

				// Remove the 'itemsPage' argument to avoid duplication in URLs.
				unset( $query_args['itemsPage'] );

				// Generate the pagination links.
				$pagination_args = array(
					'base'      => add_query_arg( array_merge( $query_args, [ 'itemsPage' => '%#%' ] ) ),
					'format'    => '',
					'current'   => $current_page,
					'total'     => $total_pages,
					'prev_text' => Realhomes_Crm_SVG::get_svg( 'previous', 'size-5' ),
					'next_text' => Realhomes_Crm_SVG::get_svg( 'next', 'size-5' ),
					'type'      => 'list', // Optional: Wrap links in a list for better styling.
				);

				echo '<div class="realhomes-crm-pagination">';
				echo paginate_links( $pagination_args );
				echo '</div>';
			}
			?>
		</div>
		<?php
	}

	/**
	 * Get items per page from URL or cookie with validation
	 *
	 * @param int $default Number of items per page to return if no valid value found in URL or cookie
	 *
	 * @return int Number of items per page
	 */
	public static function get_items_per_page( $default = 5 ) {
		$allowed_values = Realhomes_Crm_Helper::get_setting( 'posts_per_page_list' );

		// First check URL parameter
		if ( isset( $_GET['items'] ) ) {
			$items = intval( $_GET['items'] );
			// Validate against allowed values
			if ( in_array( $items, $allowed_values ) ) {
				// Update cookie to match URL
				setcookie( 'rh_crm_items_per_page', $items, time() + ( 30 * DAY_IN_SECONDS ), '/', '', is_ssl(), false );

				return $items;
			}
		}

		// Then check cookie
		if ( isset( $_COOKIE['rh_crm_items_per_page'] ) ) {
			$items = intval( $_COOKIE['rh_crm_items_per_page'] );
			// Validate against allowed values
			if ( in_array( $items, $allowed_values ) ) {
				return $items;
			}
		}

		// Return default
		return $default;
	}

	public static function items_per_page_selector( $items_per_page ) {
		?>
		<div class="flex items-center">
			<div class="grid grid-cols-1">
				<select id="rh-crm-items-per-page" name="number" autocomplete="number" class="min-w-[120px] col-start-1 row-start-1 w-full appearance-none rounded-md bg-white !py-2 pl-3 pr-8 text-base text-gray-900 outline outline-1 -outline-offset-1 outline-gray-300 focus:outline focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6 !shadow-sm">
					<?php
					foreach ( Realhomes_Crm_Helper::get_setting( 'posts_per_page_list' ) as $item_number ) {
						echo '<option value="' . esc_attr( $item_number ) . '" ' . selected( $item_number, $items_per_page ) . '>' . $item_number . esc_html__( ' / page', REALHOMES_CRM_TEXT_DOMAIN ) . '</option>';
					}
					?>
				</select>
				<?php Realhomes_Crm_SVG::the_svg( 'chevron', 'size-5' ); ?>
			</div>
		</div>
		<?php
	}

	public static function listing_footer( $offset, $current_page, $items_per_page, $total_items, $page_type ) {
		?>
		<div id="rh-crm-<?php echo Realhomes_Crm_Helper::get_page_type_plurals()[ $page_type ]; ?>-footer" class="flex items-center justify-between border-t border-gray-200 bg-white px-4 py-6 sm:px-6">
			<?php
			Realhomes_Crm_Helper::showing_items_info( $offset, $items_per_page, $total_items );
			Realhomes_Crm_Helper::items_pagination( $current_page, $items_per_page, $total_items );
			Realhomes_Crm_Helper::items_per_page_selector( $items_per_page );
			?>

		</div>
		<?php
	}

	public static function items_select_checkbox() {
		?>
		<div class="group grid size-7 grid-cols-1">
			<input id="rh-crm-item-select-all" type="checkbox" class="cursor-pointer col-start-1 row-start-1 appearance-none rounded border border-gray-300 bg-white checked:border-[var(--dashboard-primary-color)] checked:bg-[var(--dashboard-primary-color)] focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-[var(--dashboard-primary-color)]">
			<?php Realhomes_Crm_SVG::the_svg( 'checkbox_tick', 'size-6' ); ?>
		</div>
		<?php
	}

	public static function item_select_checkbox( $page_type, $item_id ) {
		?>
		<div class="group grid size-7 grid-cols-1">
			<input id="<?php echo esc_attr( $page_type ); ?>-<?php echo esc_attr( $item_id ); ?>" name="rh-crm-item-select[]" type="checkbox" value="<?php echo esc_attr( $item_id ); ?>" class="cursor-pointer col-start-1 row-start-1 appearance-none rounded border border-gray-300 bg-white checked:border-[var(--dashboard-primary-color)] checked:bg-[var(--dashboard-primary-color)] focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-[var(--dashboard-primary-color)]">
			<?php Realhomes_Crm_SVG::the_svg( 'checkbox_tick', 'size-6' ); ?>
		</div>
		<?php
	}

	public static function selected_items_action_buttons( $page_type ) {
		?>
		<div id="rh-crm-<?php echo Realhomes_Crm_Helper::get_page_type_plurals()[ $page_type ]; ?>-actions" class="w-full relative">
			<div id="rh-crm-selected-items-actions-warp" class="hidden fixed bottom-16 flex items-center z-10 px-8 py-4 bg-white border border-slate-200 shadow-xl rounded-lg text-gray-900">
				<div class="border-r border-slate-200 mr-4 pr-4">
					<span id="rh-crm-selected-items-counter" class="inline-block text-center p-2 bg-slate-100 rounded-full text-xs min-w-[24px] mr-2">0</span> <?php esc_html_e( 'selected', REALHOMES_CRM_TEXT_DOMAIN ); ?>
				</div>
				<button data-type="<?php echo esc_attr( $page_type ); ?>" id="rh-crm-delete-selected-items" class="flex items-center text-red-600 hover:text-red-800 rounded">
					<?php
					Realhomes_Crm_SVG::the_svg( 'delete', '-ml-0.5 size-5 mr-1' );
					esc_html_e( 'Delete', REALHOMES_CRM_TEXT_DOMAIN );
					?>
				</button>
			</div>
		</div>
		<?php
	}

	public static function prepare_listing_page_args( $page_type ) {
		$methods = [
			'activity' => [ Realhomes_Crm_Activities::class, 'fetch_activities' ],
			'inquiry'  => [ Realhomes_Crm_Inquiries::class, 'fetch_inquiries' ],
			'lead'     => [ Realhomes_Crm_Leads::class, 'fetch_leads' ],
			'deal'     => [ Realhomes_Crm_Deals::class, 'fetch_deals' ],
			'task'     => [ Realhomes_Crm_Tasks::class, 'fetch_tasks' ],
		];

		$posts_per_page_list = Realhomes_Crm_Helper::get_setting( 'posts_per_page_list' );

		$args                   = [];
		$args['items_per_page'] = Realhomes_Crm_Helper::get_items_per_page( $posts_per_page_list[0] );
		$args['current_page']   = isset( $_GET['itemsPage'] ) ? intval( $_GET['itemsPage'] ) : 1;
		$args['offset']         = ( $args['current_page'] - 1 ) * $args['items_per_page'];
		$args['searched_word']  = isset( $_GET['posts_search'] ) ? sanitize_text_field( $_GET['posts_search'] ) : '';
		$args['total_items']    = count( call_user_func( $methods[ $page_type ], array( 'number' => 2000, 'keyword' => $args['searched_word'] ), array( 'id' ) ) );

		return $args;
	}

	public static function listing_search_form( $searched_keyword, $page_type = 'item', $items_total = 0 ) {
		?>
		<div class="w-full flex">
			<form id="dashboard-search-form" class="relative sm:min-w-[350px] w-full max-w-full md:max-w-sm !mb-0" method="GET" action="<?php echo esc_url( $_SERVER['REQUEST_URI'] ); ?>">
				<input id="dashboard-search-form-input" class="w-full bg-transparent placeholder:text-slate-400 text-slate-700 text-sm border border-slate-200 rounded-md pl-6 pr-28 py-4 transition duration-300 ease focus:outline-none focus:border-slate-400 hover:border-slate-300 shadow-sm focus:shadow" placeholder="<?php esc_html_e( "Start searching {$page_type}s...",
					REALHOMES_CRM_TEXT_DOMAIN ); ?>" name="posts_search" />
				<button id="dashboard-search-form-submit-button" class="rh-crm-btn-primary absolute top-2 right-2 flex items-center py-3 px-5 max-w-[130px] border border-transparent" type="submit">
					<?php
					Realhomes_Crm_SVG::the_svg( 'search', 'size-5 mr-2' );
					esc_html_e( 'Search', REALHOMES_CRM_TEXT_DOMAIN );
					?>
				</button>
			</form>
			<?php
			if ( $searched_keyword ) {
				?>
				<div class="flex items-center w-full max-w-xs px-3 py-2.5 text-gray-500 bg-white rounded-lg shadow ml-6" role="alert">
					<div class="inline-flex items-center justify-center flex-shrink-0 w-8 h-8 text-blue-500 bg-blue-100 rounded-lg">
						<?php Realhomes_Crm_SVG::the_svg( 'search', 'size-4' ); ?>
					</div>
					<div class="ms-3 text-sm font-normal"><?php echo esc_attr( $searched_keyword ); ?></div>
					<a href="<?php echo esc_url( preg_replace( '/(&|\?)posts_search=[^&]*/',
						'',
						$_SERVER['REQUEST_URI'] ) ); ?>" class="ms-auto -mx-1.5 -my-1.5 bg-white text-gray-400 hover:text-gray-900 rounded-lg focus:ring-2 focus:ring-gray-300 p-1.5 hover:bg-gray-100 inline-flex items-center justify-center h-8 w-8" data-dismiss-target="#toast-default" aria-label="Close">
						<?php Realhomes_Crm_SVG::the_svg( 'close', 'size-5 text-gray-500' ); ?>
					</a>
				</div>
				<?php
			}
			?>
		</div>
		<?php
	}

	public static function listing_header( $page_type, $searched_keyword = '', $buttons = [ 'export', 'import', 'add' ] ) {
		$import_pages = Realhomes_Crm_Helper::get_page_type_plurals();
		?>
		<div id="rh-crm-<?php echo Realhomes_Crm_Helper::get_page_type_plurals()[ $page_type ]; ?>-header" class="flex flex-col md:flex-row items-center justify-between border-b border-gray-200 bg-white px-4 py-6 sm:px-6">
			<?php
			Realhomes_Crm_Helper::listing_search_form( $searched_keyword, $page_type );
			?>
			<div class="flex justify-between w-full max-w-xl pt-6 md:pt-0">
				<?php
				if ( in_array( 'export', $buttons ) ) {
					?>
					<button id="rh-crm-export-items" data-type="<?php echo esc_attr( $import_pages[ $page_type ] ); ?>" class="rh-crm-btn-primary">
						<?php
						Realhomes_Crm_SVG::the_svg( 'loader', 'size-5 mr-2 hidden' );
						Realhomes_Crm_SVG::the_svg( 'export', 'size-5 mr-2' );
						esc_html_e( 'Export', REALHOMES_CRM_TEXT_DOMAIN );
						?>
					</button>
					<?php
				}

				if ( in_array( 'import', $buttons ) ) {
					?>
					<a href="<?php echo esc_url( realhomes_get_dashboard_page_url( 'properties-crm' ) . '&submodule=crm/import&import=' . $import_pages[ $page_type ] ) ?>" id="rh-crm-import-items" data-type="<?php echo esc_attr( $page_type ); ?>" class="rh-crm-btn-primary">
						<?php
						Realhomes_Crm_SVG::the_svg( 'import', 'fill-white size-5 mr-2' );
						esc_html_e( 'Import', REALHOMES_CRM_TEXT_DOMAIN );
						?>
					</a>
					<?php
				}

				if ( in_array( 'add', $buttons ) ) {
					?>
					<button data-type="<?php echo esc_attr( $page_type ); ?>" data-action="add" data-id="" class="rh-crm-add-edit-item rh-crm-btn-primary">
						<?php
						Realhomes_Crm_SVG::the_svg( 'loader', 'size-5 mr-2 hidden' );
						Realhomes_Crm_SVG::the_svg( 'plus', 'size-5 mr-2' );
						esc_html_e( 'Add New', REALHOMES_CRM_TEXT_DOMAIN );
						?>
					</button>
					<?php
				}
				?>
			</div>

		</div>
		<?php
	}

	public static function get_import_type() {
		$default_type = 'leads';
		$import_type  = isset( $_GET['import'] ) ? sanitize_text_field( $_GET['import'] ) : $default_type;

		$importable = array( 'leads', 'inquiries', 'deals', 'tasks' );

		return in_array( $import_type, $importable ) ? $import_type : $default_type;
	}
}