<?php
/**
 * The file that defines all the actions for the deals management.
 *
 * @link       https://realhomes.io/
 * @since      1.0.0
 *
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 */

/**
 * The deals management class.
 *
 * This is used to define all the actions and data management of deals.
 *
 * @since      1.0.0
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 * @author     InspiryThemes <fahid@inspirythemes.com>
 */
class Realhomes_Crm_Deals {

	/**
	 * Insert a new custom deal into the database.
	 *
	 * @param $data
	 *
	 * @return int
	 */
	public static function insert_custom_deal( $data ) {
		global $wpdb;

		// Define the table name
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_deals;

		// Prepare deal data to be inserted, excluding created and updated fields
		$deal_data = array(
			'title'           => isset( $data['title'] ) ? sanitize_text_field( $data['title'] ) : '',
			'description'     => isset( $data['description'] ) ? sanitize_textarea_field( $data['description'] ) : '',
			'lead_id'         => isset( $data['lead-id'] ) ? intval( $data['lead-id'] ) : '',
			'agent_id'        => isset( $data['agent-id'] ) ? intval( $data['agent-id'] ) : '',
			'deal_stage'      => isset( $data['deal-stage'] ) ? sanitize_text_field( $data['deal-stage'] ) : '',
			'probability'     => isset( $data['probability'] ) ? sanitize_text_field( $data['probability'] ) : '',
			'last_step'       => isset( $data['last-step'] ) ? sanitize_text_field( $data['last-step'] ) : '',
			'last_step_date'  => isset( $data['last-step-date'] ) ? sanitize_text_field( $data['last-step-date'] ) : '',
			'next_step'       => isset( $data['next-step'] ) ? sanitize_text_field( $data['next-step'] ) : '',
			'next_step_date'  => isset( $data['next-step-date'] ) ? sanitize_text_field( $data['next-step-date'] ) : '',
			'estimated_value' => isset( $data['estimated-value'] ) ? sanitize_text_field( $data['estimated-value'] ) : '',
		);

		// Insert the data into the database
		$result = $wpdb->insert(
			$table_name,
			$deal_data,
			array(
				'%s',  // title
				'%s',  // description
				'%d',  // lead_id
				'%d',  // agent_id
				'%s',  // deal_stage
				'%s',  // probability
				'%s',  // last_step
				'%s',  // last_step_date
				'%s',  // next_step
				'%s',  // next_step_date
				'%d',  // estimated_value
			)
		);

		if ( $result === false ) {
			// Handle the error if insert failed
			error_log( 'Database insert deal failed: ' . $wpdb->last_error );
		}

		return $wpdb->insert_id;
	}

	/**
	 * Update an existing deal in the database.
	 *
	 * @param $data
	 *
	 * @return bool|int|mixed|mysqli_result|null
	 */
	public static function update_deal( $data ) {
		global $wpdb;

		// Define the table name
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_deals;

		// Prepare sanitized deal data
		$deal_data = array(
			'title'           => sanitize_text_field( $data['title'] ),
			'description'     => sanitize_textarea_field( $data['description'] ),
			'lead_id'         => intval( $data['lead-id'] ),
			'agent_id'        => intval( $data['agent-id'] ),
			'last_step'       => sanitize_text_field( $data['last-step'] ),
			'last_step_date'  => sanitize_text_field( $data['last-step-date'] ),
			'next_step'       => sanitize_text_field( $data['next-step'] ),
			'next_step_date'  => sanitize_text_field( $data['next-step-date'] ),
			'deal_stage'      => sanitize_text_field( $data['deal-stage'] ),
			'probability'     => sanitize_text_field( $data['probability'] ),
			'estimated_value' => sanitize_text_field( $data['estimated-value'] ),
		);

		// Specify the format for each field
		$format = array(
			'%s',  // title
			'%s',  // description
			'%d',  // lead_id
			'%d',  // agent_id
			'%s',  // last_step
			'%s',  // last_step_date
			'%s',  // next_step
			'%s',  // next_step_date
			'%s',  // deal_stage
			'%s',  // probability
			'%d',  // estimated_value
		);

		// Update the deal in the database
		$result = $wpdb->update(
			$table_name,
			$deal_data,
			array( 'id' => intval( $data['item_id'] ) ),
			$format,
			array( '%d' )
		);

		if ( $result === false ) {
			// Handle the error if update failed
			error_log( 'Database update deal failed: ' . $wpdb->last_error );
		}

		return $result;
	}

	/**
	 * Delete a single deal based on the given deal ID.
	 *
	 * @param int $deal_id The ID of the deal to delete.
	 *
	 * @return void
	 */
	public static function delete_deal( $deal_id ) {
		global $wpdb;

		// Ensure the deal ID is valid and not empty
		if ( ! empty( $deal_id ) && intval( $deal_id ) > 0 ) {
			// Table name
			$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_deals;

			// Prepare and execute delete query
			$result = $wpdb->delete( $table_name, array( 'id' => intval( $deal_id ) ) );

			if ( $result === false ) {
				// Log error if delete operation failed
				error_log( 'Deal delete failed: ' . $wpdb->last_error );
			}

			return $result;
		}
	}


	/**
	 * Retrieves a single deal from the database based on the provided deal ID.
	 *
	 * @param int $deal_id The ID of the deal to retrieve.
	 *
	 * @return object|null The deal object if found, or null if not found.
	 */
	public static function fetch_deal( $deal_id ) {
		global $wpdb;

		// Table name for deals
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_deals;

		// Prepare and execute the query to retrieve the deal
		$query = $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d", $deal_id );
		$deal  = $wpdb->get_row( $query );

		return $deal;
	}

	/**
	 * Retrieves a list of deals from the database with optional filters and pagination.
	 *
	 * Allows filtering by keyword, lead ID, deal stage, agent ID, and sorting by column,
	 * as well as setting a limit and offset for pagination.
	 *
	 * @param array $args Optional. Arguments to filter and paginate deals.
	 *
	 * @return array|object|null List of deals or null if none found.
	 * @global wpdb $wpdb WordPress database object.
	 */
	public static function fetch_deals( array $args = array() ) {
		global $wpdb;

		// Table name for deals & leads
		$deals_table = $wpdb->prefix . Realhomes_Crm_Helper::$table_deals; // Adjust table name if needed
		$leads_table = $wpdb->prefix . Realhomes_Crm_Helper::$table_leads; // Adjust table name if needed

		// Default query arguments
		$default_args = array(
			'number'  => 10,     // Number of results to return
			'offset'  => 0,      // Starting point for pagination
			'orderby' => 'created_at',  // Column to sort by
			'order'   => 'DESC'  // Order direction (ASC or DESC)
		);

		// Merge passed arguments with defaults
		$args = wp_parse_args( $args, $default_args );

		// Start building the query
		$query = "SELECT * FROM $deals_table WHERE 1=1";

		if ( ! empty( $args['keyword'] ) ) {
			$keyword = '%' . $wpdb->esc_like( $args['keyword'] ) . '%';

			// Add JOIN to leads table
			$query = "SELECT d.* FROM {$deals_table} d
              LEFT JOIN {$leads_table} l ON d.lead_id = l.id 
              WHERE 1=1";

			$query .= $wpdb->prepare(
				" AND (d.id LIKE %s OR d.title LIKE %s OR d.description LIKE %s OR d.last_step LIKE %s OR d.next_step LIKE %s OR d.deal_stage LIKE %s OR d.probability LIKE %s OR l.display_name LIKE %s)",
				$keyword,
				$keyword,
				$keyword,
				$keyword,
				$keyword,
				$keyword,
				$keyword,
				$keyword
			);
		}


		// Filter by lead ID
		if ( ! empty( $args['lead_id'] ) ) {
			$query .= $wpdb->prepare( " AND lead_id = %d", $args['lead_id'] );
		}

		// Filter by agent ID
		if ( ! empty( $args['agent_id'] ) ) {
			$query .= $wpdb->prepare( " AND agent_id = %d", $args['agent_id'] );
		}

		// Filter by deal stage
		if ( ! empty( $args['deal_stage'] ) ) {
			$query .= $wpdb->prepare( " AND deal_stage = %s", $args['deal_stage'] );
		}

		// Add ordering and limit to the query
		$query .= $wpdb->prepare( " ORDER BY {$args['orderby']} {$args['order']} LIMIT %d OFFSET %d", $args['number'], $args['offset'] );

		// Execute the query and get the results
		$result = $wpdb->get_results( $query );

		if ( $result === false ) {
			// Handle the error if fetch failed
			error_log( 'Database fetch deals failed: ' . $wpdb->last_error );
		}

		return $result;
	}

	/**
	 * Handle exporting deals as a CSV file.
	 *
	 * @return void
	 */
	public static function export_deals() {
		// Retrieve all deals data
		$deals = self::fetch_deals();

		// Set CSV headers
		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=deals' . '.csv' );

		// Open PHP output stream as the file
		$output = fopen( 'php://output', 'w' );

		// Define CSV column headers
		fputcsv( $output, array( 'ID', 'Title', 'Description', 'Contact', 'Agent', 'Deal Stage', 'Probability', 'Last Step', 'Last Step Date', 'Next Step', 'Next Step Date', 'Estimated Value' ) );

		// Add each deal's data as a new row in the CSV
		foreach ( $deals as $deal ) {
			$lead_name = $agent_name = '';

			// Prepare lead name
			if ( $deal->lead_id ) {
				$lead = Realhomes_Crm_Leads::fetch_lead( $deal->lead_id );

				if ( $lead ) {
					$lead_name = $lead->first_name . ' ' . $lead->last_name;
				}
			}

			// Prepare agent name
			if ( $deal->agent_id ) {
				$agent_name = get_the_title( $deal->agent_id );
			}

			fputcsv( $output, array(
				$deal->id,
				$deal->title,
				$deal->description,
				$lead_name,
				$agent_name,
				$deal->deal_stage,
				$deal->probability,
				$deal->last_step,
				$deal->last_step_date,
				$deal->next_step,
				$deal->next_step_date,
				$deal->estimated_value,
			) );
		}

		// Close the output stream
		fclose( $output );

		exit;
	}

	/**
	 * Import deals from the uploaded CSV file.
	 *
	 * @param $file_handle
	 * @param $headers
	 * @param $fields_mapping
	 *
	 * @return int[]
	 */
	public static function import_deals( $file_handle, $headers, $fields_mapping ) {
		// Array to hold import results
		$import_results = [
			'inserted'   => 0,
			'duplicates' => 0,
			'errors'     => 0
		];


		// Loop through each row
		while ( ( $csv_data = fgetcsv( $file_handle ) ) !== false ) {
			$deal_data = [];
			foreach ( $fields_mapping as $field_mapping => $csv_field ) {
				$index                       = array_search( $csv_field, $headers );
				$deal_data[ $field_mapping ] = $index !== false ? $csv_data[ $index ] : '';
			}
			// Check for duplicates based on title
			global $wpdb;
			$existing_deal = $wpdb->get_var( $wpdb->prepare(
				"SELECT COUNT(*) FROM " . $wpdb->prefix . Realhomes_Crm_Helper::$table_deals . " WHERE title = %s",
				$deal_data['title']
			) );

			if ( $existing_deal ) {
				$import_results['duplicates'] ++;
				continue; // Skip duplicate entries
			}

			// Insert deal into the database
			$result = $wpdb->insert(
				$wpdb->prefix . Realhomes_Crm_Helper::$table_deals, // Deals table name
				[
					'title'           => sanitize_text_field( $deal_data['title'] ),
					'description'     => sanitize_textarea_field( $deal_data['description'] ),
					'deal_stage'      => sanitize_text_field( $deal_data['deal_stage'] ),
					'probability'     => sanitize_text_field( $deal_data['probability'] ),
					'last_step'       => sanitize_text_field( $deal_data['last_step'] ),
					'last_step_date'  => sanitize_text_field( $deal_data['last_step_date'] ),
					'next_step'       => sanitize_text_field( $deal_data['next_step'] ),
					'next_step_date'  => sanitize_text_field( $deal_data['next_step_date'] ),
					'estimated_value' => floatval( $deal_data['estimated_value'] ),
				],
				[ '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s' ]
			);

			// Track import results
			if ( $result ) {
				$import_results['inserted'] ++;
			} else {
				$import_results['errors'] ++;
			}
		}

		return $import_results;
	}
}