<?php
/**
 * The file that defines all the AJAX handlers for the RealHomes CRM.
 *
 * @link       https://realhomes.io/
 * @since      1.0.0
 *
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 */

/**
 * The AJAX class for RealHomes CRM.
 *
 * This class is used to define all the AJAX handlers and associated logic
 * for the RealHomes CRM module.
 *
 * @since      1.0.0
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 * @author     InspiryThemes <fahid@inspirythemes.com>
 */
class Realhomes_Crm_Ajax {

	/**
	 * Class instance variable to create an instance of class
	 *
	 * @var null
	 */
	private static $instance = null;

	/**
	 * Create a single instance of the class to support all ajax requests
	 *
	 * @return self|null
	 */
	public static function get_instance(): ?Realhomes_Crm_Ajax {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Initialize AJAX handlers.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Get markup for item preview, add or edit templates
		add_action( 'wp_ajax_realhomes_crm_get_item_markup', [ $this, 'get_item_markup' ] );
		add_action( 'wp_ajax_realhomes_crm_add_edit_item_markup', [ $this, 'get_add_edit_item_markup' ] );

		// An item add, update, delete or adding note to an item
		add_action( 'wp_ajax_realhomes_crm_add_update_item', [ $this, 'add_update_item' ] );
		add_action( 'wp_ajax_realhomes_crm_delete_item', [ $this, 'delete_item' ] );
		add_action( 'wp_ajax_realhomes_crm_delete_items', [ $this, 'delete_items' ] );
		add_action( 'wp_ajax_realhomes_crm_add_item_note', [ $this, 'add_item_note' ] );

		// Send matched listings to an inquiry via email to the attached lead
		add_action( 'wp_ajax_realhomes_crm_send_matched_listings', [ $this, 'send_inquiry_matched_listings' ] );

		// Export & import
		add_action( 'wp_ajax_realhomes_crm_export_items', [ $this, 'export_items_csv_file' ] );
		add_action( 'wp_ajax_realhomes_crm_upload_csv_file', [ $this, 'upload_csv_file' ] );
		add_action( 'wp_ajax_realhomes_crm_delete_csv_file', [ $this, 'delete_csv_file' ] );
		add_action( 'wp_ajax_realhomes_crm_import_csv_file', [ $this, 'import_csv_file' ] );

		// ChartJS data
		add_action( 'wp_ajax_realhomes_crm_get_leads_stats_data', [ $this, 'get_leads_stats_data' ] );
		add_action( 'wp_ajax_realhomes_crm_get_deals_stats_data', [ $this, 'get_deals_stats_data' ] );
	}

	/**
	 * Get an item preview markup to display in the drawer
	 *
	 * @return void
	 */
	public function get_item_markup() {
		$data        = $_POST['data'];
		$item_id     = $data['item_id'];
		$item_type   = $data['item_type'];
		$item_action = $data['item_action'];

		$template = Realhomes_Crm_Helper::get_page_type_plurals();

		ob_start();
		get_template_part( "common/dashboard/crm/{$template[$item_type]}/detail", null, array( 'item_id' => $item_id, 'item_type' => $item_type, 'item_action' => $item_action ) );
		$item_markup = ob_get_clean();
		ob_end_clean();


		wp_send_json_success( [ 'message' => 'received request for markup', 'item_markup' => $item_markup ] );
	}

	/**
	 * get an item add/edit markup to use in the drawer
	 *
	 * @return void
	 */
	public function get_add_edit_item_markup() {
		$data        = $_POST['data'];
		$item_id     = $data['item_id'];
		$item_type   = $data['item_type'];
		$item_action = $data['item_action'];

		$template = Realhomes_Crm_Helper::get_page_type_plurals();

		ob_start();
		get_template_part( "common/dashboard/crm/{$template[$item_type]}/add-edit", null, array( 'item_id' => $item_id, 'item_type' => $item_type, 'item_action' => $item_action ) );
		$item_markup = ob_get_clean();
		ob_end_clean();

		wp_send_json_success( [ 'message' => 'received request for markup', 'item_markup' => $item_markup ] );
	}

	/**
	 * Add or update an item in the database
	 *
	 * @return void
	 */
	public function add_update_item() {
		// Verify nonce for security
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		// Get the note data from the AJAX request
		parse_str( $_POST['data'], $data );

		$data          = array_map( 'stripslashes', $data ); // Fix escaped quotes
		$item_type     = $data['item_type'] ?? '';
		$action_type   = $data['action_type'] ?? '';
		$callable_func = 'update' === $action_type ? 'update' : 'insert_custom';

		// Define mapping of item types to their corresponding to update or add methods
		$methods = [
			'activity' => [ Realhomes_Crm_Activities::class, $callable_func . '_activity' ],
			'inquiry'  => [ Realhomes_Crm_Inquiries::class, $callable_func . '_inquiry' ],
			'lead'     => [ Realhomes_Crm_Leads::class, $callable_func . '_lead' ],
			'deal'     => [ Realhomes_Crm_Deals::class, $callable_func . '_deal' ],
			'task'     => [ Realhomes_Crm_Tasks::class, $callable_func . '_task' ],
		];

		// Check if the item type is supported and has a corresponding to delete method
		if ( ! isset( $methods[ $item_type ] ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Unsupported item type.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		// Call the corresponding to update method
		$call_method = $methods[ $item_type ];
		if ( is_callable( $call_method ) && call_user_func( $call_method, $data ) ) {
			wp_send_json_success( [ 'message' => esc_html__( ucfirst( $item_type ) . ' ' . $action_type . ' successful.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		} else {
			wp_send_json_error( [ 'message' => esc_html__( 'Failed to ' . $action_type . ' the ' . $item_type . '. Please try again.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}
	}

	/**
	 * Delete an item from database based on given ID and item type
	 *
	 * @return void
	 */
	function delete_item() {
		// Verify nonce for security
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		// Get the item type and ID from the AJAX request
		$data      = $_POST['data'];
		$item_type = $data['item_type'] ?? '';
		$item_id   = isset( $data['item_id'] ) ? intval( $data['item_id'] ) : 0;

		// Check for valid input
		if ( ! $item_type || ! $item_id ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Invalid item type or ID to delete an item.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		// Define mapping of item types to their corresponding to delete methods
		$delete_methods = [
			'activity' => [ Realhomes_Crm_Activities::class, 'delete_activity' ],
			'inquiry'  => [ Realhomes_Crm_Inquiries::class, 'delete_inquiry' ],
			'lead'     => [ Realhomes_Crm_Leads::class, 'delete_lead' ],
			'deal'     => [ Realhomes_Crm_Deals::class, 'delete_deal' ],
			'task'     => [ Realhomes_Crm_Tasks::class, 'delete_task' ],
			'note'     => [ Realhomes_Crm_Notes::class, 'delete_note' ],
		];

		// Check if the item type is supported and has a corresponding to delete method
		if ( ! isset( $delete_methods[ $item_type ] ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Unsupported item type.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		// Call the corresponding to delete method
		$delete_method = $delete_methods[ $item_type ];
		if ( is_callable( $delete_method ) && call_user_func( $delete_method, $item_id ) ) {
			wp_send_json_success( [ 'message' => esc_html__( ucfirst( $item_type ) . ' deleted successfully.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		} else {
			wp_send_json_error( [ 'message' => esc_html__( 'Failed to delete the ' . $item_type . '. Please try again.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}
	}

	function delete_items() {
		// Verify nonce for security
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		// Get the item type and IDs from the AJAX request
		$data      = $_POST['data'];
		$item_type = $data['item_type'] ?? '';
		$item_ids  = isset( $data['item_ids'] ) && is_array( $data['item_ids'] ) ? array_map( 'intval', $data['item_ids'] ) : [];

		// Check for valid input
		if ( ! $item_type || empty( $item_ids ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Invalid item type or IDs to delete items.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		// Define mapping of item types to their corresponding to delete methods
		$delete_methods = [
			'activity' => [ Realhomes_Crm_Activities::class, 'delete_activity' ],
			'inquiry'  => [ Realhomes_Crm_Inquiries::class, 'delete_inquiry' ],
			'lead'     => [ Realhomes_Crm_Leads::class, 'delete_lead' ],
			'deal'     => [ Realhomes_Crm_Deals::class, 'delete_deal' ],
			'task'     => [ Realhomes_Crm_Tasks::class, 'delete_task' ],
			'note'     => [ Realhomes_Crm_Notes::class, 'delete_note' ],
		];

		// Check if the item type is supported and has a corresponding to delete method
		if ( ! isset( $delete_methods[ $item_type ] ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Unsupported item type.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		// Call the corresponding to delete method
		$delete_method = $delete_methods[ $item_type ];
		$errors        = 0;

		if ( is_callable( $delete_method ) ) {
			foreach ( $item_ids as $item_id ) {
				if ( ! call_user_func( $delete_method, $item_id ) ) {
					$errors ++;
				}
			}

			// Respond with success or partial success
			if ( $errors === 0 ) {
				wp_send_json_success( [ 'message' => esc_html__( ucfirst( $item_type ) . 's deleted successfully.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
			} elseif ( $errors < count( $item_ids ) ) {
				wp_send_json_success( [ 'message' => esc_html__( 'Some ' . $item_type . 's could not be deleted. Please try again.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
			} else {
				wp_send_json_error( [ 'message' => esc_html__( 'Failed to delete all ' . $item_type . 's. Please try again.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
			}
		} else {
			wp_send_json_error( [ 'message' => esc_html__( 'Failed to process the delete request.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}
	}


	/**
	 * Add an item note in the database with related item ID and type
	 *
	 * @return void
	 */
	function add_item_note() {
		// Verify nonce for security
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		// Get the note data from the AJAX request
		parse_str( $_POST['data'], $data );
		$data = array_map( 'stripslashes', $data ); // Fix escaped quotes

		if ( empty( $data['note'] ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Please type a note.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		} elseif ( empty( $data['related_item_id'] ) || empty( $data['type'] ) || empty( $data['user_id'] ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Incorrect data supplied.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		// Insert the note to the database
		$result = Realhomes_Crm_Notes::insert_note( $data );
		if ( $result['success'] ) {
			$note_id     = $result['insert_id'];
			$note_markup = Realhomes_Crm_Notes::fetch_note_ajax( $note_id );
			wp_send_json_success( [ 'note_id' => $note_id, 'note_markup' => $note_markup, 'message' => ucfirst( $data['type'] ) . ' ' . esc_html__( 'note is added.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		wp_send_json_error( [ 'message' => esc_html__( 'Failed to add the note.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
	}

	/**
	 * Send an inquiry matched listing to the attached lead email
	 *
	 * @return void
	 */
	public function send_inquiry_matched_listings(): void {
		// Verify nonce for security
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		parse_str( $_POST['data'], $data );
		$data = array_map( 'stripslashes', $data ); // Fix escaped quotes

		if ( ! empty( $data['lead_email'] ) ) {
			$lead_email    = sanitize_email( $data['lead_email'] );
			$inquiry_query = json_decode( $data['inquiry_query'], true );

			// Proceed only if a valid email is provided
			if ( is_email( $lead_email ) ) {
				// Apply the filter to get the matched properties
				$query_args         = apply_filters( 'realhomes_crm_match_listing_args', $inquiry_query );
				$matched_properties = new WP_Query( $query_args );

				if ( $matched_properties->have_posts() ) {
					// Start building the email content
					$email_body = [];

					while ( $matched_properties->have_posts() ) {
						$matched_properties->the_post();

						$email_body[] = array(
							'name'  => get_the_title(),
							'value' => get_permalink(),
						);
					}

					// Reset post data after custom query
					wp_reset_postdata();

					// Email subject
					$email_subject = esc_html__( 'Checkout these properties matching your inquiry', 'easy-real-estate' );

					// Start of email body
					$headers = array( 'Content-Type: text/html; charset=UTF-8' );

					// Generate email body using a template
					$email_body = ere_email_template( $email_body, 'rh-crm-matched-listing' );

					// Send the email
					if ( wp_mail( $lead_email, $email_subject, $email_body, $headers ) ) {
						wp_send_json_success( [ 'message' => esc_html__( 'Matched listing sent successfully.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
					} else {
						wp_send_json_error( [ 'message' => esc_html__( 'Something went wrong.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
					}
				}
			} else {
				wp_send_json_error( [ 'message' => esc_html__( 'Incorrect lead email.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
			}
		}

		wp_send_json_error( [ 'message' => esc_html__( 'Lead email is empty.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
	}

	public function export_items_csv_file() {
		$data        = $_POST['data'];
		$export_type = $data['export_type'];

		// Define mapping of item types to their corresponding to export methods
		$export_methods = [
			'inquiries' => [ Realhomes_Crm_Inquiries::class, 'export_inquiries' ],
			'leads'     => [ Realhomes_Crm_Leads::class, 'export_leads' ],
			'deals'     => [ Realhomes_Crm_Deals::class, 'export_deals' ],
			'tasks'     => [ Realhomes_Crm_Tasks::class, 'export_tasks' ],
		];

		// Check if the export type is supported and has a corresponding to export method
		if ( ! isset( $export_methods[ $export_type ] ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Unsupported export type.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		// Call the corresponding to export method
		$export_method = $export_methods[ $export_type ];

		if ( ! is_callable( $export_method ) || ! call_user_func( $export_method, $export_type ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Failed to export the ' . $export_type . 's. Please try again.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}
	}

	public function upload_csv_file() {
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		// Check user capabilities - require at least 'edit_posts' permission
//		if ( ! current_user_can( 'edit_posts' ) ) {
//			wp_send_json_error( [ 'message' => esc_html__( 'You do not have permission to upload files.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
//		}

		// Validate the uploaded file
		if ( empty( $_FILES['csv_file'] ) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Invalid file upload.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}


		// Get the uploaded file info
		$file = $_FILES['csv_file'];

		// Validate file type using WordPress function
		$file_type = wp_check_filetype( $file['name'], [ 'csv' => 'text/csv' ] );

		if ( $file_type['ext'] !== 'csv' || $file_type['type'] !== 'text/csv' ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Invalid file type. Only CSV files are allowed.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		// Additional MIME type validation
		$file_info = finfo_open( FILEINFO_MIME_TYPE );
		$mime_type = finfo_file( $file_info, $file['tmp_name'] );
		finfo_close( $file_info );

		$allowed_mime_types = [ 'text/csv', 'text/plain', 'application/csv', 'application/vnd.ms-excel' ];
		if ( ! in_array( $mime_type, $allowed_mime_types ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Invalid file content. Only CSV files are allowed.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		// Sanitize filename to prevent directory traversal**
		$valid_file_name = sanitize_file_name( $file['name'] );
		$valid_file_name = preg_replace( '/[^a-zA-Z0-9._-]/', '', $valid_file_name );

		// Get the uploaded file info
		$upload_dir = wp_upload_dir();
		$target_dir = trailingslashit( $upload_dir['basedir'] ) . 'realhomes-crm/csv-import/';

		// Ensure target directory exists
		if ( ! file_exists( $target_dir ) ) {
			wp_mkdir_p( $target_dir );
		}

		$file_name   = pathinfo( $valid_file_name, PATHINFO_FILENAME );
		$file_ext    = pathinfo( $valid_file_name, PATHINFO_EXTENSION );
		$target_file = $target_dir . $valid_file_name;

		// Check if the file already exists and append a numeric postfix if needed
		$counter = 1;
		while ( file_exists( $target_file ) ) {
			$target_file = $target_dir . $file_name . '-' . $counter . '.' . $file_ext;
			$counter ++;
		}

		// Move the uploaded file to the target directory
		if ( move_uploaded_file( $file['tmp_name'], $target_file ) ) {
			$file_name     = basename( $target_file );
			$file_size     = file_exists( $target_file ) ? round( filesize( $target_file ) / 1024, 0 ) : 0; // File size in KB
			$uploaded_date = current_time( 'mysql' );

			// Save file information to the current user's meta
			$current_user_id = get_current_user_id();
			if ( $current_user_id ) {
				$user_files = get_user_meta( $current_user_id, 'rh_crm_uploaded_csv_files', true );

				if ( ! is_array( $user_files ) ) {
					$user_files = [];
				}

				// Add the new file information
				$user_files[] = [
					'file_name'     => $file_name,
					'file_size'     => $file_size,
					'uploaded_date' => $uploaded_date,
					'imported_date' => null, // Initialize as null for now
				];

				// Save back to user meta
				update_user_meta( $current_user_id, 'rh_crm_uploaded_csv_files', $user_files );
			}

			wp_send_json_success( [
				'message'       => esc_html__( 'File uploaded successfully.', REALHOMES_CRM_TEXT_DOMAIN ),
				'file_name'     => $file_name,
				'file_size'     => $file_size,
				'uploaded_date' => $uploaded_date,
				'imported_date' => null, // Initialize as null for now
			] );
		} else {
			wp_send_json_error( [ 'message' => esc_html__( 'Failed to upload the file.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}
	}

	public function delete_csv_file() {
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		$file_name  = sanitize_file_name( $_POST['file_name'] );
		$upload_dir = wp_upload_dir();
		$file_path  = trailingslashit( $upload_dir['basedir'] ) . 'realhomes-crm/csv-import/' . $file_name;

		if ( file_exists( $file_path ) ) {
			unlink( $file_path );
		}

		// Remove the file information from the user's meta
		$current_user_id = get_current_user_id();
		if ( $current_user_id ) {
			$user_files = get_user_meta( $current_user_id, 'rh_crm_uploaded_csv_files', true );

			if ( is_array( $user_files ) ) {
				// Filter out the deleted file
				$user_files = array_filter( $user_files, function ( $file ) use ( $file_name ) {
					return $file['file_name'] !== $file_name;
				} );

				// Save the updated meta
				update_user_meta( $current_user_id, 'rh_crm_uploaded_csv_files', $user_files );
			}
		}

		wp_send_json_success( [ 'message' => esc_html__( 'File deleted successfully.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
	}

	public function import_csv_file() {
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		// Get the note data from the AJAX request
		parse_str( $_POST['data'], $data );

		if ( ! isset( $data['csv_file_name'], $data['fields_mapping'] ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Invalid request. File name or fields mapping is missing.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		$file_name  = sanitize_text_field( $data['csv_file_name'] );
		$upload_dir = wp_upload_dir();
		$file_path  = $upload_dir['basedir'] . '/realhomes-crm/csv-import/' . $file_name;

		if ( ! file_exists( $file_path ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'File not found: ', REALHOMES_CRM_TEXT_DOMAIN ) . $file_name ] );
		}

		$fields_mapping = $data['fields_mapping'];

		// File processing logic remains the same
		$file_handle = fopen( $file_path, 'r' );
		$headers     = fgetcsv( $file_handle ); // CSV headers

		$import_methods = [
			'leads'     => [ Realhomes_Crm_Leads::class, 'import_leads' ],
			'inquiries' => [ Realhomes_Crm_Inquiries::class, 'import_inquiries' ],
			'deals'     => [ Realhomes_Crm_Deals::class, 'import_deals' ],
			'tasks'     => [ Realhomes_Crm_Tasks::class, 'import_tasks' ],
		];

		// Check if the item type is supported and has a corresponding to delete method
		if ( ! isset( $import_methods[ $data['import_type'] ] ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Unsupported import type.', REALHOMES_CRM_TEXT_DOMAIN ) ] );
		}

		$import_results = call_user_func( $import_methods[ $data['import_type'] ], $file_handle, $headers, $fields_mapping );

		fclose( $file_handle );

		// Update last import date for the same file in user meta
		$current_user_id = get_current_user_id();
		if ( $current_user_id ) {
			$user_files = get_user_meta( $current_user_id, 'rh_crm_uploaded_csv_files', true );

			if ( is_array( $user_files ) ) {
				foreach ( $user_files as &$file ) {
					if ( $file['file_name'] === $file_name ) {
						$file['imported_date'] = current_time( 'mysql' ); // Update the imported date
						break; // No need to loop further once the file is found
					}
				}
				// Save the updated array back to user meta
				update_user_meta( $current_user_id, 'rh_crm_uploaded_csv_files', $user_files );
			}
		}

		wp_send_json_success( [ 'import_results' => $import_results ] );
	}

	public function get_leads_stats_data() {
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		// Retrieve current and previous leads counts from database
		global $wpdb;
		$leads_table = $wpdb->prefix . 'realhomes_crm_leads';
		$now         = current_time( 'mysql' );

		// Queries for leads count
		$new_leads_24hrs = $wpdb->get_var( "SELECT COUNT(*) FROM $leads_table WHERE created_at >= DATE_SUB('$now', INTERVAL 1 DAY)" );
		$previous_24hrs  = $wpdb->get_var( "SELECT COUNT(*) FROM $leads_table WHERE created_at >= DATE_SUB('$now', INTERVAL 2 DAY) AND created_at < DATE_SUB('$now', INTERVAL 1 DAY)" );
		$new_leads_week  = $wpdb->get_var( "SELECT COUNT(*) FROM $leads_table WHERE created_at >= DATE_SUB('$now', INTERVAL 1 WEEK)" );
		$previous_week   = $wpdb->get_var( "SELECT COUNT(*) FROM $leads_table WHERE created_at >= DATE_SUB('$now', INTERVAL 2 WEEK) AND created_at < DATE_SUB('$now', INTERVAL 1 WEEK)" );
		$new_leads_month = $wpdb->get_var( "SELECT COUNT(*) FROM $leads_table WHERE created_at >= DATE_SUB('$now', INTERVAL 1 MONTH)" );
		$previous_month  = $wpdb->get_var( "SELECT COUNT(*) FROM $leads_table WHERE created_at >= DATE_SUB('$now', INTERVAL 2 MONTH) AND created_at < DATE_SUB('$now', INTERVAL 1 MONTH)" );

		$get_ratio_arrow = 'Realhomes_Crm_Helper::get_ratio_arrow';

		// Return leads stats with progress comparison
		wp_send_json_success( [
			'24 Hours'   => [ $new_leads_24hrs, $get_ratio_arrow( $new_leads_24hrs, $previous_24hrs ) ],
			'Last Week'  => [ $new_leads_week, $get_ratio_arrow( $new_leads_week, $previous_week ) ],
			'Last Month' => [ $new_leads_month, $get_ratio_arrow( $new_leads_month, $previous_month ) ]
		] );
	}

	public function get_deals_stats_data() {
		check_ajax_referer( 'realhomes_crm_ajax_nonce', 'security' );

		global $wpdb;
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_deals; // Replace with actual table name

		// Query to get deals count by status
		$results = $wpdb->get_results( "
            SELECT deal_stage, COUNT(*) as count 
            FROM $table_name 
            GROUP BY deal_stage
        ",
			ARRAY_A );

		$labels        = $data = [];
		$stages        = Realhomes_Crm_Helper::get_setting( 'deal_stages', explode: true );
		$stages_colors = Realhomes_Crm_Helper::get_setting( 'deal_stages_colors', explode: true );

		foreach ( $stages as $stage ) {
			if ( in_array( $stage, array_column( $results, 'deal_stage' ) ) ) {
				$key      = array_search( $stage, array_column( $results, 'deal_stage' ) );
				$labels[] = $results[ $key ]['deal_stage'];
				$data[]   = (int) $results[ $key ]['count'];
			} else {
				$labels[] = ucfirst( $stage );
				$data[]   = 0;
			}
		}

		wp_send_json_success( [
			'labels' => $labels,
			'data'   => $data,
			'colors' => $stages_colors,
		] );
	}
}

Realhomes_Crm_Ajax::get_instance();