<?php
/**
 * The file that defines all the actions for the activities management.
 *
 * @link       https://realhomes.io/
 * @since      1.0.0
 *
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 */

/**
 * The leads management class.
 *
 * This is used to define all the actions and data management of activites.
 *
 * @since      1.0.0
 * @package    Realhomes_Crm
 * @subpackage Realhomes_Crm/includes
 * @author     InspiryThemes <fahid@inspirythemes.com>
 */
class Realhomes_Crm_Activities {

	/**
	 * Record activity when a booking is received
	 *
	 * @param  $booking_details The booking details
	 */
	public function record_booking_activity( $booking_id ): void {
		$booking_details = get_post_custom( $booking_id );

//		error_log( print_r( $booking_details, true ) );

		// Prepare activity data matching your structure
		$activity_data = array(
			'type' => 'booking',
			'data' => array(
				'name'       => sanitize_text_field( $booking_details['rvr_renter_name'][0] ),
				'email'      => sanitize_email( $booking_details['rvr_renter_email'][0] ),
				'checkin'    => sanitize_text_field( $booking_details['rvr_check_in'][0] ),
				'checkout'   => sanitize_text_field( $booking_details['rvr_check_out'][0] ),
				'source'     => sanitize_text_field( $booking_details['rvr_property_title'][0] ),
				'source_url' => sanitize_text_field( $booking_details['rvr_property_url'][0] ),
				'source_id'  => sanitize_text_field( $booking_details['rvr_property_id'][0] ),
				'booking_id' => $booking_id,
			)
		);

		// Add mobile if exists
		if ( ! empty( $booking_details['rvr_renter_phone'] ) ) {
			$activity_data['data']['mobile'] = sanitize_text_field( $booking_details['rvr_renter_phone'][0] );
		}

		// Trigger the activity recording action
		$this->insert_custom_activity( $activity_data );
	}

	/**
	 * Record activity when a comment is posted
	 *
	 * @param int $comment_id The comment ID
	 */
	public function record_comment_activity( int $comment_id ): void {
		// Get comment object for more reliable data
		$comment = get_comment( $comment_id );

		if ( ! $comment ) {
			return;
		}

		// Check if this is a review (has rating meta)
		$rating    = get_comment_meta( $comment_id, 'inspiry_rating', true );
		$is_review = ! empty( $rating );

		// Get mobile if stored in comment meta
		$mobile = get_comment_meta( $comment_id, 'mobile', true );

		// Prepare activity data matching your structure
		$activity_data = array(
			'type' => $is_review ? 'review' : 'comment',
			'data' => array(
				'name'       => $comment->comment_author,
				'email'      => $comment->comment_author_email,
				'message'    => $comment->comment_content,
				'source'     => get_the_title( $comment->comment_post_ID ),
				'source_url' => get_comment_link( $comment_id ),
				'source_id'  => $comment->comment_post_ID,
				'comment_id' => $comment->comment_ID,
			)
		);

		// Add mobile if exists
		if ( ! empty( $mobile ) ) {
			$activity_data['data']['mobile'] = $mobile;
		}

		// Add rating if this is a review
		if ( $is_review ) {
			$activity_data['data']['rating'] = intval( $rating );
		}

		// Add user ID if logged in
		if ( $comment->user_id ) {
			$activity_data['data']['user_id'] = $comment->user_id;
		}

		// Trigger the activity recording action
		$this->insert_custom_activity( $activity_data );
	}

	/**
	 * Inserting a new custom activity into the CRM activities table.
	 */
	public function insert_custom_activity( $activity_data ) {
		global $wpdb;

		// Prepare data for insertion
		$data = array(
			'type'       => sanitize_text_field( $activity_data['type'] ),
			'data'       => json_encode( $activity_data['data'] ),
			'created_at' => current_time( 'mysql' ),
		);

		// Define the data format
		$format = array( '%s', '%s', '%s' );

		// Insert the data into the activities table
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_activities; // Replace with your actual table name
		$inserted   = $wpdb->insert( $table_name, $data, $format );

		// Check for successful insertion
		if ( $inserted === false ) {
			// Handle the error if insert failed
			error_log( 'Database insert activity failed: ' . $wpdb->last_error );
		}

		return $inserted;
	}

	/**
	 * Insert a new activity into the CRM activities table.
	 *
	 * @return int|false The ID of the newly inserted activity, or false on failure.
	 */
	public function insert_activity( $current_hook ) {
		global $wpdb;

		error_log( 'current hook:' . $current_hook );

		$activity_data = [];
		$type          = 'lead';

		// Check required fields in $_POST data and sanitize
		$activity_data['name']  = isset( $_POST['name'] ) ? sanitize_text_field( $_POST['name'] ) : '';
		$activity_data['email'] = isset( $_POST['email'] ) ? sanitize_email( $_POST['email'] ) : '';

		if ( 'ere_after_contact_form_submission' === $current_hook ) {
			$type = 'lead_contact';
		} elseif ( 'ere_after_agent_agency_form_submission' === $current_hook ) {
			$type = 'lead_agent_agency';
		} elseif ( 'ere_after_cfos_submission' === $current_hook ) {
			$type = 'lead_cfos';
		} elseif ( 'rhea_after_inquiry_form_submission' === $current_hook ) {
			$type = 'lead_inquiry';
		}

		// Preparing phone number
		$possible_number_keys = [ 'number', 'phone' ];

		foreach ( $possible_number_keys as $key ) {
			if ( isset( $_POST[ $key ] ) ) {
				$activity_data['mobile'] = sanitize_text_field( $_POST[ $key ] );
				break; // Stop at the first found value
			}
		}

		// Prepare message
		if ( ! empty( $_POST['message'] ) ) {
			$activity_data['message'] = sanitize_text_field( $_POST['message'] );
		}

		// Prepare source ID
		$possible_source_keys = [ 'property_id', 'the_id', 'agent_id' ];
		$source_id            = '';

		foreach ( $possible_source_keys as $key ) {
			if ( isset( $_POST[ $key ] ) ) {
				$source_id = intval( $_POST[ $key ] );
				break; // Stop at the first found value
			}
		}

		$activity_data['source_id']  = $source_id;
		$activity_data['source']     = get_the_title( $source_id );
		$activity_data['source_url'] = get_the_permalink( $source_id );

		// Define the table name
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_activities;

		// Insert data into the table
		$result = $wpdb->insert(
			$table_name,
			array(
				'type'       => $type,
				'data'       => json_encode( $activity_data ),
				'created_at' => current_time( 'mysql' ),
			),
			array(
				'%s', // type
				'%s', // data
				'%s'  // created_at
			)
		);

		if ( $result === false ) {
			// Handle the error if insert failed
			error_log( 'Database insert activity failed: ' . $wpdb->last_error );
		}

		return $result;
	}

	/**
	 * Delete an activity based on the given activity ID.
	 *
	 * @param int $activity_id The ID of the activity to delete.
	 *
	 * @return bool True on successful deletion, false otherwise.
	 */
	public static function delete_activity( $activity_id ) {
		global $wpdb;

		// Validate and sanitize the activity ID
		$activity_id = intval( $activity_id );

		// Ensure the activity ID is valid
		if ( $activity_id <= 0 ) {
			return false;
		}

		// Define the table name
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_activities;

		// Delete the activity from the table
		$result = $wpdb->delete( $table_name, array( 'id' => $activity_id ), array( '%d' ) );

		if ( $result === false ) {
			// Handle the error if delete failed
			error_log( 'Database delete activity failed: ' . $wpdb->last_error );
		}

		return $result;
	}

	/**
	 * Fetch activities from the database.
	 *
	 * @param array $args    {
	 *                       Optional. Arguments to filter and sort the activities.
	 *
	 * @type string $type    Filter activities by type.
	 * @type string $orderby Column to sort by. Default is 'created_at'.
	 * @type string $order   Sort order. Accepts 'ASC' or 'DESC'. Default is 'DESC'.
	 * @type int    $limit   Number of activities to fetch. Default is 10.
	 *                       }
	 *
	 * @return array|null List of activities or null if none found.
	 */
	public static function fetch_activities( $args = [] ) {
		global $wpdb;

		// Default arguments
		$defaults = [
			'type'    => '',
			'orderby' => 'created_at',
			'order'   => 'DESC',
			'limit'   => 100,
		];

		// Merge defaults with user-supplied arguments
		$args = wp_parse_args( $args, $defaults );

		// Sanitize inputs
		$type    = sanitize_text_field( $args['type'] );
		$orderby = in_array( $args['orderby'], [ 'id', 'type', 'data', 'created_at' ] ) ? $args['orderby'] : 'created_at';
		$order   = strtoupper( $args['order'] ) === 'ASC' ? 'ASC' : 'DESC';
		$limit   = intval( $args['limit'] ) > 0 ? intval( $args['limit'] ) : 10;

		// Base query
		$table_name = $wpdb->prefix . Realhomes_Crm_Helper::$table_activities;
		$query      = "SELECT * FROM $table_name";

		// Add filtering if a type is provided
		if ( ! empty( $type ) ) {
			$query .= $wpdb->prepare( " WHERE type = %s", $type );
		}

		// Add sorting and limit
		$query .= " ORDER BY $orderby $order LIMIT %d";
		$query = $wpdb->prepare( $query, $limit );

		// Execute the query
		return $wpdb->get_results( $query );
	}

}