<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Prevent direct access

class Realhomes_CRM_Settings {
	private static $instance = null;
	private        $options;

	public static function get_instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	private function __construct() {
		$this->options = get_option( 'realhomes_crm_settings', [] );
		add_action( 'admin_menu', [ $this, 'add_settings_page' ], 999 );
		add_action( 'admin_init', [ $this, 'register_settings' ] );
	}

	// Add Settings Page in WP Admin
	public function add_settings_page() {
		add_submenu_page(
			'real_homes', // Parent menu slug
			esc_html__( 'RealHomes CRM Settings', REALHOMES_CRM_TEXT_DOMAIN ),
			esc_html__( 'CRM Settings', REALHOMES_CRM_TEXT_DOMAIN ),
			'manage_options',
			'realhomes-crm-settings',
			[ $this, 'render_settings_page' ]
		);
	}

	// Register settings
	public function register_settings() {
		register_setting( 'realhomes_crm_settings', 'realhomes_crm_settings', [ 'sanitize_callback' => [ $this, 'sanitize_settings' ] ] );

		// Tabs
		$tabs = [ 'general', 'leads', 'inquiries', 'deals', 'tasks' ];
		foreach ( $tabs as $tab ) {
			add_settings_section( $tab, ucfirst( $tab ) . sprintf( ' %s', esc_html__( 'Settings', REALHOMES_CRM_TEXT_DOMAIN ) ), '', 'realhomes-crm-settings-' . $tab );
		}

		// Settings Fields
		$this->add_general_settings();
		$this->add_leads_settings();
		$this->add_inquiries_settings();
		$this->add_deals_settings();
		$this->add_tasks_settings();
	}

	private function add_general_settings() {

		add_settings_field(
			'empty_values_fallback',
			esc_html__( 'Empty Fields Fallback Value', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_text_field' ],
			'realhomes-crm-settings-general',
			'general',
			[
				'id'      => 'empty_values_fallback',
				'default' => Realhomes_Crm_Helper::setting_default( 'empty_values_fallback' ),
				'desc'    => esc_html__( 'Define a fallback value to display when a field is empty.', REALHOMES_CRM_TEXT_DOMAIN )
			]
		);

	}

	private function add_leads_settings() {
		add_settings_field(
			'lead_name_prefixes',
			esc_html__( 'Name Prefixes', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-leads',
			'leads',
			[
				'id'      => 'lead_name_prefixes',
				'default' => Realhomes_Crm_Helper::setting_default( 'lead_name_prefixes' )
			]
		);

		add_settings_field(
			'lead_types',
			esc_html__( 'Lead Types', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-leads',
			'leads',
			[
				'id'      => 'lead_types',
				'default' => Realhomes_Crm_Helper::setting_default( 'lead_types' )
			]
		);

		add_settings_field(
			'lead_statuses',
			esc_html__( 'Lead Statuses', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-leads',
			'leads',
			[
				'id'      => 'lead_statuses',
				'default' => Realhomes_Crm_Helper::setting_default( 'lead_statuses' )
			]
		);

	}

	private function add_inquiries_settings() {

		add_settings_field(
			'inquiry_types',
			esc_html__( 'Inquiry Types', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-inquiries',
			'inquiries',
			[
				'id'      => 'inquiry_types',
				'default' => Realhomes_Crm_Helper::setting_default( 'inquiry_types' )
			]
		);

		add_settings_field(
			'inquiry_statuses',
			esc_html__( 'Inquiry Statuses', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-inquiries',
			'inquiries',
			[
				'id'      => 'inquiry_statuses',
				'default' => Realhomes_Crm_Helper::setting_default( 'inquiry_statuses' )
			]
		);

	}

	private function add_deals_settings() {

		add_settings_field(
			'deal_stages',
			esc_html__( 'Deal Stages', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-deals',
			'deals',
			[
				'id'      => 'deal_stages',
				'default' => Realhomes_Crm_Helper::setting_default( 'deal_stages' )
			]
		);

		add_settings_field(
			'deal_stages_colors',
			esc_html__( 'Deal Stages Colors', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-deals',
			'deals',
			[
				'id'      => 'deal_stages_colors',
				'default' => Realhomes_Crm_Helper::setting_default( 'deal_stages_colors' )
			]
		);

		add_settings_field(
			'deal_probabilities',
			esc_html__( 'Deal Probabilities', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-deals',
			'deals',
			[
				'id'      => 'deal_probabilities',
				'default' => Realhomes_Crm_Helper::setting_default( 'deal_probabilities' )
			]
		);

		add_settings_field(
			'deal_last_steps',
			esc_html__( 'Deal Last Steps', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-deals',
			'deals',
			[
				'id'      => 'deal_last_steps',
				'default' => Realhomes_Crm_Helper::setting_default( 'deal_last_steps' )
			]
		);

		add_settings_field(
			'deal_next_steps',
			esc_html__( 'Deal Next Steps', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-deals',
			'deals',
			[
				'id'      => 'deal_next_steps',
				'default' => Realhomes_Crm_Helper::setting_default( 'deal_next_steps' )
			]
		);

	}

	private function add_tasks_settings() {

		add_settings_field(
			'task_types',
			esc_html__( 'Task Types', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-tasks',
			'tasks',
			[
				'id'      => 'task_types',
				'default' => Realhomes_Crm_Helper::setting_default( 'task_types' )
			]
		);

		add_settings_field(
			'task_statuses',
			esc_html__( 'Task Statuses', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-tasks',
			'tasks',
			[
				'id'      => 'task_statuses',
				'default' => Realhomes_Crm_Helper::setting_default( 'task_statuses' )
			]
		);

		add_settings_field(
			'task_priorities',
			esc_html__( 'Task Priorities', REALHOMES_CRM_TEXT_DOMAIN ),
			[ $this, 'render_textarea_field' ],
			'realhomes-crm-settings-tasks',
			'tasks',
			[
				'id'      => 'task_priorities',
				'default' => Realhomes_Crm_Helper::setting_default( 'task_priorities' )
			]
		);
	}

	// Render Settings Page
	public function render_settings_page() {
		?>
        <div id="realhomes-settings-wrap">
			<?php
			$this->the_header();
			$this->the_content();
			$this->the_footer();
			?>
        </div>
		<?php
	}

	public function the_content() {
		?>
        <div class="settings-content">
			<?php

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', REALHOMES_CRM_TEXT_DOMAIN ) );
			}

			settings_errors();

			$this->tabs_nav();
			?>
            <div class="form-wrapper">
                <form method="post" action="options.php">
					<?php
					settings_fields( 'realhomes_crm_settings' );
					do_settings_sections( 'realhomes-crm-settings-' . $this->get_current_tab() );
					submit_button();
					?>
                </form>
            </div>
        </div>
		<?php
	}

	public function the_header() {
		?>
        <header class="settings-header">
            <h1><?php esc_html_e( 'Realhomes CRM Settings', REALHOMES_CRM_TEXT_DOMAIN ); ?><span class="current-version-tag"><?php echo REALHOMES_CRM_VERSION; ?></span></h1>
            <p class="credit">
                <a class="logo-wrap" href="https://themeforest.net/item/real-homes-wordpress-real-estate-theme/5373914?aid=inspirythemes" target="_blank">
                    <svg xmlns="http://www.w3.org/2000/svg" height="29" width="29" viewBox="0 0 36 41">
                        <style>
                            .a {
                                fill: #4E637B;
                            }

                            .b {
                                fill: white;
                            }

                            .c {
                                fill: #27313D !important;
                            }
                        </style>
                        <g>
                            <path d="M25.5 14.6C28.9 16.6 30.6 17.5 34 19.5L34 11.1C34 10.2 33.5 9.4 32.8 9 30.1 7.5 28.4 6.5 25.5 4.8L25.5 14.6Z" class="a"></path>
                            <path d="M15.8 38.4C16.5 38.8 17.4 38.8 18.2 38.4 20.8 36.9 22.5 35.9 25.5 34.2 22.1 32.2 20.4 31.3 17 29.3 13.6 31.3 11.9 32.2 8.5 34.2 11.5 35.9 13.1 36.9 15.8 38.4" mask="url(#mask-2)" class="a"></path>
                            <path d="M24.3 25.1C25 24.7 25.5 23.9 25.5 23L25.5 14.6 17 19.5 17 29.3 24.3 25.1Z" fill="#C8ED1E"></path>
                            <path d="M18.2 10.4C17.4 10 16.5 10 15.8 10.4L8.5 14.6 17 19.5 25.5 14.6 18.2 10.4Z" fill="#F9FAF8"></path>
                            <path d="M8.5 23C8.5 23.9 8.9 24.7 9.7 25.1L17 29.3 17 19.5 8.5 14.6 8.5 23Z" fill="#88B2D7"></path>
                            <path d="M8.5 14.6C5.1 16.6 3.4 17.5 0 19.5L0 11.1C0 10.2 0.5 9.4 1.2 9 3.8 7.5 5.5 6.5 8.5 4.8L8.5 14.6Z" mask="url(#mask-4)" class="a"></path>
                            <path d="M34 27.9L34 19.5 25.5 14.6 25.5 23C25.5 23.4 25.4 23.8 25.1 24.2L33.6 29.1C33.8 28.7 34 28.3 34 27.9" fill="#5E9E2D"></path>
                            <path d="M25.1 24.2C24.9 24.6 24.6 24.9 24.3 25.1L17 29.3 25.5 34.2 32.8 30C33.1 29.8 33.4 29.5 33.6 29.1L25.1 24.2Z" fill="#6FBF2C"></path>
                            <path d="M17 10.1C17.4 10.1 17.8 10.2 18.2 10.4L25.5 14.6 25.5 4.8 18.2 0.6C17.8 0.4 17.4 0.3 17 0.3L17 10.1Z" fill="#BDD2E1"></path>
                            <path d="M1.2 30L8.5 34.2 17 29.3 9.7 25.1C9.3 24.9 9 24.6 8.8 24.2L0.3 29.1C0.5 29.5 0.8 29.8 1.2 30" fill="#418EDA"></path>
                            <path d="M8.8 24.2C8.6 23.8 8.5 23.4 8.5 23L8.5 14.6 0 19.5 0 27.9C0 28.3 0.1 28.7 0.3 29.1L8.8 24.2Z" fill="#3570AA"></path>
                            <path d="M15.8 0.6L8.5 4.8 8.5 14.6 15.8 10.4C16.2 10.2 16.6 10.1 17 10.1L17 0.3C16.6 0.3 16.2 0.4 15.8 0.6" fill="#A7BAC8"></path>
                        </g>
                    </svg>
                    InspiryThemes
                </a>
            </p>
        </header>
		<?php
	}

	public function the_footer() {
		?>
        <footer class="settings-footer">
			<?php
			$doc_url = 'https://realhomes.io/documentation/';
			?>
            <p>
                <span class="dashicons dashicons-editor-help"></span>
				<?php printf( esc_html__( 'For help, please consult the %1$s documentation %2$s of the theme.', REALHOMES_CRM_TEXT_DOMAIN ), '<a href="' . esc_attr( $doc_url ) . '" target="_blank">', '</a>' ); ?>
            </p>
            <p>
                <span class="dashicons dashicons-feedback"></span>
				<?php printf( esc_html__( 'For feedback, please provide your %1$s feedback here! %2$s', REALHOMES_CRM_TEXT_DOMAIN ), '<a href="' . esc_url( add_query_arg( array( 'page' => 'realhomes-feedback' ), get_admin_url() . 'admin.php' ) ) . '" target="_blank">', '</a>' ); ?>
            </p>
        </footer>
		<?php
	}

	public function tabs_nav() {

		$tabs        = $this->tabs();
		$current_tab = $this->get_current_tab();
		?>
        <div class="nav-tab-wrapper">
			<?php
			if ( ! empty( $tabs ) && is_array( $tabs ) ) {
				foreach ( $tabs as $slug => $title ) {
					$active_tab_class = ( $current_tab === $slug ) ? 'nav-tab-active' : '';
					$admin_url        = ( $current_tab === $slug ) ? '#' : admin_url( 'admin.php?page=realhomes-crm-settings&tab=' . $slug );
					echo '<a class="nav-tab ' . $active_tab_class . '" href="' . esc_url_raw( $admin_url ) . '" data-tab="' . esc_attr( $slug ) . '">' . esc_html( $title ) . '</a>';
				}
			}
			?>
        </div>
		<?php
	}

	private function tabs() {

		$tabs = array(
			'general'   => esc_html__( 'General', REALHOMES_CRM_TEXT_DOMAIN ),
			'leads'     => esc_html__( 'Leads', REALHOMES_CRM_TEXT_DOMAIN ),
			'inquiries' => esc_html__( 'Inquiries', REALHOMES_CRM_TEXT_DOMAIN ),
			'deals'     => esc_html__( 'Deals', REALHOMES_CRM_TEXT_DOMAIN ),
			'tasks'     => esc_html__( 'Tasks', REALHOMES_CRM_TEXT_DOMAIN ),
		);

		// Filter to add the New Settings tabs
		return apply_filters( 'realhomes_crm_settings_tabs', $tabs );
	}

	private function get_current_tab() {
		return isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : 'general';
	}

	// Render Settings Fields
	public function render_text_field( $args ) {

		$value = isset( $this->options[ $args['id'] ] )
			? esc_attr( $this->options[ $args['id'] ] )
			: ( isset( $args['default'] ) ? esc_attr( $args['default'] ) : '' );

		echo '<input type="text" id="' . esc_attr( $args['id'] ) . '" name="realhomes_crm_settings[' . esc_attr( $args['id'] ) . ']" value="' . $value . '" class="regular-text">';
		echo isset( $args['desc'] ) ? '<p class="description">' . esc_html( $args['desc'] ) . '</p>' : '';
	}

	public function render_checkbox_field( $args ) {
		$checked = isset( $this->options[ $args['id'] ] ) ? checked( 1, $this->options[ $args['id'] ], false ) : '';
		echo '<input type="checkbox" id="' . esc_attr( $args['id'] ) . '" name="realhomes_crm_settings[' . esc_attr( $args['id'] ) . ']" value="1" ' . $checked . '>';
		echo '<label for="' . esc_attr( $args['id'] ) . '"> ' . esc_html( $args['label'] ) . '</label>';
	}

	public function render_radio_field( $args ) {
		foreach ( $args['options'] as $option ) {
			$checked = ( isset( $this->options[ $args['id'] ] ) && $this->options[ $args['id'] ] === $option ) ? 'checked' : '';
			echo '<label><input type="radio" name="realhomes_crm_settings[' . esc_attr( $args['id'] ) . ']" value="' . esc_attr( $option ) . '" ' . $checked . '> ' . esc_html( $option ) . '</label><br>';
		}
	}

	public function render_select_field( $args ) {
		echo '<select id="' . esc_attr( $args['id'] ) . '" name="realhomes_crm_settings[' . esc_attr( $args['id'] ) . ']">';
		foreach ( $args['options'] as $option ) {
			$selected = ( isset( $this->options[ $args['id'] ] ) && $this->options[ $args['id'] ] === $option ) ? 'selected' : '';
			echo '<option value="' . esc_attr( $option ) . '" ' . $selected . '>' . esc_html( $option ) . '</option>';
		}
		echo '</select>';
		echo isset( $args['desc'] ) ? '<p class="description">' . esc_html( $args['desc'] ) . '</p>' : '';
	}

	public function render_textarea_field( $args ) {

		$value = isset( $this->options[ $args['id'] ] )
			? esc_textarea( $this->options[ $args['id'] ] )
			: ( isset( $args['default'] ) ? esc_textarea( $args['default'] ) : '' );

		echo '<textarea id="' . esc_attr( $args['id'] ) . '" name="realhomes_crm_settings[' . esc_attr( $args['id'] ) . ']" rows="4" class="regular-text">' . $value . '</textarea>';
		echo isset( $args['desc'] ) ? '<p class="description">' . esc_html( $args['desc'] ) . '</p>' : '';
	}

	// Sanitize and merge settings
	public function sanitize_settings( $input ) {
		$sanitized = [];
		foreach ( $input as $key => $value ) {
			if ( is_array( $value ) ) {
				$sanitized[ $key ] = array_map( 'sanitize_text_field', $value );
			} else {
				$sanitized[ $key ] = sanitize_text_field( $value );
			}
		}

		$options = get_option( 'realhomes_crm_settings', [] );

		return array_merge( $options, $sanitized );
	}
}

Realhomes_CRM_Settings::get_instance();