<?php
/**
 * Widget: Featured Properties Widget
 *
 * @since   3.0.0
 * @package easy_real_estate
 */

if ( ! class_exists( 'Featured_Properties_Widget' ) ) {

	/**
	 * Featured_Properties_Widget.
	 *
	 * Widget of featured properties.
	 *
	 * @since 3.0.0
	 */
	class Featured_Properties_Widget extends WP_Widget {

		/**
		 * Method: Constructor
		 *
		 * @since  1.0.0
		 */
		function __construct() {
			$widget_ops = array(
				'classname'                   => 'Featured_Properties_Widget',
				'description'                 => esc_html__( 'Displays Random or Recent Featured Properties.', ERE_TEXT_DOMAIN ),
				'customize_selective_refresh' => true,
			);
			parent::__construct(
				'Featured_Properties_Widget',
				esc_html__( 'RealHomes - Featured Properties', ERE_TEXT_DOMAIN ),
				$widget_ops
			);
		}

		/**
		 * Method: Widget Front-End
		 *
		 * @param array $args     - Arguments of the widget.
		 * @param array $instance - Instance of the widget.
		 */
		function widget( $args, $instance ) {

			extract( $args );

			global $featured_text;

			// Title
			if ( isset( $instance['title'] ) ) {
				$title = apply_filters( 'widget_title', $instance['title'], $instance, $this->id_base );
			}

			if ( isset( $instance['view_property'] ) ) {
				$view_property = apply_filters( 'view_property', $instance['view_property'], $instance, $this->id_base );
			}

			if ( isset( $instance['featured_text'] ) ) {
				$featured_text = apply_filters( 'featured_text', $instance['featured_text'], $instance, $this->id_base );
			}

			if ( empty( $title ) ) {
				$title = false;
			}

			// Count
			$count = 1;
			if ( isset( $instance['count'] ) ) {
				$count = intval( $instance['count'] );
			}

			$featured_properties_args = array(
				'post_type'      => 'property',
				'posts_per_page' => $count,
				'meta_query'     => array(
					array(
						'key'     => 'REAL_HOMES_featured',
						'value'   => 1,
						'compare' => '=',
						'type'    => 'NUMERIC',
					),
				),
			);

			if ( is_singular( 'property' ) ) {
				$featured_properties_args['post__not_in'] = array( get_the_ID() );
			}

			// Order by
			$sort_by = 'recent';
			if ( isset( $instance['sort_by'] ) ) {
				$sort_by = $instance['sort_by'];
			}
			if ( 'random' == $sort_by ) :
				$featured_properties_args['orderby'] = 'rand';
			else :
				$featured_properties_args['orderby'] = 'date';
			endif;

			$featured_properties_query = new WP_Query( apply_filters( 'ere_featured_properties_widget', $featured_properties_args ) );
			echo $before_widget;

			if ( $title ) :
				echo $before_title;
				echo $title;
				echo $after_title;
			endif;

			if ( 'classic' === INSPIRY_DESIGN_VARIATION ) {
				if ( $featured_properties_query->have_posts() ) {
					?>
                    <ul class="featured-properties">
						<?php
						while ( $featured_properties_query->have_posts() ) {
							$featured_properties_query->the_post();
							?>
                            <li>
                                <figure>
                                    <a href="<?php the_permalink(); ?>">
										<?php
										if ( has_post_thumbnail() ) {
											the_post_thumbnail( 'property-thumb-image' );
										} else {
											if ( function_exists( 'inspiry_image_placeholder' ) ) {
												inspiry_image_placeholder( 'property-thumb-image' );
											}
										}
										?>
                                    </a>
                                </figure>

                                <h4><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h4>
                                <p><?php ere_framework_excerpt( 7 ); ?>
									<?php
									if ( isset( $view_property ) && ! empty( $view_property ) ) {
										?>
                                        <a href="<?php the_permalink(); ?>"><?php echo esc_html( $view_property ); ?></a>
										<?php
									} else {
										?>
                                        <a href="<?php the_permalink(); ?>"><?php esc_html_e( 'Read More', ERE_TEXT_DOMAIN ); ?></a>
										<?php
									}
									?>
                                </p>
								<?php echo '<span class="price">' . ere_get_property_price() . '</span>'; ?>
                            </li>
							<?php
						}
						?>
                    </ul>
					<?php
					wp_reset_query();
				} else {
					?>
                    <ul class="featured-properties">
						<?php
						echo '<li>';
						esc_html_e( 'No Featured Property Found!', ERE_TEXT_DOMAIN );
						echo '</li>';
						?>
                    </ul>
					<?php
				}
			} else if ( 'modern' === INSPIRY_DESIGN_VARIATION ) {

				global $this_widget_id;
				$this_widget_id = $this->id;

				if ( $featured_properties_query->have_posts() ) {
					if ( isset( $instance['card_design'] ) && 'default' != $instance['card_design'] ) {
						$property_card_variation = $instance['card_design'];
					} else {
						$inspiry_property_card_variation = get_option( 'inspiry_property_card_variation', '1' );
						$property_card_variation         = $inspiry_property_card_variation;
					}
					while ( $featured_properties_query->have_posts() ) {
						$featured_properties_query->the_post();

						ere_get_template_part( 'includes/widgets/grid-cards/grid-card-' . $property_card_variation );
					}
					wp_reset_postdata();
				} else {
					?>
                    <div class="rh_alert-wrapper rh_alert__widget">
                        <h4 class="no-results"><?php esc_html_e( 'No Featured Property Found!', ERE_TEXT_DOMAIN ); ?></h4>
                    </div>
					<?php
				}
			} else if ( 'ultra' === INSPIRY_DESIGN_VARIATION ) {
				global $this_widget_id;
				$this_widget_id = $this->id;

				if ( $featured_properties_query->have_posts() ) {
					?>
                    <div class="ere-ultra-side-properties-wrapper">
                        <div class="ere-ultra-side-properties ere-ultra-properties-slider owl-carousel rh-ultra-slider-nav">
							<?php

							while ( $featured_properties_query->have_posts() ) {
								$featured_properties_query->the_post();
								?>
                                <div class="ere-properties-slide item">
									<?php
									ere_get_template_part( 'includes/widgets/ultra-grid-cards/grid-card-2' );

									?>
                                </div>
								<?php
							}
							?>
                        </div>
						<?php
						if ( $featured_properties_query->post_count > 1 ) {
							?>
                            <div class="rh-ultra-widget-dots"></div>
							<?php
						}
						?>
                    </div>
					<?php
					wp_reset_postdata();
				} else {
					?>
                    <div class="rh_alert-wrapper rh_alert__widget">
                        <h4 class="no-results"><?php esc_html_e( 'No Featured Property Found!', ERE_TEXT_DOMAIN ); ?></h4>
                    </div>
					<?php
				}

			}

			echo $after_widget;
		}

		/**
		 * Method: Widget Backend Form
		 *
		 * @param array $instance - Instance of the widget.
		 *
		 * @return void
		 */
		function form( $instance ) {
			if ( 'ultra' === INSPIRY_DESIGN_VARIATION ) {
				$label_property = '';
			} else if ( 'modern' === INSPIRY_DESIGN_VARIATION ) {
				$label_property = esc_html__( 'View Property', ERE_TEXT_DOMAIN );
			} else if ( 'classic' === INSPIRY_DESIGN_VARIATION ) {
				$label_property = esc_html__( 'Read More', ERE_TEXT_DOMAIN );
			}
			$instance      = wp_parse_args(
				(array)$instance,
				array(
					'title'         => esc_html__( 'Featured Properties', ERE_TEXT_DOMAIN ),
					'count'         => 1,
					'sort_by'       => 'random',
					'view_property' => $label_property,
					'featured_text' => esc_html__( 'Featured', ERE_TEXT_DOMAIN ),
				)
			);
			$view_property = $instance['view_property'];
			$featured_text = $instance['featured_text'];
			$title         = $instance['title'];
			$card_design   = isset( $instance['card_design'] ) ? $instance['card_design'] : 'default';
			$count         = $instance['count'];
			$sort_by       = $instance['sort_by'];
			?>
            <p>
                <label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Widget Title', ERE_TEXT_DOMAIN ); ?></label>
                <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
            </p>
			<?php
			if ( 'ultra' !== INSPIRY_DESIGN_VARIATION ) {
				?>
                <p>
                    <label for="<?php echo esc_attr( $this->get_field_id( 'card_design' ) ); ?>"><?php esc_html_e( 'Property Card Design', ERE_TEXT_DOMAIN ); ?></label>
                    <select name="<?php echo esc_attr( $this->get_field_name( 'card_design' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'card_design' ) ); ?>" class="widefat">
                        <option value="default"<?php selected( $card_design, 'default' ); ?>><?php esc_html_e( 'Default', ERE_TEXT_DOMAIN ); ?></option>
                        <option value="1"<?php selected( $card_design, '1' ); ?>><?php esc_html_e( 'One', ERE_TEXT_DOMAIN ); ?></option>
                        <option value="2"<?php selected( $card_design, '2' ); ?>><?php esc_html_e( 'Two', ERE_TEXT_DOMAIN ); ?></option>
                        <option value="3"<?php selected( $card_design, '3' ); ?>><?php esc_html_e( 'Three', ERE_TEXT_DOMAIN ); ?></option>
                        <option value="4"<?php selected( $card_design, '4' ); ?>><?php esc_html_e( 'Four', ERE_TEXT_DOMAIN ); ?></option>
                        <option value="5"<?php selected( $card_design, '5' ); ?>><?php esc_html_e( 'Five', ERE_TEXT_DOMAIN ); ?></option>
                    </select>
                </p>
				<?php
			}
			?>
            <p>
                <label for="<?php echo esc_attr( $this->get_field_id( 'count' ) ); ?>"><?php esc_html_e( 'Number of Properties', ERE_TEXT_DOMAIN ); ?></label>
                <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'count' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'count' ) ); ?>" type="text" value="<?php echo esc_attr( $count ); ?>" />
            </p>
            <p>
                <label for="<?php echo esc_attr( $this->get_field_id( 'sort_by' ) ); ?>"><?php esc_html_e( 'Sort By:', ERE_TEXT_DOMAIN ); ?></label>
                <select name="<?php echo esc_attr( $this->get_field_name( 'sort_by' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'sort_by' ) ); ?>" class="widefat">
                    <option value="recent" <?php selected( $sort_by, 'recent' ); ?>><?php esc_html_e( 'Most Recent', ERE_TEXT_DOMAIN ); ?></option>
                    <option value="random" <?php selected( $sort_by, 'random' ); ?>><?php esc_html_e( 'Random', ERE_TEXT_DOMAIN ); ?></option>
                </select>
            </p><p>
				<?php
				if ( 'modern' === INSPIRY_DESIGN_VARIATION ) {
					?>
                    <label for="<?php echo esc_attr( $this->get_field_id( 'view_property' ) ); ?>"><?php esc_html_e( 'View Property', ERE_TEXT_DOMAIN ); ?></label>
					<?php
				} else if ( 'classic' === INSPIRY_DESIGN_VARIATION ) {
					?>
                    <label for="<?php echo esc_attr( $this->get_field_id( 'view_property' ) ); ?>"><?php esc_html_e( 'Read More', ERE_TEXT_DOMAIN ); ?></label>
					<?php
				}
				if ( 'ultra' !== INSPIRY_DESIGN_VARIATION ) {
					?>
                    <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'view_property' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'view_property' ) ); ?>" type="text" value="<?php echo esc_attr( $view_property ); ?>" />
					<?php
				}
				?>
            </p>
			<?php
			if ( 'modern' === INSPIRY_DESIGN_VARIATION ) {
				?>
                <p>
                    <label for="<?php echo esc_attr( $this->get_field_id( 'featured_text' ) ); ?>"><?php esc_html_e( 'Featured Tag Text', ERE_TEXT_DOMAIN ); ?></label>
                    <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'featured_text' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'featured_text' ) ); ?>" type="text" value="<?php echo esc_attr( $featured_text ); ?>" />
                </p>
				<?php
			}
			?><?php
		}

		/**
		 * Method: Widget Update Function
		 *
		 * @param array $new_instance - New instance of the widget.
		 * @param array $old_instance - Old instance of the widget.
		 *
		 * @return array
		 */
		function update( $new_instance, $old_instance ) {
			$instance = $old_instance;

			$instance['title']         = sanitize_text_field( isset( $new_instance['title'] ) ? $new_instance['title'] : '' );
			$instance['view_property'] = sanitize_text_field( isset( $new_instance['view_property'] ) ? $new_instance['view_property'] : '' );
			$instance['featured_text'] = sanitize_text_field( isset( $new_instance['featured_text'] ) ? $new_instance['featured_text'] : '' );
			$instance['count']         = isset( $new_instance['count'] ) ? $new_instance['count'] : 1;
			$instance['sort_by']       = isset( $new_instance['sort_by'] ) ? $new_instance['sort_by'] : '';
			$instance['card_design']   = isset( $new_instance['card_design'] ) ? $new_instance['card_design'] : '';


			return $instance;
		}

	}
}


if ( ! function_exists( 'register_featured_properties_widget' ) ) {
	function register_featured_properties_widget() {
		register_widget( 'Featured_Properties_Widget' );
	}

	add_action( 'widgets_init', 'register_featured_properties_widget' );
}
