<?php
/**
 * Social login callbacks catcher functions.
 * These functions also manage user registration and sign-in.
 *
 * @since      0.5.3
 * @package    easy-real-estate
 * @subpackage easy-real-estate/includes/social-login
 */

if ( ere_is_social_login_enabled( 'facebook' ) && ( isset( $_GET['code'] ) && isset( $_GET['state'] ) ) ) {
	add_action( 'init', 'ere_facebook_oauth_login' );
} else if ( ere_is_social_login_enabled( 'google' ) && isset( $_GET['code'] ) ) {
	add_action( 'init', 'ere_google_oauth_login' );
} else if ( ere_is_social_login_enabled( 'twitter' ) && isset( $_GET['oauth_token'] ) && isset( $_GET['oauth_verifier'] ) ) {
	add_action( 'init', 'ere_twitter_oauth_login' );
}

if ( ! function_exists( 'ere_facebook_oauth_login' ) ) {
	/**
	 * Facebook profile login.
	 */
	function ere_facebook_oauth_login() {

		// Facebook library.
		require_once ERE_PLUGIN_DIR . 'includes/social-login/libs/facebook/autoload.php';

		if ( class_exists( 'Facebook\Facebook' ) && null !== ere_social_login_app_keys( 'facebook' ) ) {

			$fb_app_keys = ere_social_login_app_keys( 'facebook' );
			$fb_args     = array(
				'app_id'                => $fb_app_keys['app_id'],
				'app_secret'            => $fb_app_keys['app_secret'],
				'default_graph_version' => 'v2.10',
			);

			$fb = new Facebook\Facebook( $fb_args );

			$helper = $fb->getRedirectLoginHelper();

			if ( isset( $_GET['state'] ) ) {
				$helper->getPersistentDataHandler()->set( 'state', $_GET['state'] );
			}

			try {
				$access_token_obj = $helper->getAccessToken();
			} catch ( Facebook\Exception\ResponseException $e ) {
				// When Graph returns an error.
				echo esc_html__( 'Graph returned an error: ', ERE_TEXT_DOMAIN ) . esc_html( $e->getMessage() );
				exit;
			} catch ( Facebook\Exception\SDKException $e ) {
				// When validation fails or other local issues.
				echo esc_html__( 'Facebook SDK returned an error: ', ERE_TEXT_DOMAIN ) . esc_html( $e->getMessage() );
				exit;
			}

			if ( ! isset( $access_token_obj ) ) {
				if ( $helper->getError() ) {
					header( 'HTTP/1.0 401 Unauthorized' );
					echo esc_html__( 'Error: ', ERE_TEXT_DOMAIN ) . esc_html( $helper->getError() ) . '\n';
					echo esc_html__( 'Error Code: ', ERE_TEXT_DOMAIN ) . esc_html( $helper->getErrorCode() ) . '\n';
					echo esc_html__( 'Error Reason: ', ERE_TEXT_DOMAIN ) . esc_html( $helper->getErrorReason() ) . '\n';
					echo esc_html__( 'Error Description: ', ERE_TEXT_DOMAIN ) . esc_html( $helper->getErrorDescription() ) . '\n';
				} else {
					header( 'HTTP/1.0 400 Bad Request' );
					esc_html_e( 'Bad request', ERE_TEXT_DOMAIN );
				}
				exit;
			}

			$access_token = (string)$access_token_obj->getValue();

			$fb = new Facebook\Facebook(
				array(
					'app_id'                => esc_html( $fb_app_keys['app_id'] ),
					'app_secret'            => esc_html( $fb_app_keys['app_secret'] ),
					'default_graph_version' => 'v2.10',
					'default_access_token'  => $access_token,
				)
			);

			try {
				// Returns a `Facebook\Response` object.
				$response = $fb->get( '/me?fields=id,email,name,first_name,last_name' );
			} catch ( Facebook\Exception\ResponseException $e ) {
				echo esc_html__( 'Graph returned an error: ', ERE_TEXT_DOMAIN ) . esc_html( $e->getMessage() );
				exit;
			} catch ( Facebook\Exception\SDKException $e ) {
				echo esc_html__( 'Facebook SDK returned an error: ', ERE_TEXT_DOMAIN ) . esc_html( $e->getMessage() );
				exit;
			}

			$user = $response->getGraphUser();

			$register_cred['user_email']    = $user['email'];
			$register_cred['user_login']    = explode( '@', $user['email'] );
			$register_cred['user_login']    = $register_cred['user_login'][0];
			$register_cred['display_name']  = $user['name'];
			$register_cred['first_name']    = $user['first_name'];
			$register_cred['last_name']     = $user['last_name'];
			$register_cred['profile_image'] = 'https://graph.facebook.com/' . $user['id'] . '/picture?width=300&height=300';
			$register_cred['user_pass']     = $user['id'];

			// Register the user, if successful then login the user.
			ere_social_register( $register_cred );
		}
	}
}

if ( ! function_exists( 'ere_google_oauth_login' ) ) {
	/**
	 * Google oauth login.
	 */
	function ere_google_oauth_login() {

		// Google Client and Oauth libraries.
		require_once ERE_PLUGIN_DIR . 'includes/social-login/libs/google/autoload.php';

		if ( class_exists( 'Google_Client' ) && class_exists( 'Google_Service_Oauth2' ) && null !== ere_social_login_app_keys( 'google' ) ) {

			$google_app_creds     = ere_social_login_app_keys( 'google' );
			$google_client_id     = $google_app_creds['client_id'];
			$google_client_secret = $google_app_creds['client_secret'];
			$google_developer_key = $google_app_creds['api_key'];
			$google_redirect_url  = home_url();

			$google_client = new Google_Client();
			$google_client->setApplicationName( esc_html__( 'Login to', ERE_TEXT_DOMAIN ) . get_bloginfo( 'name' ) );
			$google_client->setClientId( $google_client_id );
			$google_client->setClientSecret( $google_client_secret );
			$google_client->setDeveloperKey( $google_developer_key );
			$google_client->setRedirectUri( $google_redirect_url );
			$google_client->setScopes( array( 'email', 'profile' ) );

			$google_oauth_v2 = new Google_Service_Oauth2( $google_client );
			$code            = sanitize_text_field( wp_unslash( $_GET['code'] ) );
			$google_client->fetchAccessTokenWithAuthCode( $code );

			if ( $google_client->getAccessToken() ) {

				$user = $google_oauth_v2->userinfo->get();

				$register_cred['user_email']    = $user['email'];
				$register_cred['user_login']    = explode( '@', $user['email'] );
				$register_cred['user_login']    = $register_cred['user_login'][0];
				$register_cred['display_name']  = $user['name'];
				$register_cred['first_name']    = isset( $user['given_name'] ) ? $user['given_name'] : '';
				$register_cred['last_name']     = isset( $user['family_name'] ) ? $user['family_name'] : '';
				$register_cred['profile_image'] = $user['picture'];
				$register_cred['user_pass']     = $user['id'];

				// Register the user, if successful then login the user.
				ere_social_register( $register_cred );
			}
		}
	}
}

if ( ! function_exists( 'ere_twitter_oauth_login' ) ) {
	/**
	 * Twitter oauth login.
	 */
	function ere_twitter_oauth_login() {

		// Twitter library.
		require_once ERE_PLUGIN_DIR . 'includes/social-login/libs/twitter/autoload.php';

		if ( class_exists( 'Abraham\TwitterOAuth\TwitterOAuth' ) && null !== ere_social_login_app_keys( 'twitter' ) ) {

			$twitter_app_keys = ere_social_login_app_keys( 'twitter' );
			$consumer_key     = $twitter_app_keys['consumer_key'];
			$consumer_secret  = $twitter_app_keys['consumer_secret'];

			$connection    = new Abraham\TwitterOAuth\TwitterOAuth( $consumer_key, $consumer_secret );
			$request_token = $connection->oauth( 'oauth/access_token', array( 'oauth_consumer_key' => $consumer_key, 'oauth_token' => $_GET['oauth_token'], 'oauth_verifier' => $_GET['oauth_verifier'] ) );

			$connection = new Abraham\TwitterOAuth\TwitterOAuth( $consumer_key, $consumer_secret, $request_token['oauth_token'], $request_token['oauth_token_secret'] );
			$user       = (array)$connection->get( 'account/verify_credentials', array( 'include_email' => 'true' ) );

			$register_cred['user_email']    = $user['email'];
			$register_cred['user_login']    = explode( '@', $user['email'] );
			$register_cred['user_login']    = $register_cred['user_login'][0];
			$register_cred['display_name']  = $user['name'];
			$register_cred['first_name']    = explode( ' ', $user['name'] );
			$register_cred['last_name']     = isset( $register_cred['first_name'][1] ) ? $register_cred['first_name'][1] : '';
			$register_cred['first_name']    = $register_cred['first_name'][0];
			$register_cred['profile_image'] = str_replace( '_normal', '_400x400', $user['profile_image_url_https'] );
			$register_cred['user_pass']     = $user['id'];

			// Register the user, if successful then login the user.
			ere_social_register( $register_cred );

		}
	}
}

if ( ! function_exists( 'ere_social_login' ) ) {
	/**
	 * Logging in with the social profile credentials.
	 *
	 * @param array $login_creds Login credentials.
	 */
	function ere_social_login( $login_creds ) {

		$user_signon = wp_signon( $login_creds, true );

		if ( is_wp_error( $user_signon ) ) {
			wp_safe_redirect( home_url() );
		} else {
			ere_social_login_redirect(); // Redirect the user to the edit profile page.
		}
		exit;
	}
}

if ( ! function_exists( 'ere_social_register' ) ) {
	/**
	 * User registration with social profile information.
	 *
	 * @param array $register_cred User registration credentials.
	 */
	function ere_social_register( $register_cred ) {
		// Check if OTP template is assigned
		$otp_page_id = get_option( 'realhomes_otp_page', false );
		if ( empty( $otp_page_id ) ) {
			wp_die( esc_html__( 'OTP page template is not set for social login.', ERE_TEXT_DOMAIN ) );
		} else {
			$otp_page_link = get_permalink( $otp_page_id );
		}

		/**
		 * Check if a user with the provided email already exists.
		 * If yes, send an OTP to verify and log in the user.
		 */
		$existing_user = get_user_by( 'email', $register_cred['user_email'] );
		if ( $existing_user ) {
			// Send the OTP email
			ere_send_otp_email( $existing_user->ID );

			// Redirect to OTP form
			wp_safe_redirect( add_query_arg( [ 'user_id' => $existing_user->ID ], $otp_page_link ) );
			exit;
		}

		/**
		 * Register a new user if no user exists with the provided email.
		 */
		if ( username_exists( $register_cred['user_login'] ) ) {
			// Append a random suffix to avoid conflicts with existing usernames
			$register_cred['user_login'] = $register_cred['user_login'] . '_' . wp_rand( 100, 10000 );
		}

		// Insert the new user
		$user_id = wp_insert_user( $register_cred );

		if ( ! is_wp_error( $user_id ) ) {
			// Add profile image if provided
			if ( ! empty( $register_cred['profile_image'] ) ) {
				$profile_image_id = ere_insert_social_profile_image(
					$register_cred['profile_image'],
					$register_cred['user_login'] . wp_rand( 100, 1000 ) . '.png'
				);
				update_user_meta( $user_id, 'profile_image_id', $profile_image_id );
			}

			// Send OTP email
			ere_send_otp_email( $user_id );

			// Redirect to OTP form
			wp_safe_redirect( add_query_arg( [ 'user_id' => $user_id ], $otp_page_link ) );
			exit;
		}

		// If there's an error in user registration, display a generic error
		wp_die( esc_html__( 'An error occurred during registration. Please try again.', ERE_TEXT_DOMAIN ) );
	}
}

if ( ! function_exists( 'ere_social_login_redirect' ) ) {
	/**
	 * Redirect user to edit profile page if available, otherwise to homepage.
	 */
	function ere_social_login_redirect() {
		$edit_profile_page_url = '';

		if ( function_exists( 'realhomes_get_dashboard_page_url' ) ) {
			if ( realhomes_get_dashboard_page_url() && realhomes_dashboard_module_enabled( 'inspiry_profile_module_display' ) ) {
				$edit_profile_page_url = realhomes_get_dashboard_page_url( 'profile' );
			}
		}

		if ( $edit_profile_page_url ) {
			wp_safe_redirect( $edit_profile_page_url );
		} else {
			wp_safe_redirect( home_url() );
		}
	}
}

if ( ! function_exists( 'ere_insert_social_profile_image' ) ) {
	/**
	 * Insert an image to the WordPress library from given image url.
	 *
	 * @param string $image_url URL of the image that needs to be inserted.
	 * @param string $filename  The name of image file.
	 *
	 * @return int    $attached_id ID of the image that has been inserted.
	 */
	function ere_insert_social_profile_image( $image_url, $filename = '' ) {

		$upload_dir = wp_upload_dir();
		$image_data = file_get_contents( $image_url );

		if ( empty( $filename ) ) {
			$filename = basename( $image_url );
		}

		if ( wp_mkdir_p( $upload_dir['path'] ) ) {
			$file = $upload_dir['path'] . '/' . $filename;
		} else {
			$file = $upload_dir['basedir'] . '/' . $filename;
		}

		file_put_contents( $file, $image_data );

		// $wp_filetype = wp_check_filetype( $filename, null );
		$wp_filetype['type'] = 'image/png';

		$attachment = array(
			'post_mime_type' => $wp_filetype['type'],
			'post_title'     => sanitize_file_name( $filename ),
			'post_content'   => '',
			'post_status'    => 'inherit',
		);

		$attach_id = wp_insert_attachment( $attachment, $file );
		require_once ABSPATH . 'wp-admin/includes/image.php';
		$attach_data = wp_generate_attachment_metadata( $attach_id, $file );
		wp_update_attachment_metadata( $attach_id, $attach_data );

		return $attach_id;
	}
}


if ( ! function_exists( 'ere_send_otp_email' ) ) {
	/**
	 * Send OTP Email to User
	 *
	 * This function generates a 6-digit OTP code, stores it in the user's metadata with a
	 * 10-minute expiration, and sends the OTP to the user's registered email address.
	 *
	 * @param int $user_id The ID of the user to whom the OTP should be sent.
	 *
	 * @return void
	 */
	function ere_send_otp_email( $user_id ) {
		// Generate a more memorable OTP, avoiding leading zeroes and using patterns
		$otp = ere_generate_memorable_otp();

		// Store the OTP in user meta with expiration (e.g., 5 minutes)
		update_user_meta( $user_id, 'ere_social_otp_code', $otp );
		update_user_meta( $user_id, 'ere_social_otp_expiry', time() + 300 ); // 5 minutes validity

		// Get user data
		$user       = get_userdata( $user_id );
		$email_body = array();

		// Email subject and message
		$subject = esc_html__( 'Your OTP Code', ERE_TEXT_DOMAIN );


		$headers   = array();
		$headers[] = 'Content-Type: text/html; charset=UTF-8';
		$headers   = apply_filters( 'realhomes_verify_otp_email_header', $headers );

		// Build email contents
		$mail_content   = array();
		$mail_content[] = array(
			'name'  => '',
			'value' => '<h3>' . esc_html__( 'Social login request OTP', ERE_TEXT_DOMAIN ) . '</h3>'
		);
		$mail_content[] = array(
			'name'  => '',
			'value' => sprintf(
				esc_html__( "Hi %s,\n\nYour OTP code is: %s\n\nThis code will expire in 5 minutes.\n\nIf you did not request this, please ignore this email.\n\nThanks!", ERE_TEXT_DOMAIN ),
				$user->display_name,
				$otp
			)
		);

		$mail_content = ere_email_template( $mail_content, 'ere-otp-verification-form' );

		// Send the OTP email
		wp_mail( $user->user_email, $subject, $mail_content, $headers );
	}
}


if ( ! function_exists( 'ere_show_otp_form' ) ) {
	/**
	 * Display OTP Verification Form
	 *
	 * This function renders an OTP verification form with six input fields for digits.
	 * It includes a countdown timer, a resend link, and buttons for verification or cancellation.
	 *
	 * @return void
	 */
	function ere_show_otp_form() {
		$user_id    = isset( $_GET['user_id'] ) ? absint( $_GET['user_id'] ) : 0;
		$otp_expiry = $user_id ? get_user_meta( $user_id, 'ere_social_otp_expiry', true ) : 0;

		// Calculate remaining time in seconds
		$remaining_time = max( 0, $otp_expiry - time() );

		// Convert to minutes:seconds format
		$minutes       = floor( $remaining_time / 60 );
		$seconds       = $remaining_time % 60;
		$timer_display = sprintf( "%02d:%02d", $minutes, $seconds );
		?>
        <form id="ere-otp-verification-form" method="POST" action="">
            <!-- OTP Input Fields -->
            <div class="otp-inputs">
				<?php for ( $i = 1; $i <= 6; $i++ ) : ?>
                    <input type="text" name="ere_otp_digit_<?php echo esc_attr( $i ); ?>" maxlength="1" required pattern="[0-9]" inputmode="numeric" autocomplete="off" />
				<?php endfor; ?>
            </div>

            <!-- Countdown Timer and Resend Link -->
            <p class="countdown" data-expiry-time="<?php echo esc_attr( $otp_expiry ); ?>" data-current-time="<?php echo time(); ?>" data-expired-message="<?php esc_attr_e( 'Your OTP has expired, go back and try again!', ERE_TEXT_DOMAIN ); ?>">
				<?php esc_html_e( 'Remaining time: ', ERE_TEXT_DOMAIN ); ?>
                <span id="otp-timer"><?php echo esc_html( $timer_display ); ?></span> <br />
            </p>

			<?php
			// Add nonce for security
			wp_nonce_field( 'ere_verify_otp', 'ere_otp_nonce' );
			?>

            <!-- Button Group -->
            <div class="button-group">
                <button type="submit" name="ere_social_verify_otp" class="btn verify">
					<?php esc_html_e( 'Verify', ERE_TEXT_DOMAIN ); ?>
                </button>
                <button type="button" name="ere_social_cancel_otp" class="btn cancel" data-otp-home-url="<?php echo esc_url( home_url( '/' ) ); ?>">
					<?php esc_html_e( 'Cancel', ERE_TEXT_DOMAIN ); ?>
                </button>
            </div>

            <!-- Pass user ID if available -->
			<?php
			if ( $user_id ) {
				?>
                <input type="hidden" id="ere_otp_user_id" name="ere_otp_user_id" value="<?php echo esc_attr( $user_id ); ?>" />
				<?php
			}
			?>
        </form>
		<?php
	}
}

if ( ! function_exists( 'ere_verify_social_otp' ) ) {
	function ere_verify_social_otp() {

		if ( 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
			wp_send_json_error( 'Invalid request method' );
		}

		if ( empty( $_POST['user_id'] ) ) {
			wp_send_json_error( esc_html__( 'Invalid request. User ID is missing.', ERE_TEXT_DOMAIN ) );
		}

		$user_id      = absint( $_POST['user_id'] );
		$user_ip      = $_SERVER['REMOTE_ADDR'];
		$attempt_key  = 'ere_otp_attempts_' . md5( $user_ip );
		$attempts     = (int)get_transient( $attempt_key );
		$max_attempts = (int) get_option( 'ere_otp_max_login_attempts', 5 );

		// Check attempt limit
		if ( $attempts >= $max_attempts ) {
			wp_send_json_error( esc_html__( 'Too many incorrect attempts. Please try again later.', ERE_TEXT_DOMAIN ) );
		}

		// Verify nonce
		if ( ! isset( $_POST['ere_otp_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( $_POST['ere_otp_nonce'] ), 'ere_verify_otp' ) ) {
			wp_send_json_error( esc_html__( 'Security verification failed.', ERE_TEXT_DOMAIN ) );
		}

		// Collect OTP
		$otp = '';
		for ( $i = 1; $i <= 6; $i++ ) {
			if ( empty( $_POST["ere_otp_digit_$i"] ) || ! is_numeric( $_POST["ere_otp_digit_$i"] ) ) {
				// Increment failed attempt
				set_transient( $attempt_key, $attempts + 1, 6 * HOUR_IN_SECONDS );
				wp_send_json_error( esc_html__( 'Please enter all OTP digits.', ERE_TEXT_DOMAIN ) );
			}
			$otp .= sanitize_text_field( $_POST["ere_otp_digit_$i"] );
		}

		$stored_otp  = get_user_meta( $user_id, 'ere_social_otp_code', true );
		$expiry_time = get_user_meta( $user_id, 'ere_social_otp_expiry', true );

		if ( $stored_otp && $otp === $stored_otp && time() <= $expiry_time ) {
			// Reset attempts on success
			delete_transient( $attempt_key );

			wp_clear_auth_cookie();
			wp_set_current_user( $user_id );
			wp_set_auth_cookie( $user_id );

			delete_user_meta( $user_id, 'ere_social_otp_code' );
			delete_user_meta( $user_id, 'ere_social_otp_expiry' );

			wp_send_json_success( array(
				'message'            => esc_html__( 'OTP Verified.', ERE_TEXT_DOMAIN ),
				'remaining_attempts' => $max_attempts
			) );
		} else {
			// Failed attempt
			set_transient( $attempt_key, $attempts + 1, 6 * HOUR_IN_SECONDS );
			$remaining_attempts = max( 0, $max_attempts - ++$attempts );
			wp_send_json_error( array(
				'message'            => sprintf( esc_html__( 'Invalid or expired OTP. %d attempts remaining. Please try again.', ERE_TEXT_DOMAIN ), $remaining_attempts ),
				'remaining_attempts' => $remaining_attempts
			) );
		}
	}

	add_action( 'wp_ajax_nopriv_ere_verify_social_otp', 'ere_verify_social_otp' );
}


if ( ! function_exists( 'ere_generate_memorable_otp' ) ) {
	/**
	 * Generates a 6-digit OTP where three digits are random, and the others follow a memorable pattern.
	 *
	 * This function creates an OTP with a balance of security and usability by including three random digits
	 * and three digits from a predefined set of memorable patterns.
	 *
	 * @return string Generated OTP.
	 */
	function ere_generate_memorable_otp() {
		// Predefined memorable patterns (3 digits) avoiding zero
		$patterns = [ '123', '456', '789', '147', '258', '369', '159', '753', '951', '852', '753', '951', '258', '147' ];

		// Randomly pick a 3-digit pattern
		$memorable_part = $patterns[ wp_rand( 0, count( $patterns ) - 1 ) ];

		// Generate three random digits avoiding zero
		$random_part = '';
		for ( $i = 0; $i < 3; $i++ ) {
			$random_part .= wp_rand( 1, 9 );
		}

		// Combine the memorable part and random part in random positions
		$combined_otp = wp_rand( 0, 1 ) ? $memorable_part . $random_part : $random_part . $memorable_part;

		return $combined_otp;
	}
}