<?php
/**
 * This file contains Property_Analytics class and initializing function.
 *
 * @since   3.10
 * @package easy_real_estate
 */

/**
 * Provide methods to get information about Operating System and Browser.
 */
require_once ERE_PLUGIN_DIR . 'includes/property-analytics/class-operating-system-browser.php';

/**
 * Property_Analytics class is responsible to collect properties' views data.
 */
class Property_Analytics {

	/**
	 * ID of the property to which a view data has to set.
	 *
	 * @var $property_id
	 */
	protected $property_id = null;

	/**
	 * Constructor - for the initial setup of property analytics.
	 */
	public function __construct() {

	}

	/**
	 * Insert property view to the database.
	 *
	 * @param int $property_id
	 *
	 * @return mixed
	 */
	public function insert_property_view( $property_id = null ) {

		global $wpdb;
		$os_br = new Operating_System_Browser();

		$time_stamp         = current_time( 'timestamp' );
		$user_ip            = $this->get_the_user_ip();
		$browser            = $os_br->show_info( 'browser' );
		$os                 = $os_br->show_info( 'os' );
		$device             = $os_br->show_info( 'device' );
		$property_author_id = get_post_field( 'post_author', $property_id );
		$property_viewer_id = 0;

		if ( is_user_logged_in() ) {
			$property_viewer_id = get_current_user_id();
		}

		if ( empty( $property_id ) || empty( $time_stamp ) || empty( $user_ip ) ) {
			return;
		} else {
			$country = empty( $this->get_the_country_by_ip( $user_ip ) ) ? '' : $this->get_the_country_by_ip( $user_ip );
			if ( empty( $country ) ) {
				$country = 'Unknown';
			}
		}

		$wpdb->insert(
			"{$wpdb->prefix}inspiry_property_analytics",
			array(
				'PID'                => $property_id,
				'TimeStamp'          => $time_stamp,
				'IP'                 => $user_ip,
				'Browser'            => $browser,
				'OS'                 => $os,
				'Country'            => $country,
				'Device'             => $device,
				'Property_Author_ID' => $property_author_id,
				'Property_Viewer_ID' => $property_viewer_id
			),
			array(
				'%d',    // PID value type.
				'%s',    // TimeStamp value type.
				'%s',    // IP value type.
				'%s',    // Browser value type.
				'%s',    // OS value type.
				'%s',    // Country value type.
				'%s',    // Device value type.
				'%s',    // Property Author ID
				'%s'    // Property Viewer ID
			)
		); // db call ok.

		return $wpdb->insert_id;
	}

	/**
	 * Returns current user's IP address
	 */
	public function get_the_user_ip() {

		$user_ip = '';

		if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
			// check ip from share internet.
			$user_ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
		} else if ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			// to check ip is pass from proxy.
			$user_ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
		} else if ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
			$user_ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
		}

		if ( ! empty( $user_ip ) ) {
			$user_ip = explode( ',', $user_ip );
			$ip      = $user_ip[0];
		}

		return $ip;
	}

	/**
	 * Returns the Country againts given IP address.
	 *
	 * @param string $user_ip User ip address.
	 *
	 * @return string
	 */
	public function get_the_country_by_ip( $user_ip ) {
		$country_info = unserialize( file_get_contents( 'http://www.geoplugin.net/php.gp?ip=' . sanitize_text_field( $user_ip ) ) );

		if ( ! empty( $country_info['geoplugin_countryName'] ) ) {
			return $country_info['geoplugin_countryName']; // return country name from available country information.
		}

		return '';
	}
}

/**
 * Initialize Property Analytics Function
 */
function ere_initialize_property_analytics() {
	new Property_Analytics();
}

add_action( 'wp', 'ere_initialize_property_analytics' );


if ( ! function_exists( 'ere_create_property_analytics_table' ) ) {
	/**
	 * Create or update the required database table.
	 * Only runs on dashboard template or single property with analytics enabled.
	 */
	function ere_create_property_analytics_table() {
		global $wpdb;

		// Check if we're on the dashboard template or single property with analytics enabled
		$is_dashboard       = is_page_template( 'dashboard.php' );
		$is_single_property = is_singular( 'property' ) && 'enabled' === get_option( 'inspiry_property_analytics_status', 'enabled' );

		if ( ! $is_dashboard && ! $is_single_property ) {
			return;
		}

		$table_name = $wpdb->prefix . 'inspiry_property_analytics';

		// Check if table already exists
		if ( $wpdb->get_var( "SHOW TABLES LIKE '$table_name'" ) == $table_name ) {
			return;
		}

		$charset_collate = $wpdb->get_charset_collate();

		// Full table schema
		$sql_query = "CREATE TABLE $table_name (
            RID INT(20) NOT NULL AUTO_INCREMENT,
            PID INT(20) NOT NULL,
            TimeStamp VARCHAR(25) NOT NULL,
            IP VARCHAR(20),
            Browser VARCHAR(20),
            OS VARCHAR(20),
            Country VARCHAR(60),
            Device VARCHAR(60),
            Property_Author_ID INT(20),
            Property_Viewer_ID INT(20),
            UNIQUE KEY id (RID)
        ) $charset_collate;";

		// Use dbDelta to create or update the table
		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql_query );

		// Debugging: Check for errors
		if ( ! empty( $wpdb->last_error ) ) {
			error_log( 'Error creating or updating table: ' . $wpdb->last_error );
		}
	}

	// Hook into WordPress initialization with lower priority
	add_action( 'wp', 'ere_create_property_analytics_table', 20 );
}


if ( ! function_exists( 'ere_update_property_author_ids' ) ) {
	/**
	 * Update Property_Author_ID for older rows in the analytics table.
	 */
	function ere_update_property_author_ids() {
		global $wpdb;

		// Check if the update has already been performed.
		$update_flag = get_option( 'ere_property_author_update_done' );

		if ( $update_flag ) {
			error_log( 'Update function skipped: Update already performed.' );

			return;
		}

		// Table name
		$table_name = $wpdb->prefix . 'inspiry_property_analytics';

		// Get all rows where Property_Author_ID is still NULL or 0
		$rows = $wpdb->get_results( "SELECT RID, PID FROM $table_name WHERE Property_Author_ID IS NULL OR Property_Author_ID = 0" );

		if ( empty( $rows ) ) {
			error_log( 'No rows to update: All records are already updated.' );
		} else {
			foreach ( $rows as $row ) {
				$property_id = $row->PID;
				$record_id   = $row->RID;

				// Fetch property author ID
				$property_author_id = get_post_field( 'post_author', $property_id );

				// Log what’s happening
				if ( $property_author_id ) {
					error_log( "Updating RID $record_id: Property ID $property_id, Author ID $property_author_id" );

					// Update the Property_Author_ID field
					$result = $wpdb->update(
						$table_name,
						array( 'Property_Author_ID' => $property_author_id ),
						array( 'RID' => $record_id ),
						array( '%d' ),
						array( '%d' )
					);

					// Log the result of the update
					if ( false === $result ) {
						error_log( "Failed to update RID $record_id: " . $wpdb->last_error );
					} else {
						error_log( "Successfully updated RID $record_id." );
					}
				} else {
					error_log( "Failed to fetch author for PID $property_id." );
				}
			}
		}

		// Mark the update as done
		update_option( 'ere_property_author_update_done', true );
		error_log( 'Update function completed successfully.' );
	}

	// Hook into theme activation
	add_action( 'after_switch_theme', 'ere_update_property_author_ids' );
}
