<?php
/**
 * This file contains Property_Analytics_View class and 'inspiry_get_property_views' Ajax handler
 * to display property analytics data.
 *
 * @since   3.10
 * @package easy_real_estate
 */

/**
 * Property_Analytics_View class is responsible to display properties' views data.
 */
class Property_Analytics_View {

	/**
	 * ID of the property that will be used as 'key' to reterive property views data.
	 *
	 * @var int $property_id
	 */
	private $property_id;

	/**
	 * Constructor function that set the class data variable on instantiating.
	 *
	 * @param int $property_id ID of the property that will be used as 'key' to reterive property views data.
	 *
	 * @return void
	 */
	public function __construct( $property_id = 0 ) {
		if ( ! empty( $property_id ) ) {
			$this->property_id = $property_id;
		} else {
			global $wp_query;
			if ( is_object( $wp_query->post ) ) {
				$this->property_id = $wp_query->post;
			}
		}
	}

	/**
	 * Return views based on given Type and Date.
	 *
	 * @param string $type Property views type that's required. By default all views will be returned.
	 * @param string $date The date for which property views are required. By default views will be returned for all dates.
	 *
	 * @return string Count of property views.
	 */
	public function get_views( $type = 'all', $date = '' ) {

		global $wpdb;
		$table_name  = $wpdb->prefix . "inspiry_property_analytics";
		$property_id = $this->property_id;
		$date        = empty( $date ) ? time() : $date; // Use current timestamp if date is empty.

		switch ( $type ) {
			case 'property_unique_views':
				// Getting all unique views
				$views = $this->get_views_by_period( array(
					'time_start'         => date( 'Y-m-d 00:00:00', $date ),
					'time_end'           => date( 'Y-m-d 23:59:59', $date ),
					'post_id'            => $property_id,
					'unique'             => true
				) );

				// Access unique views from the result.
				$result = $views->UniqueIPCount ?? 0;
				break;

			case 'property_today_views':
				// Today's views for a property
				$views = $this->get_views_by_period( array(
					'time_start'         => date( 'Y-m-d 00:00:00', $date ),
					'time_end'           => date( 'Y-m-d 23:59:59', $date ),
					'post_id'            => $property_id,
				) );

				$result = $views->TotalViews ?? 0;
				break;

			case 'this_week_views':
				// This week's views for a property
				$views = $this->get_views_by_period( array(
					'time_start'         => date( 'Y-m-d 00:00:00', strtotime( 'monday this week', $date ) ),
					'time_end'           => date( 'Y-m-d 23:59:59', strtotime( 'sunday this week', $date ) ),
					'post_id'            => $property_id,
				) );

				$result = $views->TotalViews ?? 0;
				break;

			case 'two_weeks_views':
				// Views from the last two weeks
				$views = $this->get_views_by_period( array(
					'time_start'         => date( 'Y-m-d 00:00:00', strtotime( '-14 days', $date ) ),
					'time_end'           => date( 'Y-m-d 23:59:59', $date ),
					'post_id'            => $property_id,
				) );

				$result = $views->TotalViews ?? 0;
				break;

			case 'this_month_views':
				// Views for the current month
				$views = $this->get_views_by_period( array(
					'time_start'         => date( 'Y-m-01 00:00:00', $date ), // First day of the month,
					'time_end'           => date( 'Y-m-t 23:59:59', $date ), // Last day of the month
					'post_id'            => $property_id,
				) );

				$result = $views->TotalViews ?? 0;
				break;

			case 'property_total_views':
				// All-time views for the property
				$result = $wpdb->get_var(
					$wpdb->prepare(
						"SELECT COUNT(*) FROM {$table_name} WHERE PID = %d",
						$property_id
					)
				);
				break;

			case 'property_views_history':
				// All analytics records for the property
				$result = $wpdb->get_results(
					$wpdb->prepare(
						"SELECT * FROM {$table_name} WHERE PID = %d",
						$property_id
					)
				);
				break;

			default:
				// Total views for all properties
				$result = $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );
				break;
		}

		return $result;
	}

	/**
	 * Retrieve property views for a given time period.
	 *
	 * This function fetches the number of views for properties within a specified time period.
	 * Additional filters can be applied using the `$args` parameter, such as property ID,
	 * property author ID, or viewer ID.
	 *
	 * @param array $args {
	 *     Array of details to filter views and define the time period.
	 *
	 *     @type string $time_start Start time of the period in 'Y-m-d H:i:s' format (required).
	 *     @type string $time_end   End time of the period in 'Y-m-d H:i:s' format (required).
	 *     @type bool   $unique     Whether to count unique views by IP. Default false.
	 *     @type int    $post_id    Specific property ID to filter views. Default 0.
	 *     @type int    $author_id  Property author's ID. Default 0.
	 *     @type int    $property_viewer_id Viewer ID to filter views. Default 0.
	 * }
	 *
	 * @return array|null The views data retrieved from the database or null on failure.
	 */
	private function get_views_by_period( $args = [] ) {
		// Provide default values for $args
		$defaults = [
			'time_start'         => '',
			'time_end'           => '',
			'unique'             => false,
			'post_id'            => 0,
			'author_id'          => 0,
			'property_viewer_id' => 0,
		];

		// Merge the provided details with defaults
		$args = wp_parse_args( $args, $defaults );

		// Validate and sanitize start and end times
		$time_start = strtotime( $args['time_start'] );
		$time_end   = strtotime( $args['time_end'] );

		// Return early if timestamps are invalid
		if ( ! $time_start || ! $time_end ) {
			return null; // Or throw an exception based on error handling strategy
		}

		// Update $args with processed time values
		$args['time_start'] = $time_start;
		$args['time_end']   = $time_end;

		// Return the views data using the helper function
		return ere_get_properties_by_time_period( $args );
	}

}

if ( ! function_exists( 'inspiry_get_property_views' ) ) {
	/**
	 * Ajax request handler to fulfill requested property
	 * views result based on given property ID.
	 */
	function inspiry_get_property_views() {

		// Security verification.
		if ( empty( $_REQUEST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['nonce'] ) ), 'ere-property-analytics' ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Security verification failed, please refresh the page and try again.', ERE_TEXT_DOMAIN ) ] );
		}

		// Ensure property_id is provided.
		if ( isset( $_POST['property_id'] ) && ! empty( $_POST['property_id'] ) ) {

			$property_id = sanitize_text_field( wp_unslash( $_POST['property_id'] ) );

			// Add a view to the property of given ID.
			$property_view_class = new Property_Analytics();
			$property_view_rid   = $property_view_class->insert_property_view( $property_id );
			$property_views      = new Property_Analytics_View( $property_id );
			$views               = (array)$property_views->get_views( 'property_views_history' );
			$view_time           = array();

			// Format the timestamps for display.
			foreach ( $views as $view ) {
				$view              = (array)$view;
				$view['TimeStamp'] = explode( ' ', $view['TimeStamp'] );
				$view_time[]       = date( 'd-M-Y', intval( $view['TimeStamp'][0] ) );
			}

			$view_time = array_values( array_unique( $view_time ) );

			// Prepare views for the last X days (default: 14).
			$total_views = array();
			$num_of_days = get_option( 'inspiry_property_analytics_time_period', 14 );
			$view_time   = array_slice( $view_time, -( intval( $num_of_days ) ) );
			foreach ( $view_time as $time_stamp ) {
				$time_stamp    = strtotime( $time_stamp );
				$total_views[] = $property_views->get_views( 'property_today_views', $time_stamp );
			}

			// Respond with success if the view was saved.
			if ( 0 < intval( $property_view_rid ) ) {
				wp_send_json_success( [
					'rid'   => $property_view_rid,
					'dates' => $view_time,
					'views' => $total_views
				] );
			} else {
				wp_send_json_error( [ 'message' => esc_html__( 'Failed to save property view.', ERE_TEXT_DOMAIN ) ] );
			}
		}

		// If property_id is missing.
		wp_send_json_error( [ 'message' => esc_html__( 'Property ID is missing.', ERE_TEXT_DOMAIN ) ] );
	}

	add_action( 'wp_ajax_inspiry_property_views', 'inspiry_get_property_views' );
	add_action( 'wp_ajax_nopriv_inspiry_property_views', 'inspiry_get_property_views' );
}

if ( ! function_exists( 'inspiry_get_property_summed_views' ) ) {
	/**
	 * Return property summed up views for the configured timestamp.
	 */
	function inspiry_get_property_summed_views( $property_id ) {

		$property_transient_key = 'ere_property_' . $property_id . '_views_detail';

		// Getting the existing copy of property views transient data
		if ( false === ( $property_views = get_transient( $property_transient_key ) ) ) {

			$property_views = array();

			// It wasn't there, so regenerate the data and save the transient
			$property_views_obj         = new Property_Analytics_View( $property_id );
			$property_views['today']    = (int) ($property_views_obj->get_views( 'property_today_views' ) ?? 0);
			$property_views['week']     = (int) ($property_views_obj->get_views( 'this_week_views' ) ?? 0);
			$property_views['two_week'] = (int) ($property_views_obj->get_views( 'two_weeks_views' ) ?? 0);
			$property_views['month']    = (int) ($property_views_obj->get_views( 'this_month_views' ) ?? 0);
			$property_views['all']      = (int) ($property_views_obj->get_views( 'property_total_views' ) ?? 0);

			// Setting up the transient views detail value for this property
			delete_transient( $property_transient_key );
		}

		$time_period = get_option( 'inspiry_property_analytics_time_period', 14 );
		switch ( $time_period ) {
			case 1:
				$return_views = $property_views['today'];
				break;
			case 7:
				$return_views = $property_views['week'];
				break;
			case 14:
				$return_views = $property_views['two_week'];
				break;
			case 30:
				$return_views = $property_views['month'];
				break;
			default:
				$return_views = $property_views['all'];
		}

		return intval( $return_views );
	}
}

if ( ! function_exists( 'ere_get_properties_by_view_count' ) ) {
	/**
	 * Return properties array by views count
	 *
	 * @since 2.4.0
	 *
	 * @param int $num Mention how many properties are required. Will return all if empty
	 *
	 * @return string
	 */
	function ere_get_properties_by_view_count( $num = -1, $author_id = 0 ) {
		global $wpdb;

		// Base query
		$sql_query = "SELECT PID, COUNT(PID) AS PID_Count FROM {$wpdb->prefix}inspiry_property_analytics";

		// Add condition for author ID if provided
		$conditions = [];
		if ( intval( $author_id ) > 0 ) {
			$conditions[] = $wpdb->prepare( "Property_Author_ID = %d", $author_id );
		}

		// Append WHERE clause if there are conditions
		if ( ! empty( $conditions ) ) {
			$sql_query .= ' WHERE ' . implode( ' AND ', $conditions );
		}

		// Grouping and ordering
		$sql_query .= " GROUP BY PID ORDER BY PID_Count DESC";

		// Add LIMIT clause if $num is specified
		if ( intval( $num ) > 0 ) {
			$sql_query .= " LIMIT " . intval( $num );
		}

		return $wpdb->get_results( $sql_query );
	}

}

if ( ! function_exists( 'ere_get_properties_by_time_period' ) ) {
	/**
	 * Return properties IDs from a given time till date.
	 *
	 * @since 2.4.0
	 *
	 * @param array $args This function manages the following arguments:
	 *                    - unique (bool): If the return values should be unique by IP.
	 *                    - time_start (int): Unix timestamp int value for starting time range.
	 *                    - time_end (int): Unix timestamp int value for ending time range.
	 *                    - post_id (int): Specific property post ID.
	 *                    - property_author_id (int): Specific property author ID.
	 *                    - property_viewer_id (int): Specific property viewer ID.
	 *
	 * @return array Array of values from the table.
	 */
	function ere_get_properties_by_time_period( $args = array() ) {

		$unique             = ! empty( $args['unique'] );
		$timestamp_start    = ! empty( $args['time_start'] ) ? $args['time_start'] : 0;
		$timestamp_end      = ! empty( $args['time_end'] ) ? $args['time_end'] : 0;
		$post_id            = ! empty( $args['post_id'] ) ? $args['post_id'] : 0;
		$property_author_id = ! empty( $args['author_id'] ) ? $args['author_id'] : 0;
		$property_viewer_id = ! empty( $args['viewer_id'] ) ? $args['viewer_id'] : 0;

		global $wpdb;
		$table_name = $wpdb->prefix . 'inspiry_property_analytics';

		// Start constructing the SQL query
		$sql_query = "SELECT COUNT(*) AS TotalViews";

		if ( $unique ) {
			$sql_query .= ", COUNT(DISTINCT IP) AS UniqueViews";
		}

		$sql_query .= " FROM $table_name";

		$conditions = array();

		// Add condition for $post_id if provided
		if ( $post_id ) {
			$conditions[] = $wpdb->prepare( "PID = %d", $post_id );
		}

		// Add conditions for time period
		if ( $timestamp_start > 0 && $timestamp_end > 0 ) {
			$conditions[] = $wpdb->prepare( "TimeStamp >= %d AND TimeStamp <= %d", $timestamp_start, $timestamp_end );
		} else if ( $timestamp_start > 0 ) {
			$conditions[] = $wpdb->prepare( "TimeStamp >= %d", $timestamp_start );
		} else if ( $timestamp_end > 0 ) {
			$conditions[] = $wpdb->prepare( "TimeStamp <= %d", $timestamp_end );
		}

		// Add condition for property_author_id if provided
		if ( $property_author_id ) {
			$conditions[] = $wpdb->prepare( "Property_Author_ID = %d", $property_author_id );
		}

		// Add condition for property_viewer_id if provided
		if ( $property_viewer_id ) {
			$conditions[] = $wpdb->prepare( "Property_Viewer_ID = %d", $property_viewer_id );
		}

		// Combine conditions
		if ( ! empty( $conditions ) ) {
			$sql_query .= ' WHERE ' . implode( ' AND ', $conditions );
		}

		// Execute the query and return a single result
		return $wpdb->get_row( $sql_query );
	}
}


if ( ! function_exists( 'ere_get_all_time_views' ) ) {
	/**
	 * Return all time (total) property views for a website
	 *
	 * @since 2.4.1
	 *
	 * @return array    Array of values from the table
	 */
	function ere_get_all_time_views() {

		global $wpdb;
		$table_name = $wpdb->prefix . 'inspiry_property_analytics';

		$query       = "SHOW TABLE STATUS WHERE Name = '$table_name'";
		$table_stats = $wpdb->get_results( $query );

		if ( ! $table_stats ) {
			// Query failed, display error message
			return "Error: " . $wpdb->last_error;
		} else {
			return $table_stats;
		}
	}
}

if ( ! function_exists( 'ere_get_analytics_views_sorted_data' ) ) {
	/**
	 * Return properties IDs from given time till date
	 *
	 * @since 2.4.0
	 *
	 * @param string $args          It contains the following arguments
	 *                              $column:    (string)    Name of the column in analytics table. Default: Country
	 *                              $limit:     (int)       Limit of the top most counted values. Default: 4
	 *                              $others:    (boolean)   If rest of the remaining values should be shown
	 *                              combined in 'Others'
	 *
	 * @return array
	 */
	function ere_get_analytics_views_sorted_data( $args = array() ) {
		$column        = 'Country';
		$limit         = 4;
		$includeOthers = false;
		$others_label  = esc_html__( 'Others', ERE_TEXT_DOMAIN );
		$post_id       = 0;
		$author_id     = 0;

		if ( is_array( $args ) && 0 < count( $args ) ) {
			$column        = isset( $args['column'] ) ? $args['column'] : 'Country';
			$limit         = isset( $args['limit'] ) ? $args['limit'] : 4;
			$includeOthers = $args['others'] ?? false;
			$others_label  = isset( $args['others_label'] ) ? $args['others_label'] : esc_html__( 'Others', ERE_TEXT_DOMAIN );
			$post_id       = isset( $args['post_id'] ) ? intval( $args['post_id'] ) : 0;
			$author_id     = isset( $args['author_id'] ) ? intval( $args['author_id'] ) : 0;
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'inspiry_property_analytics';

		if ( ! $includeOthers ) {
			$sql_query = "SELECT $column, COUNT(*) AS ValueCount FROM $table_name";

			// Add conditions for post_id or author_id
			$conditions = [];
			if ( $post_id ) {
				$conditions[] = $wpdb->prepare( "PID = %d", $post_id );
			}
			if ( $author_id ) {
				$conditions[] = $wpdb->prepare( "Property_Author_ID = %d", $author_id );
			}

			// Combine conditions if present
			if ( ! empty( $conditions ) ) {
				$sql_query .= ' WHERE ' . implode( ' AND ', $conditions );
			}

			$sql_query .= " GROUP BY $column ORDER BY ValueCount DESC LIMIT $limit";

			$results = $wpdb->get_results( $sql_query );
		} else {
			$sql_query = "WITH RankedValues AS (
                                    SELECT
                                        $column,
                                        COUNT(*) AS ValueCount,
                                        ROW_NUMBER() OVER (ORDER BY COUNT(*) DESC) AS RowNum
                                    FROM
                                        $table_name";

			// Add conditions for post_id or author_id
			$conditions = [];
			if ( $post_id ) {
				$conditions[] = $wpdb->prepare( "PID = %d", $post_id );
			}
			if ( $author_id ) {
				$conditions[] = $wpdb->prepare( "Property_Author_ID = %d", $author_id );
			}

			// Combine conditions if present
			if ( ! empty( $conditions ) ) {
				$sql_query .= ' WHERE ' . implode( ' AND ', $conditions );
			}

			$sql_query .= " GROUP BY
                                $column
                                )
                                SELECT
                                    CASE
                                        WHEN RowNum <= $limit THEN $column
                                        " . ( $includeOthers ? "ELSE 'Others'" : "" ) . "
                                    END AS $column,
                                    SUM(ValueCount) AS ValueCount
                                FROM
                                    RankedValues
                                GROUP BY
                                    CASE
                                        WHEN RowNum <= $limit THEN $column
                                        " . ( $includeOthers ? "ELSE 'Others'" : "" ) . "
                                    END
                                ORDER BY
                                    ValueCount DESC;";

			$results = $wpdb->get_results( $sql_query );

			usort( $results, function ( $a, $b ) use ( $column, $others_label ) {
				return ere_custom_sort_analytics_chart_array( $a, $b, $column, $others_label );
			} );
		}

		return $results;
	}

}

if ( ! function_exists( 'ere_check_and_convert_timestamps_once' ) ) {
	/**
	 * This function converts standard time and date format in TimeStamp
	 * column of $wpdb inspiry_property_analytics column to a proper timestamp
	 * It runs first time only and then gets disabled to make sure it doesn't run
	 * on each load.
	 * It is required to fix the timestamps' column to be used in date manipulation for
	 * analytics.
	 *
	 * @since 2.4.0
	 *
	 */
	function ere_check_and_convert_timestamps_once() {
		// Check if the flag is set
		if ( get_option( 'timestamps_checked' ) ) {
			return; // The check has already been performed
		}

		global $wpdb;

		$table_name = $wpdb->prefix . 'inspiry_property_analytics';

		$sql_query = "
			UPDATE $table_name
			SET TimeStamp = UNIX_TIMESTAMP(STR_TO_DATE(TimeStamp, '%Y-%m-%d %H:%i:%s'))
			WHERE TimeStamp LIKE '____-__-__ __:__:__';
		";

		// Performing the conversion here
		$wpdb->query( $sql_query );

		// Set the flag to indicate that the check has been performed
		update_option( 'timestamps_checked', true );
	}

	add_action( 'init', 'ere_check_and_convert_timestamps_once' );
}

if ( ! function_exists( 'ere_custom_sort_analytics_chart_array' ) ) {
	/**
	 * Function to manage a custom sort for an array using usort()
	 * It is sorting the array elements by associative key $column
	 * It is also keeping the 'others' column at the end
	 *
	 * @since 2.4.0
	 *
	 * @param mixed  $a
	 * @param mixed  $b
	 * @param string $column
	 * @param string $value
	 *
	 * @return mixed
	 */
	function ere_custom_sort_analytics_chart_array( $a, $b, $column, $value ) {
		if ( $a->$column === $value ) {
			return 1;
		} else if ( $b->$column === $value ) {
			return -1;
		} else {
			return $a->ValueCount > $b->ValueCount ? -1 : 1;
		}
	}
}

if ( ! function_exists( 'ere_get_properties_all_time_views' ) ) {
	/**
	 * Setting all time views to transient for better performance
	 *
	 * @since 2.4.1
	 *
	 * @return int
	 */
	function ere_get_properties_all_time_views() {
		$transient_key  = 'ere_properties_all_time_views';
		$all_time_views = get_transient( $transient_key );

		if ( false === $all_time_views ) {
			global $wpdb;
			$table_name = $wpdb->prefix . 'inspiry_property_analytics';

			$sql_query = "SELECT COUNT(*) AS TotalViews FROM $table_name";

			$all_time_views = $wpdb->get_var( $sql_query );

			// Set transient with total views
			set_transient( $transient_key, $all_time_views, HOUR_IN_SECONDS );
		}

		return $all_time_views;
	}
}

/**
 * Function to update analytics views property author ID table cells
 * in batches to manage the server load
 *
 * @since 2.2.6
 */
function ere_update_analytics_property_author_ids_in_batches() {
	global $wpdb;

	// Define the table name
	$table_name = $wpdb->prefix . 'inspiry_property_analytics';

	// Fetch a batch of rows where Property_Author_ID is NULL
	$batch_size = 100; // Adjust batch size as per server capacity
	$rows       = $wpdb->get_results(
		$wpdb->prepare(
			"SELECT RID, PID FROM $table_name WHERE Property_Author_ID IS NULL LIMIT %d",
			$batch_size
		)
	);

	if ( ! empty( $rows ) ) {
		foreach ( $rows as $row ) {
			$property_id = $row->PID;
			$rid         = $row->RID;

			// Get the post author from the WordPress posts table
			$author_id = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT post_author FROM {$wpdb->prefix}posts WHERE ID = %d",
					$property_id
				)
			);

			// Update the Property_Author_ID in the custom table
			if ( $author_id ) {
				$wpdb->update(
					$table_name,
					[ 'Property_Author_ID' => $author_id ],
					[ 'RID' => $rid ],
					[ '%d' ],
					[ '%d' ]
				);
			}
		}

		// Log progress (optional)
		error_log( 'Batch update completed for Property_Author_ID.' );
	} else {
		// Mark the process as complete
		update_option( 'ere_analytics_property_author_update_completed', true );
	}
}

/**
 * Schedule the function to run periodically
 */
if ( ! wp_next_scheduled( 'ere_update_analytics_property_author_ids' ) ) {
	wp_schedule_event( time(), 'hourly', 'ere_update_analytics_property_author_ids' );
}

add_action( 'ere_update_analytics_property_author_ids', 'ere_update_analytics_property_author_ids_in_batches' );

/**
 * Hook to run only once for older installations to update views property author IDs
 *
 * @since 2.2.6
 */
register_activation_hook( __FILE__, function () {
	if ( ! get_option( 'ere_analytics_property_author_update_completed' ) ) {
		update_option( 'ere_analytics_property_author_update_completed', false );
	}
} );
