<?php
/**
 * Helping functions related to real estate
 */

if ( ! function_exists( 'ere_get_property_statuses' ) ) {
	/**
	 * Display property status.
	 *
	 * @param int $post_id - Property ID.
	 */
	function ere_get_property_statuses( $post_id ) {

		$status_terms = get_the_terms( $post_id, 'property-status' );

		if ( ! empty( $status_terms ) && ! is_wp_error( $status_terms ) ) {

			$status_names = '';
			$status_count = 0;

			foreach ( $status_terms as $term ) {
				if ( $status_count > 0 ) {
					$status_names .= ', ';  /* add comma before the term namee of 2nd and any later term */
				}
				$status_names .= $term->name;
				$status_count++;
			}

			if ( ! empty( $status_names ) ) {
				return $status_names;
			}
		}

		return '';
	}
}
if ( ! function_exists( 'ere_get_property_tags' ) ) {
	/**
	 * Display property status.
	 *
	 * @param int $post_id - Property ID.
	 * @param string $class_name - CSS Class.
	 * @param string $taxonomy - Property Taxonomy.
	 */
	function ere_get_property_tags( $post_id , $class_name = 'rh-ultra-status', $taxonomy = 'property-status' ) {
		$status_terms = get_the_terms( $post_id, $taxonomy );
		if ( empty( $status_terms ) || is_wp_error( $status_terms ) ) {
            return;
        }

		foreach ( $status_terms as $term ) {
			$status_classes = sprintf( 'property-status-%s %s', $term->slug, $class_name );
			?>
			<a href="<?php echo esc_url( get_term_link( $term->term_id ) ) ?>" class="<?php echo esc_attr( $status_classes ) ?>"><?php echo esc_html( $term->name ); ?></a>
			<?php
		}
	}
}


if ( ! function_exists( 'ere_get_property_types' ) ) {
	/**
	 * Get property types
	 *
	 * @param $property_post_id
	 * @param bool $term_link
	 * @return string
	 */
	function ere_get_property_types( $property_post_id, $term_link = false ) {
		$type_terms = get_the_terms( $property_post_id, "property-type" );
		if ( ! empty( $type_terms ) ) {
			$type_count         = count( $type_terms );
			$property_types_str = '<small>';
			$loop_count         = 1;
			foreach ( $type_terms as $type_term ) {
				if ( true === $term_link ) {
					$property_types_str .= '<a href="' . get_term_link( $type_term->term_id ) . '">' . $type_term->name . '</a>';
				} else {
					$property_types_str .= $type_term->name;
				}
				if ( $loop_count < $type_count && $type_count > 1 ) {
					$property_types_str .= ', ';
				}
				$loop_count++;
			}
			$property_types_str .= '</small>';
		} else {
			$property_types_str = '&nbsp;';
		}

		return $property_types_str;
	}
}


if( !function_exists( 'ere_any_text' ) ) :
	/**
	 * Return text string for word 'Any'
	 *
	 * @return string
	 */
	function ere_any_text() {
		$ere_any_text = get_option( 'inspiry_any_text' );
		if ( $ere_any_text ) {
			return $ere_any_text;
		}
		return esc_html__( 'Any', ERE_TEXT_DOMAIN );
	}
endif;


if( !function_exists( 'ere_is_search_page_configured' ) ) :
	/**
	 * Check if search page settings are configured
	 */
	function ere_is_search_page_configured() {

		/* Check search page */
		$inspiry_search_page = get_option('inspiry_search_page');
		if ( ! empty( $inspiry_search_page ) ) {
			return true;
		}

		/* Check search url which is deprecated and this code is to provide backward compatibility */
		$theme_search_url = get_option('theme_search_url');
		if ( ! empty( $theme_search_url ) ) {
			return true;
		}

		/* Return false if all fails */
		return false;
	}
endif;

if ( ! function_exists( 'ere_skip_sticky_properties' ) ) :
	/**
	 * Skip sticky properties
	 */
	function ere_skip_sticky_properties(){
		$skip_sticky = get_option( 'inspiry_listing_skip_sticky', false );
		if ( $skip_sticky ) {
			remove_filter( 'the_posts', 'inspiry_make_properties_stick_at_top', 10 );
		}
    }
endif;

if ( ! function_exists( 'ere_skip_home_sticky_properties' ) ) :
	/**
	 * Skip sticky properties
	 */
	function ere_skip_home_sticky_properties(){
		$skip_sticky = get_post_meta( get_the_ID(), 'inspiry_home_skip_sticky', true );
		if ( $skip_sticky ) {
			remove_filter( 'the_posts', 'inspiry_make_properties_stick_at_top', 10 );
		}
    }
endif;


if ( ! function_exists( 'ere_get_figure_caption' ) ) {
	/**
	 * Figure caption based on property statuses
	 *
	 * @param $post_id
	 * @return string
	 */
	function ere_get_figure_caption( $post_id ) {
		$status_terms = get_the_terms( $post_id, "property-status" );
		if ( ! empty( $status_terms ) ) {
			$status_classes = '';
			$status_names = '';
			$status_count = 0;
			foreach ( $status_terms as $term ) {
				if ( $status_count > 0 ) {
					$status_names .= ', ';  /* add comma before the term namee of 2nd and any later term */
					$status_classes .= ' ';
				}
				$status_names .= $term->name;
				$status_classes .= $term->slug;
				$status_count++;
			}

			if ( ! empty( $status_names ) ) {
				return '<figcaption class="' . $status_classes . '">' . $status_names . '</figcaption>';
			}

			return '';
		}
	}
}


if ( ! function_exists( 'ere_display_figcaption' ) ) {
	/**
	 * Display figure caption for given property's post id
	 *
	 * @param $post_id
	 */
	function ere_display_figcaption( $post_id ) {
		echo ere_get_figure_caption( $post_id );
	}
}


if ( ! function_exists( 'ere_is_added_to_compare' ) ) {
	/**
	 * Check if a property is already added to compare list.
	 *
	 * @param $property_id
	 * @return bool
	 */
	function ere_is_added_to_compare( $property_id ) {

		if ( $property_id > 0 ) {
			/* check cookies for property id */
			if ( isset( $_COOKIE[ 'inspiry_compare' ] ) ) {
				$inspiry_compare 	= unserialize( $_COOKIE[ 'inspiry_compare' ] );
				if ( in_array( $property_id, $inspiry_compare ) ) {
					return true;
				}
			}
		}
		return false;

	}
}

if ( ! function_exists( 'ere_additional_details_migration' ) ) {
	/**
	 * Migrate property additioanl details from old metabox key to new metabox key.
	 *
	 * @param int $post_id Property ID of which additional details has to migrate.
	 */
	function ere_additional_details_migration( $post_id ) {

		if ( ! $post_id ) {
			return;
		}

		$additional_details = get_post_meta( $post_id, 'REAL_HOMES_additional_details', true );
		if ( ! empty( $additional_details ) ) {
			$formatted_details = array();
			foreach ( $additional_details as $field => $value ) {
				$formatted_details[] = array( $field, $value );
			}

			if ( update_post_meta( $post_id, 'REAL_HOMES_additional_details_list', $formatted_details ) ) {
				delete_post_meta( $post_id, 'REAL_HOMES_additional_details' );
			}
		} else {
			// For legacy code
			$detail_titles = get_post_meta( $post_id, 'REAL_HOMES_detail_titles', true );
			if ( ! empty( $detail_titles ) ) {
				$detail_values = get_post_meta( $post_id, 'REAL_HOMES_detail_values', true );
				if ( ! empty( $detail_values ) ) {
					$additional_details = array_combine( $detail_titles, $detail_values );
					$formatted_details = array();
					foreach ( $additional_details as $field => $value ) {
						$formatted_details[] = array( $field, $value );
					}

					if ( update_post_meta( $post_id, 'REAL_HOMES_additional_details_list', $formatted_details ) ) {
						delete_post_meta( $post_id, 'REAL_HOMES_detail_titles' );
						delete_post_meta( $post_id, 'REAL_HOMES_detail_values' );
					}
				}
			}
		}
	}
}


if ( ! function_exists( 'ere_render_property_schema' ) ) {
	/**
	 * This function triggers at single property template putting the related
	 * rich result/schema.org guided structure data for the property to appear
	 * in search results better and in detail.
	 *
	 * @since 2.2.6
	 *
	 * @echo  mixed  structure data json
	 */
	function ere_render_property_schema() {

		if ( ! is_singular( 'property' ) ) {
			return;
		}

		// Save the property ID into a variable for reuse
        $is_rvr_enabled = function_exists( 'inspiry_is_rvr_enabled' ) && inspiry_is_rvr_enabled();
		$real_estate_type = $is_rvr_enabled ? "VacationRental" : "RealEstateListing";
		$property_id      = get_the_ID();
		$property_meta    = get_post_custom( $property_id );

		// Get property meta data
		$price              = ! empty( $property_meta['REAL_HOMES_property_price'] ) ? $property_meta['REAL_HOMES_property_price'][0] : '';
		$price_prefix       = ! empty( $property_meta['REAL_HOMES_property_price_prefix'] ) ? $property_meta['REAL_HOMES_property_price_prefix'][0] : '';
		$price_postfix      = ! empty( $property_meta['REAL_HOMES_property_price_postfix'] ) ? $property_meta['REAL_HOMES_property_price_postfix'][0] : '';
		$address            = ! empty( $property_meta['REAL_HOMES_property_address'] ) ? $property_meta['REAL_HOMES_property_address'][0] : '';
		$bedrooms           = ! empty( $property_meta['REAL_HOMES_property_bedrooms'] ) ? $property_meta['REAL_HOMES_property_bedrooms'][0] : '';
		$bathrooms          = ! empty( $property_meta['REAL_HOMES_property_bedrooms'] ) ? $property_meta['REAL_HOMES_property_bedrooms'][0] : '';
		$garages            = ! empty( $property_meta['REAL_HOMES_property_garage'] ) ? $property_meta['REAL_HOMES_property_garage'][0] : '';
		$area_size          = ! empty( $property_meta['REAL_HOMES_property_size'] ) ? $property_meta['REAL_HOMES_property_size'][0] : '';
		$area_postfix       = ! empty( $property_meta['REAL_HOMES_property_size_postfix'] ) ? $property_meta['REAL_HOMES_property_size_postfix'][0] : '';
		$year_built         = ! empty( $property_meta['REAL_HOMES_property_year_built'] ) ? $property_meta['REAL_HOMES_property_year_built'][0] : '';
		$property_location  = ! empty( $property_meta['REAL_HOMES_property_location'] ) ? $property_meta['REAL_HOMES_property_location'][0] : '';
		$additional_details = ! empty( $property_meta['REAL_HOMES_additional_details_list'] ) ? $property_meta['REAL_HOMES_additional_details_list'][0] : '';
		$lat_lng            = explode( ',', $property_location );
		$latitude           = $lat_lng[0];
		$longitude          = $lat_lng[1];

		// Get taxonomies
		$property_city     = wp_get_post_terms( $property_id, 'property-city', array( 'fields' => 'names' ) );
		$property_status   = wp_get_post_terms( $property_id, 'property-status', array( 'fields' => 'names' ) );
		$property_type     = wp_get_post_terms( $property_id, 'property-type', array( 'fields' => 'names' ) );
		$property_features = wp_get_post_terms( $property_id, 'property-features', array( 'fields' => 'names' ) );

		// Construct the price with prefix and postfix
		$full_price = ( $price_prefix ? $price_prefix . ' ' : '' ) . $price . ( $price_postfix ? ' ' . $price_postfix : '' );

		// Base JSON-LD structure
		$json_ld_data = array(
			"@context"           => "https://schema.org",
			"@type"              => $real_estate_type,
			"name"               => get_the_title(),
			"description"        => get_the_excerpt(),
			"url"                => get_permalink(),
			"address"            => array(
				"@type"           => "PostalAddress",
				"streetAddress"   => $address,
				"addressLocality" => $property_city ? $property_city[0] : "",
				"addressRegion"   => "",
				"postalCode"      => "",
				"addressCountry"  => ""
			),
			"geo"                => array(
				"@type"     => "GeoCoordinates",
				"latitude"  => $latitude,
				"longitude" => $longitude
			),
			"offers"             => array(
				"@type"              => "Offer",
				"price"              => $price,
				"priceCurrency"      => ere_get_currency_sign(),
				"priceSpecification" => $full_price,
				"availability"       => $property_status ? "https://schema.org/" . str_replace( " ", "", $property_status[0] ) : "https://schema.org/InStock"
			),
			"additionalProperty" => array()
		);

		// Add features and additional properties
		if ( $bedrooms ) {
			$json_ld_data['additionalProperty'][] = array(
				"@type" => "PropertyValue",
				"name"  => "Bedrooms",
				"value" => $bedrooms
			);
		}

		if ( $bathrooms ) {
			$json_ld_data['additionalProperty'][] = array(
				"@type" => "PropertyValue",
				"name"  => "Bathrooms",
				"value" => $bathrooms
			);
		}

		if ( $garages ) {
			$json_ld_data['additionalProperty'][] = array(
				"@type" => "PropertyValue",
				"name"  => "Garages",
				"value" => $garages
			);
		}

		if ( $area_size ) {
			$json_ld_data['additionalProperty'][] = array(
				"@type"    => "PropertyValue",
				"name"     => "Area Size",
				"value"    => $area_size,
				"unitText" => $area_postfix
			);
		}

		if ( $year_built ) {
			$json_ld_data['additionalProperty'][] = array(
				"@type" => "PropertyValue",
				"name"  => "Year Built",
				"value" => $year_built
			);
		}

		if ( ! empty( $additional_details ) ) {

			// Removing empty values after unserialize raw data
			$additional_details = array_filter( unserialize( $additional_details ) );

			foreach ( $additional_details as $detail ) {
				$json_ld_data['additionalProperty'][] = array(
					"@type" => "PropertyValue",
					"name"  => $detail[0],
					"value" => $detail[1]
				);
			}
		}


		// Vacation Rental-specific data
		if ( $is_rvr_enabled ) {

			$guests            = ! empty( $property_meta['rvr_guests_capacity'] ) ? $property_meta['rvr_guests_capacity'][0] : '';
			$extra_guest_price = ! empty( $property_meta['rvr_extra_guest_price'] ) ? $property_meta['rvr_extra_guest_price'][0] : '';
			$min_nights        = ! empty( $property_meta['rvr_min_stay'] ) ? $property_meta['rvr_min_stay'][0] : '';

			if ( $price ) {
				$json_ld_data['offers']['price'] = $price; // Override price with rental price
			}

			if ( $guests ) {
				$json_ld_data['additionalProperty'][] = array(
					"@type" => "PropertyValue",
					"name"  => "Maximum Guests",
					"value" => $guests
				);
			}

			if ( $extra_guest_price ) {
				$json_ld_data['additionalProperty'][] = array(
					"@type"    => "PropertyValue",
					"name"     => "Extra Guest Price",
					"value"    => $extra_guest_price,
					"unitText" => "per guest"
				);
			}

			if ( $min_nights ) {
				$json_ld_data['additionalProperty'][] = array(
					"@type" => "PropertyValue",
					"name"  => "Minimum Nights",
					"value" => $min_nights
				);
			}
		}

		// Encode and output JSON-LD
		echo '<script type="application/ld+json">' . wp_json_encode( $json_ld_data ) . '</script>';
	}

	add_action( 'wp_head', 'ere_render_property_schema' );
}

if ( ! function_exists( 'ere_render_agent_schema' ) ) {
	/**
	 * This function triggers at single agent template putting the related
	 * rich result/schema.org guided structure data for the agent to appear
	 * in search results better and in detail.
	 *
	 * @since 2.2.6
	 *
	 * @echo  mixed  structure data json
	 */
	function ere_render_agent_schema() {

		if ( ! is_singular( 'agent' ) ) {
			return;
		}

		$agent_id   = get_the_ID();
		$agent_meta = get_post_custom( $agent_id );

		$name           = get_the_title( $agent_id );
		$description    = get_the_excerpt( $agent_id );
		$mobile         = ! empty( $agent_meta['REAL_HOMES_mobile_number'] ) ? $agent_meta['REAL_HOMES_mobile_number'] : '';
		$office         = ! empty( $agent_meta['REAL_HOMES_office_number'] ) ? $agent_meta['REAL_HOMES_office_number'] : '';
		$whatsapp       = ! empty( $agent_meta['REAL_HOMES_whatsapp_number'] ) ? $agent_meta['REAL_HOMES_whatsapp_number'] : '';
		$fax            = ! empty( $agent_meta['REAL_HOMES_fax_number'] ) ? $agent_meta['REAL_HOMES_fax_number'] : '';
		$website        = ! empty( $agent_meta['REAL_HOMES_website'] ) ? $agent_meta['REAL_HOMES_website'] : '';
		$email          = ! empty( $agent_meta['REAL_HOMES_agent_email'] ) ? $agent_meta['REAL_HOMES_agent_email'] : '';
		$agency_id      = ! empty( $agent_meta['REAL_HOMES_agency'] ) ? $agent_meta['REAL_HOMES_agency'] : '';
		$agency_name    = ! empty( $agency_id ) ? get_the_title( $agency_id ) : '';
		$license_number = ! empty( $agent_meta['REAL_HOMES_license_number'] ) ? $agent_meta['REAL_HOMES_license_number'] : '';
		$address        = ! empty( $agent_meta['REAL_HOMES_address'] ) ? $agent_meta['REAL_HOMES_address'] : '';

		// Social Media URLs
		$social_profiles = array_filter( array(
			'Facebook'  => ! empty( $agent_meta['REAL_HOMES_facebook_url'] ) ? $agent_meta['REAL_HOMES_facebook_url'] : '',
			'Twitter'   => ! empty( $agent_meta['REAL_HOMES_twitter_url'] ) ? $agent_meta['REAL_HOMES_twitter_url'] : '',
			'LinkedIn'  => ! empty( $agent_meta['REAL_HOMES_linked_in_url'] ) ? $agent_meta['REAL_HOMES_linked_in_url'] : '',
			'Instagram' => ! empty( $agent_meta['inspiry_instagram_url'] ) ? $agent_meta['inspiry_instagram_url'] : '',
			'TikTok'    => ! empty( $agent_meta['realhomes_tiktok_url'] ) ? $agent_meta['realhomes_tiktok_url'] : '',
			'Pinterest' => ! empty( $agent_meta['inspiry_pinterest_url'] ) ? $agent_meta['inspiry_pinterest_url'] : '',
			'Youtube'   => ! empty( $agent_meta['inspiry_youtube_url'] ) ? $agent_meta['inspiry_youtube_url'] : ''
		) );

		// Get areaServed from agent-location taxonomy
		$agent_locations = wp_get_post_terms( $agent_id, 'agent-location', array( 'fields' => 'names' ) );

		// Prepare JSON-LD data
		$schema_data = array(
			"@context"    => "https://schema.org",
			"@type"       => "RealEstateAgent",
			"name"        => $name,
			"description" => $description,
			"telephone"   => $mobile,
			"email"       => $email,
			"url"         => $website,
			"areaServed"  => ! is_wp_error( $agent_locations ) && ! empty( $agent_locations ) ? $agent_locations : [],
		);

		if ( $office ) {
			$schema_data["contactPoint"][] = array(
				"@type"       => "ContactPoint",
				"telephone"   => $office,
				"contactType" => "Office Contact" // You can adjust this as needed
			);
		}

		if ( $whatsapp ) {
			$schema_data["contactPoint"][] = array(
				"@type"       => "ContactPoint",
				"telephone"   => $whatsapp,
				"contactType" => "WhatsApp Contact" // You can adjust this as needed
			);
		}

		if ( $fax ) {
			$schema_data["faxNumber"] = $fax;
		}

		if ( $agency_name ) {
			$schema_data["memberOf"] = array(
				"@type" => "Organization",
				"name"  => $agency_name
			);
		}

		if ( $license_number ) {
			$schema_data["identifier"] = array(
				"@type" => "PropertyValue",
				"name"  => "License Number",
				"value" => $license_number
			);
		}

		if ( $address ) {
			$schema_data["address"] = array(
                "@type"           => "PostalAddress",
                "streetAddress"   => $address,
                "addressLocality" => "",
                "addressRegion"   => "",
                "postalCode"      => "",
                "addressCountry"  => ""
			);
		}

		if ( ! empty( $social_profiles ) ) {
			foreach ( $social_profiles as $key => $social_profile ) {
				$schema_data["contactPoint"][] = array(
					"@type"       => "ContactPoint",
					"telephone"   => $social_profile,
					"contactType" => $key
				);
			}
		}

		// Output JSON-LD script
		echo '<script type="application/ld+json">' . wp_json_encode( $schema_data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) . '</script>';
	}

	add_action( 'wp_head', 'ere_render_agent_schema' );
}

if ( ! function_exists( 'ere_render_agency_schema' ) ) {
	/**
	 * This function triggers at single agency template putting the related
	 * rich result/schema.org guided structure data for the agency to appear
	 * in search results better and in detail.
	 *
	 * @since 2.2.6
	 *
	 * @echo  mixed  structure data json
	 */
	function ere_render_agency_schema() {
		if ( ! is_singular( 'agency' ) ) {
			return;
		}

		$agency_id   = get_the_ID();
		$agency_meta = get_post_custom( $agency_id );
		$agency_name = get_the_title( $agency_id );
		$office      = ! empty( $agency_meta['REAL_HOMES_office_number'] ) ? $agency_meta['REAL_HOMES_office_number'] : '';
		$mobile      = ! empty( $agency_meta['REAL_HOMES_mobile_number'] ) ? $agency_meta['REAL_HOMES_mobile_number'] : '';
		$whatsapp    = ! empty( $agency_meta['REAL_HOMES_whatsapp_number'] ) ? $agency_meta['REAL_HOMES_whatsapp_number'] : '';
		$fax         = ! empty( $agency_meta['REAL_HOMES_fax_number'] ) ? $agency_meta['REAL_HOMES_fax_number'] : '';
		$website     = ! empty( $agency_meta['REAL_HOMES_website'] ) ? $agency_meta['REAL_HOMES_website'] : '';
		$email       = ! empty( $agency_meta['REAL_HOMES_agency_email'] ) ? $agency_meta['REAL_HOMES_agency_email'] : '';
		$address     = ! empty( $agency_meta['REAL_HOMES_address'] ) ? $agency_meta['REAL_HOMES_address'] : '';

		// Social Media URLs
		$social_profiles = array_filter( array(
			'Facebook'  => ! empty( $agency_meta['REAL_HOMES_facebook_url'] ) ? $agency_meta['REAL_HOMES_facebook_url'] : '',
			'Twitter'   => ! empty( $agency_meta['REAL_HOMES_twitter_url'] ) ? $agency_meta['REAL_HOMES_twitter_url'] : '',
			'LinkedIn'  => ! empty( $agency_meta['REAL_HOMES_linked_in_url'] ) ? $agency_meta['REAL_HOMES_linked_in_url'] : '',
			'Instagram' => ! empty( $agency_meta['inspiry_instagram_url'] ) ? $agency_meta['inspiry_instagram_url'] : '',
			'TikTok'    => ! empty( $agency_meta['realhomes_tiktok_url'] ) ? $agency_meta['realhomes_tiktok_url'] : '',
			'Pinterest' => ! empty( $agency_meta['inspiry_pinterest_url'] ) ? $agency_meta['inspiry_pinterest_url'] : '',
			'Youtube'   => ! empty( $agency_meta['inspiry_youtube_url'] ) ? $agency_meta['inspiry_youtube_url'] : ''
		) );

		// Get areaServed from agency-location taxonomy
		$agency_locations = wp_get_post_terms( $agency_id, 'agency-location', array( 'fields' => 'names' ) );

		// Prepare JSON-LD data
		$schema_data = array(
			"@context"   => "https://schema.org",
			"@type"      => "Organization",
			"name"       => $agency_name,
			"url"        => $website,
			"telephone"  => $mobile,
			"email"      => $email,
			"areaServed" => ! is_wp_error( $agency_locations ) && ! empty( $agency_locations ) ? $agency_locations : []
		);

		if ( $office ) {
			$schema_data["contactPoint"][] = array(
				"@type"       => "ContactPoint",
				"telephone"   => $office,
				"contactType" => "Office Contact" // You can adjust this as needed
			);
		}

		if ( $whatsapp ) {
			$schema_data["contactPoint"][] = array(
				"@type"       => "ContactPoint",
				"telephone"   => $whatsapp,
				"contactType" => "WhatsApp Contact" // You can adjust this as needed
			);
		}

		if ( $fax ) {
			$schema_data["faxNumber"] = $fax;
		}

		if ( $address ) {
			$schema_data["address"] =  array(
				"@type"           => "PostalAddress",
				"streetAddress"   => $address,
				"addressLocality" => "",
				"addressRegion"   => "",
				"postalCode"      => "",
				"addressCountry"  => ""
			);
		}

		if ( ! empty( $social_profiles ) ) {
			foreach ( $social_profiles as $key => $social_profile ) {
				$schema_data["contactPoint"][] = array(
					"@type"       => "ContactPoint",
					"telephone"   => $social_profile,
					"contactType" => $key
				);
			}
		}

		// Output JSON-LD script
		echo '<script type="application/ld+json">' . wp_json_encode( $schema_data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) . '</script>';
	}

	add_action( 'wp_head', 'ere_render_agency_schema' );
}