<?php
/**
 * Agency Custom Post Type
 */

if ( ! function_exists( 'ere_register_agency_post_type' ) ) {
	/**
	 * Register Custom Post Type : Agency
	 */
	function ere_register_agency_post_type() {

		$labels = array(
			'name'                  => esc_html_x( 'Agencies', 'Post Type General Name', ERE_TEXT_DOMAIN ),
			'singular_name'         => esc_html_x( 'Agency', 'Post Type Singular Name', ERE_TEXT_DOMAIN ),
			'menu_name'             => esc_html__( 'Agencies', ERE_TEXT_DOMAIN ),
			'name_admin_bar'        => esc_html__( 'Agency', ERE_TEXT_DOMAIN ),
			'archives'              => esc_html__( 'Agency Archives', ERE_TEXT_DOMAIN ),
			'attributes'            => esc_html__( 'Agency Attributes', ERE_TEXT_DOMAIN ),
			'parent_item_colon'     => esc_html__( 'Parent Agency:', ERE_TEXT_DOMAIN ),
			'all_items'             => esc_html__( 'All Agencies', ERE_TEXT_DOMAIN ),
			'add_new_item'          => esc_html__( 'Add New Agency', ERE_TEXT_DOMAIN ),
			'add_new'               => esc_html__( 'Add New', ERE_TEXT_DOMAIN ),
			'new_item'              => esc_html__( 'New Agency', ERE_TEXT_DOMAIN ),
			'edit_item'             => esc_html__( 'Edit Agency', ERE_TEXT_DOMAIN ),
			'update_item'           => esc_html__( 'Update Agency', ERE_TEXT_DOMAIN ),
			'view_item'             => esc_html__( 'View Agency', ERE_TEXT_DOMAIN ),
			'view_items'            => esc_html__( 'View Agencies', ERE_TEXT_DOMAIN ),
			'search_items'          => esc_html__( 'Search Agency', ERE_TEXT_DOMAIN ),
			'not_found'             => esc_html__( 'Not found', ERE_TEXT_DOMAIN ),
			'not_found_in_trash'    => esc_html__( 'Not found in Trash', ERE_TEXT_DOMAIN ),
			'featured_image'        => esc_html__( 'Agency Logo Image', ERE_TEXT_DOMAIN ),
			'set_featured_image'    => esc_html__( 'Set agency logo image', ERE_TEXT_DOMAIN ),
			'remove_featured_image' => esc_html__( 'Remove agency logo image', ERE_TEXT_DOMAIN ),
			'use_featured_image'    => esc_html__( 'Use as agency logo image', ERE_TEXT_DOMAIN ),
			'insert_into_item'      => esc_html__( 'Insert into agency', ERE_TEXT_DOMAIN ),
			'uploaded_to_this_item' => esc_html__( 'Uploaded to this agency', ERE_TEXT_DOMAIN ),
			'items_list'            => esc_html__( 'Agencies list', ERE_TEXT_DOMAIN ),
			'items_list_navigation' => esc_html__( 'Agencies list navigation', ERE_TEXT_DOMAIN ),
			'filter_items_list'     => esc_html__( 'Filter agencies list', ERE_TEXT_DOMAIN ),
		);
		$args   = array(
			'label'               => esc_html__( 'Agency', ERE_TEXT_DOMAIN ),
			'description'         => esc_html__( 'An agency is a company of realtors.', ERE_TEXT_DOMAIN ),
			'labels'              => $labels,
			'hierarchical'        => false,
			'public'              => true,
			'show_ui'             => true,
			'show_in_menu'        => false,
			'menu_position'       => 5,
			'menu_icon'           => 'dashicons-groups',
			'show_in_admin_bar'   => true,
			'show_in_nav_menus'   => true,
			'can_export'          => true,
			'has_archive'         => ( defined( 'INSPIRY_DESIGN_VARIATION' ) && 'ultra' === INSPIRY_DESIGN_VARIATION ),
			'exclude_from_search' => true,
			'publicly_queryable'  => true,
			'capability_type'     => 'post',
			'supports'            => array( 'title', 'editor', 'excerpt', 'thumbnail', 'revisions', 'comments' ),
			'rewrite'             => array(
				'slug' => apply_filters( 'inspiry_agency_slug', esc_html__( 'agency', ERE_TEXT_DOMAIN ) ),
			),
			'show_in_rest'        => true,
			'rest_base'           => apply_filters( 'inspiry_agency_rest_base', esc_html__( 'agencies', ERE_TEXT_DOMAIN ) )
		);

		// Filter the arguments to register agency post type.
		$args = apply_filters( 'inspiry_agency_post_type_args', $args );
		register_post_type( 'agency', $args );

	}

	add_action( 'init', 'ere_register_agency_post_type', 0 );

}


if ( ! function_exists( 'ere_set_agency_slug' ) ) {
	/**
	 * This function set agency's url slug by hooking itself with related filter
	 *
	 * @param string $existing_slug - Existing slug.
	 *
	 * @return string
	 */
	function ere_set_agency_slug( $existing_slug ) {
		$new_slug = get_option( 'inspiry_agency_slug' );
		if ( ! empty( $new_slug ) ) {
			return $new_slug;
		}

		return $existing_slug;
	}

	add_filter( 'inspiry_agency_slug', 'ere_set_agency_slug' );
}


if ( ! function_exists( 'ere_register_agency_location_taxonomy' ) ) {
	/**
	 * Register Agency Location Taxonomy
	 *
	 * @since 2.0.0
	 * @return void
	 */
	function ere_register_agency_location_taxonomy() {
		if ( taxonomy_exists( 'agency-location' ) ) {
			return;
		}

		$city_labels = array(
			'name'                       => esc_html__( 'Agency Locations', ERE_TEXT_DOMAIN ),
			'singular_name'              => esc_html__( 'Agency Location', ERE_TEXT_DOMAIN ),
			'search_items'               => esc_html__( 'Search Agency Locations', ERE_TEXT_DOMAIN ),
			'popular_items'              => esc_html__( 'Popular Agency Locations', ERE_TEXT_DOMAIN ),
			'all_items'                  => esc_html__( 'All Agency Locations', ERE_TEXT_DOMAIN ),
			'parent_item'                => esc_html__( 'Parent Agency Location', ERE_TEXT_DOMAIN ),
			'parent_item_colon'          => esc_html__( 'Parent Agency Location:', ERE_TEXT_DOMAIN ),
			'edit_item'                  => esc_html__( 'Edit Agency Location', ERE_TEXT_DOMAIN ),
			'update_item'                => esc_html__( 'Update Agency Location', ERE_TEXT_DOMAIN ),
			'add_new_item'               => esc_html__( 'Add New Agency Location', ERE_TEXT_DOMAIN ),
			'new_item_name'              => esc_html__( 'New Agency Location Name', ERE_TEXT_DOMAIN ),
			'separate_items_with_commas' => esc_html__( 'Separate Agency Locations with commas', ERE_TEXT_DOMAIN ),
			'add_or_remove_items'        => esc_html__( 'Add or remove Agency Locations', ERE_TEXT_DOMAIN ),
			'choose_from_most_used'      => esc_html__( 'Choose from the most used Agency Locations', ERE_TEXT_DOMAIN ),
			'menu_name'                  => esc_html__( 'Agency Locations', ERE_TEXT_DOMAIN ),
		);

		register_taxonomy(
			'agency-location',
			array( 'agency' ),
			array(
				'hierarchical' => true,
				'labels'       => apply_filters( 'ere_agency_location_labels', $city_labels ),
				'show_ui'      => true,
				'show_in_menu' => 'easy-real-estate',
				'query_var'    => true,
				'rewrite'      => array(
					'slug' => apply_filters( 'ere_agency_location_slug', esc_html__( 'agency-location', ERE_TEXT_DOMAIN ) ),
				),
				'show_in_rest' => true,
				'rest_base'    => apply_filters( 'ere_agency_location_rest_base', esc_html__( 'agency-locations', ERE_TEXT_DOMAIN ) )
			)
		);
	}

	add_action( 'init', 'ere_register_agency_location_taxonomy', 0 );
}

if ( ! function_exists( 'ere_set_agency_location_slug' ) ) {
	/**
	 * This function set agency location's URL slug by hooking itself with related filter
	 *
	 * @since 2.0.0
	 *
	 * @param string $existing_slug - Existing agency location slug.
	 *
	 * @return string
	 */
	function ere_set_agency_location_slug( $existing_slug ) {
		$new_slug = get_option( 'ere_agency_location_slug' );
		if ( ! empty( $new_slug ) ) {
			return $new_slug;
		}

		return $existing_slug;
	}

	add_filter( 'ere_agency_location_slug', 'ere_set_agency_location_slug' );
	add_filter( 'ere_agency_location_rest_base', 'ere_set_agency_location_slug' );
}

if ( ! function_exists( 'ere_agency_edit_columns' ) ) {
	/**
	 * Custom columns for agencies.
	 *
	 * @param array $columns - Columns array.
	 *
	 * @return array
	 */
	function ere_agency_edit_columns( $columns ) {

		$columns = array(
			'cb'               => '<input type="checkbox" />',
			'title'            => esc_html__( 'Agency', ERE_TEXT_DOMAIN ),
			'agency-thumbnail' => esc_html__( 'Thumbnail', ERE_TEXT_DOMAIN ),
			'location'         => esc_html__( 'Location', ERE_TEXT_DOMAIN ),
			'total_properties' => esc_html__( 'Total Properties', ERE_TEXT_DOMAIN ),
			'published'        => esc_html__( 'Published Properties', ERE_TEXT_DOMAIN ),
			'others'           => esc_html__( 'Other Properties', ERE_TEXT_DOMAIN ),
			'rating'           => esc_html__( 'Average Rating', ERE_TEXT_DOMAIN ),
			'date'             => esc_html__( 'Created', ERE_TEXT_DOMAIN ),
		);

		/**
		 * WPML Support
		 */
		if ( defined( 'ICL_SITEPRESS_VERSION' ) ) {
			global $sitepress;
			$wpml_columns = new WPML_Custom_Columns( $sitepress );
			$columns      = $wpml_columns->add_posts_management_column( $columns );
		}

		/**
		 * Reverse the array for RTL
		 */
		if ( is_rtl() ) {
			$columns = array_reverse( $columns );
		}

		return $columns;
	}

	add_filter( 'manage_edit-agency_columns', 'ere_agency_edit_columns' );
}


if ( ! function_exists( 'ere_agency_custom_columns' ) ) {

	/**
	 * Custom column values for agency post type.
	 *
	 * @param string $column    - Name of the column.
	 * @param int    $agency_id - ID of the current agency.
	 *
	 */
	function ere_agency_custom_columns( $column, $agency_id ) {

		// Switch cases against column names.
		switch ( $column ) {
			case 'agency-thumbnail':
				if ( has_post_thumbnail( $agency_id ) ) {
					?>
                    <a href="<?php the_permalink(); ?>" target="_blank">
						<?php the_post_thumbnail( array( 130, 130 ) ); ?>
                    </a>
					<?php
				} else {
					esc_html_e( 'No Thumbnail', ERE_TEXT_DOMAIN );
				}
				break;
			// Agency Location
			case 'location':
				$agency_locations = ere_admin_taxonomy_terms( $agency_id, 'agency-location', 'agency' );
				echo ! $agency_locations ? esc_html__( 'None', ERE_TEXT_DOMAIN ) : $agency_locations;
				break;
			// Total properties.
			case 'total_properties':
				$listed_properties = ere_get_agency_properties_count( $agency_id );
				echo ( ! empty( $listed_properties ) ) ? esc_html( $listed_properties ) : 0;
				break;
			// Total published properties.
			case 'published':
				$published_properties = ere_get_agency_properties_count( $agency_id, 'publish' );
				echo ( ! empty( $published_properties ) ) ? esc_html( $published_properties ) : 0;
				break;
			// Other properties.
			case 'others':
				$property_status   = array( 'pending', 'draft', 'private', 'future' );
				$others_properties = ere_get_agency_properties_count( $agency_id, $property_status );
				echo ( ! empty( $others_properties ) ) ? esc_html( $others_properties ) : 0;
				break;
			// Average Rating
			case 'rating':
				$average_rating = get_post_meta( $agency_id, 'realhomes_post_average_rating', true );
				echo ( ! empty( $average_rating ) ) ? esc_html( $average_rating ) : 0;
				break;
			default:
				break;
		}
	}

	add_action( 'manage_agency_posts_custom_column', 'ere_agency_custom_columns', 10, 2 );
}

if ( ! function_exists( 'ere_agency_filter_fields_admin' ) ) {
	/**
	 * Add custom filter fields for agency on admin
	 *
	 * @since 2.2.2
	 */
	function ere_agency_filter_fields_admin() {

		global $post_type;
		if ( $post_type == 'agency' ) {

			// Agency Location Dropdown Option
			$agency_location_args = array(
				'show_option_all' => esc_html__( 'All Agency Locations', ERE_TEXT_DOMAIN ),
				'orderby'         => 'NAME',
				'order'           => 'ASC',
				'name'            => 'agency_location',
				'taxonomy'        => 'agency-location'
			);
			if ( isset( $_GET['agency_location'] ) ) {
				$agency_location_args['selected'] = sanitize_text_field( $_GET['agency_location'] );
			}
			wp_dropdown_categories( $agency_location_args );

			// Ratings Dropdown Option
			$selected_rating = 0;
			if ( isset( $_GET['agency_rating'] ) ) {
				$selected_rating = sanitize_text_field( $_GET['agency_rating'] );
			}
			?>
            <select name="agency_rating" id="agency-rating-drowdown">
                <option value="0" <?php selected( 0, $selected_rating ); ?>><?php esc_html_e( 'All Ratings', ERE_TEXT_DOMAIN ); ?></option>
                <option value="1" <?php selected( 1, $selected_rating ); ?>><?php esc_html_e( '1 Star', ERE_TEXT_DOMAIN ); ?></option>
                <option value="2" <?php selected( 2, $selected_rating ); ?>><?php esc_html_e( '2 Stars', ERE_TEXT_DOMAIN ); ?></option>
                <option value="3" <?php selected( 3, $selected_rating ); ?>><?php esc_html_e( '3 Stars', ERE_TEXT_DOMAIN ); ?></option>
                <option value="4" <?php selected( 4, $selected_rating ); ?>><?php esc_html_e( '4 Stars', ERE_TEXT_DOMAIN ); ?></option>
                <option value="5" <?php selected( 5, $selected_rating ); ?>><?php esc_html_e( '5 Stars', ERE_TEXT_DOMAIN ); ?></option>
            </select>
			<?php
		}
	}

	add_action( 'restrict_manage_posts', 'ere_agency_filter_fields_admin' );
}

if ( ! function_exists( 'ere_agency_filter_admin' ) ) {
	/**
	 * Restrict the agency by the chosen filters
	 *
	 * @since 2.2.2
	 *
	 * @param $query
	 */
	function ere_agency_filter_admin( $query ) {

		global $post_type, $pagenow;

		// If we are currently on the edit screen of the agency post-type listings
		if ( $pagenow == 'edit.php' && $post_type == 'agency' ) {

			$tax_query  = array();
			$meta_query = array();

			// Agent Location Filter
			if ( isset( $_GET['agency_location'] ) && ! empty( $_GET['agency_location'] ) ) {

				// Get the desired property location
				$agent_location = sanitize_text_field( $_GET['agency_location'] );

				// If the agency location is not 0 (which means all)
				if ( $agent_location != 0 ) {
					$tax_query[] = array(
						'taxonomy' => 'agency-location',
						'field'    => 'ID',
						'terms'    => array( $agent_location )
					);

				}
			}

			// Agency Ratings Filter
			if ( isset( $_GET['agency_rating'] ) && ! empty( $_GET['agency_rating'] ) ) {

				$meta_query[] = array(
					'key'     => 'realhomes_post_average_rating',
					'value'   => sanitize_text_field( $_GET['agency_rating'] ),
					'compare' => '=',
				);

			}

			if ( ! empty( $meta_query ) ) {
				$query->query_vars['meta_query'] = $meta_query;
			}

			if ( ! empty( $tax_query ) ) {
				$query->query_vars['tax_query'] = $tax_query;
			}
		}
	}

	add_action( 'pre_get_posts', 'ere_agency_filter_admin' );
}