<?php
/**
 * ERE Purchase API
 *
 * Handles the purchase verification.
 *
 * @package realhomes/framework/functions
 */
if ( ! class_exists( 'ERE_Purchase_API' ) ) {

	class ERE_Purchase_API {

		public function __construct() {
			if ( ! self::status() ) {
				$this->auto_register();
				add_action( 'admin_notices', array( $this, 'register_license' ) );
			}

			if ( self::status() && get_transient( 'inspiry_activation_thanks' ) ) {
				add_action( 'admin_notices', array( $this, 'after_register_notice' ) );
			}
		}

		static function status() {
			return ( 'yes' == get_option( 'inspiry_realhomes_registered', 'no' ) );
		}

		public function after_register_notice() {
			?>
            <div class="notice notice-success realhomes-registration is-dismissible">
                <p>Thanks for verifying RealHomes purchase!</p>
            </div>
			<?php
			delete_transient( 'inspiry_activation_thanks' );
		}

		public function register_license() {

			$verification       = false;
			$purchase_code     = '';
			$envato_username   = '';
			$verification_error = '';

			// Only run verification if form was submitted
			if ( $_SERVER['REQUEST_METHOD'] === 'POST' ) {

				// Sanitize and validate the input
				$purchase_code   = isset( $_POST['purchase_code'] ) ? sanitize_text_field( $_POST['purchase_code'] ) : '';
				$envato_username = isset( $_POST['envato_username'] ) ? sanitize_text_field( $_POST['envato_username'] ) : '';

				// Check if the purchase code and Envato username are valid
				// Also avoid any errors if user manipulated the form by removing "required" attribute
				if ( empty( $purchase_code ) ) {
					$verification_error = esc_html__( 'Please enter an item purchase code.', RH_TEXT_DOMAIN );
				} else if ( ! preg_match( '/^([a-f0-9]{8})-(([a-f0-9]{4})-){3}([a-f0-9]{12})$/i', $purchase_code ) ) {
					$verification_error = esc_html__( 'Wrong format! Please enter an item purchase code with correct format.', RH_TEXT_DOMAIN );
				} else if ( empty( $envato_username ) ) {
					$verification_error = esc_html__( 'Please enter your ThemeForest username.', RH_TEXT_DOMAIN );
				} else {

					// Verify the purchase code and Envato username
					$item_purchase = $this->verify_purchase_code( $purchase_code, $envato_username );

					if ( ! is_wp_error( $item_purchase ) && $item_purchase === true ) {
						$verification = true;
					} else {
						$verification_error = $item_purchase->errors['error'][0];
					}

					if ( $verification ) {

						set_transient( 'inspiry_activation_thanks', true, 5 );

						global $wp;
						if ( $wp->request ) {
							$current_url = add_query_arg( $wp->query_string, '', home_url( $wp->request ) );
						} else {
							$current_url = admin_url();
						}
						inspiry_redirect( $current_url );
					}
				}
			}

			if ( ! get_transient( 'inspiry_activation_thanks' ) ) {
				$attempt_notice = get_option( 'realhomes_verification_attempt_notice' );
				$notice_class   = ( 'true' === $attempt_notice ) ? 'error' : 'info';
				?>
                <div class="notice notice-<?php echo esc_attr( $notice_class ); ?> realhomes-registration">
					<?php
					if ( 'true' === $attempt_notice ) {
						?>
                        <h3>RealHomes License Purchase Verification Failed</h3>
                        <p>Your existing item purchase code couldn't be verified. Verify your purchase by providing a valid item purchase code, This will allow you to <strong>install plugins</strong>, <strong>import demo contents</strong> and avail <strong>auto updates</strong>.</p>
						<?php
					} else {
						?>
                        <h3>RealHomes License Purchase Verification</h3>
                        <p>Verify your purchase by providing item purchase code, This will allow you to <strong>install plugins</strong>, <strong>import demo contents</strong> and avail <strong>auto updates</strong>.</p>
						<?php
					}
					?>
                    <form method="post">
                        <input type="text" class="regular-text" name="purchase_code" placeholder="Enter Item Purchase Code" required value="<?php echo esc_html( $purchase_code ); ?>">
                        <input type="text" class="regular-text" name="envato_username" placeholder="Enter ThemeForest Username" required value="<?php echo esc_html( $envato_username ); ?>">
                        <input type="submit" class="button button-primary" value="Verify">
                    </form>
					<?php
					if ( ! empty( $verification_error ) ) {
						echo '<p class="error">' . esc_html( $verification_error ) . '</p>';
					}
					?>
                    <p>You can consult <a href="https://support.inspirythemes.com/knowledgebase/how-to-get-themeforest-item-purchase-code/" target="_blank">this knowledge base article</a> to learn how to get ThemeForest item purchase code, Otherwise you can buy <a href="https://themeforest.net/item/real-homes-wordpress-real-estate-theme/5373914" target="_blank">RealHomes fresh license</a> that comes with lifetime updates and 6 months extendable support.</p>
                </div>
				<?php
			}
		}

		private function auto_register() {

			$existing_purchase_code   = get_option( 'envato_purchase_code_5373914' );
			$existing_envato_username = get_option( 'envato_username_5373914' );

			if ( ! empty( $existing_purchase_code ) && ! empty( $existing_envato_username ) ) {
				$item_purchase = $this->verify_purchase_code( $existing_purchase_code, $existing_envato_username );

				if ( is_wp_error( $item_purchase ) || $item_purchase != true ) {
					delete_option( 'envato_purchase_code_5373914' );
					delete_option( 'envato_username_5373914' );
				} else {
					set_transient( 'inspiry_activation_thanks', true, 5 );

				}
			}
		}

		public function verify_purchase_code( $code, $envato_username ) {

			$envato_token = 'XLOSUxHcOKTC1TrljyoeWJ10ptng74a7';
			$code         = trim( $code );
			$username     = trim( $envato_username );
			$error        = new WP_Error();

			// Username is required
			if ( empty( $username ) ) {
				$error->add( 'error', esc_html__( 'Envato username is required for license verification.', RH_TEXT_DOMAIN ) );
				return $error;
			}

			// Prepare the API request
			$envato_apiurl            = "https://api.envato.com/v3/market/author/sale?code=" . $code;
			$envato_header            = array();
			$envato_header['headers'] = array(
				"Authorization" => "Bearer " . $envato_token,
				'User-Agent'    => 'RealHomes Theme Activation'
			);

			// Make the API request
			$envato_purchases = wp_safe_remote_request( $envato_apiurl, $envato_header );

			// Proceed only if the request was successful and response body is not empty
			if ( ! is_wp_error( $envato_purchases ) && ! empty( $envato_purchases['body'] ) ) {
				$purchases_body = json_decode( $envato_purchases['body'], true );

				// Username must match
				if ( ! empty( $purchases_body['buyer'] ) && $purchases_body['buyer'] !== $username ) {
					$error->add( 'error', esc_html__( 'The provided Envato username does not match the purchase code.', RH_TEXT_DOMAIN ) );
					return $error;
				}

				if ( isset( $purchases_body['item'] ) ) {
					$item_info = $purchases_body['item'];

					if ( isset( $item_info['id'] ) && '5373914' === str_replace( ' ', '', $item_info['id'] ) ) {
						update_option( 'envato_purchase_code_5373914', sanitize_text_field( $code ) );
						update_option( 'envato_username_5373914', sanitize_text_field( $username ) );

						update_option( 'inspiry_realhomes_registered', 'yes' );
						delete_option( 'realhomes_verification_attempt_notice' );

						return true;
					} else {
						$error->add( 'error', esc_html__( 'Unable to verify license. Please check the purchase code and try again.', RH_TEXT_DOMAIN ) );

						return $error;
					}
				} else {

					if ( ! empty( $purchases_body['description'] ) ) {
						$error->add( 'error', esc_html( $purchases_body['description'] ) );
					} else {
						$error->add( 'error', esc_html__( 'Something went wrong :(', RH_TEXT_DOMAIN ) );
					}

					return $error;
				}

			} else {
				$error->add( 'error', esc_html__( 'Problem in connecting... Please try again in a few minutes!', RH_TEXT_DOMAIN ) );

				return $error;
			}
		}
	}
}
