<?php

/**
 * RealHomes Cache Handler
 *
 * A comprehensive caching solution for RealHomes theme using WordPress native APIs
 *
 * @since 4.4.0
 */
class RealHomes_Cache {
	/**
	 * Cache group name
	 */
	const CACHE_GROUP = 'realhomes';

	/**
	 * Cache expiration in constants
	 */
	const SHORT_EXPIRATION   = 1 * HOUR_IN_SECONDS;    // 1 hour
	const MEDIUM_EXPIRATION  = 12 * HOUR_IN_SECONDS;   // 12 hours
	const LONG_EXPIRATION    = 24 * HOUR_IN_SECONDS;   // 24 hours
	const DEFAULT_EXPIRATION = self::MEDIUM_EXPIRATION;

	/**
	 * Get cached data
	 *
	 * @param string $key   Cache key
	 * @param string $group Cache group (optional)
	 *
	 * @return mixed Cached data or false if not found
	 */
	public static function get( $key, $group = '' ) {
		$group = empty( $group ) ? self::CACHE_GROUP : $group;

		// Try object cache first
		$data = wp_cache_get( $key, $group );

		if ( false === $data ) {
			// Fallback to transient
			$data = get_transient( "{$group}_{$key}" );

			if ( $data !== false ) {
				// Store in object cache for future requests
				wp_cache_set( $key, $data, $group, self::DEFAULT_EXPIRATION );
			}
		}

		return $data;
	}

	/**
	 * Set cache data
	 *
	 * @param string $key        Cache key
	 * @param mixed  $data       Data to cache
	 * @param string $group      Cache group (optional)
	 * @param int    $expiration Expiration time in seconds (optional)
	 *
	 * @return bool True on success, false on failure
	 */
	public static function set( $key, $data, $group = '', $expiration = null ) {
		$group      = empty( $group ) ? self::CACHE_GROUP : $group;
		$expiration = $expiration ?? self::DEFAULT_EXPIRATION;

		// Store in both object cache and transient
		$set_in_object_cache = wp_cache_set( $key, $data, $group, $expiration );
		$set_in_transient    = set_transient( "{$group}_{$key}", $data, $expiration );

		return $set_in_object_cache && $set_in_transient;
	}

	/**
	 * Delete cached data
	 *
	 * @param string $key   Cache key
	 * @param string $group Cache group (optional)
	 *
	 * @return bool True on success, false on failure
	 */
	public static function delete( $key, $group = '' ) {
		$group = empty( $group ) ? self::CACHE_GROUP : $group;

		// Delete from both object cache and transient
		$deleted_from_object_cache = wp_cache_delete( $key, $group );
		$deleted_from_transient    = delete_transient( "{$group}_{$key}" );

		return $deleted_from_object_cache || $deleted_from_transient;
	}

	/**
	 * Get cached data with fallback callback
	 *
	 * @param string   $key        Cache key
	 * @param callable $callback   Fallback callback
	 * @param string   $group      Cache group (optional)
	 * @param int      $expiration Expiration time in seconds (optional)
	 *
	 * @return mixed Cached data or callback result
	 */
	public static function remember( $key, callable $callback, $group = '', $expiration = null ) {
		$data = self::get( $key, $group );

		if ( false === $data ) {
			$data = call_user_func( $callback );
			self::set( $key, $data, $group, $expiration );
		}

		return $data;
	}

	/**
	 * Get cached user avatar with fallbacks
	 *
	 * @param int|null $user_id   User ID (defaults to current user)
	 * @param int      $size      Avatar size in pixels
	 * @param string   $css_class CSS class for avatar image
	 *
	 * @return string            HTML img tag or empty string
	 */
	public static function get_cached_avatar( $user_id = null, $size = 150, $css_class = 'rh_modal_profile_img' ) {
		// Default to current user if no ID provided
		if ( null === $user_id ) {
			if ( ! is_user_logged_in() ) {
				return '';
			}

			$user_id = get_current_user_id();
		}

		// Use the remember pattern for caching
		return RealHomes_Cache::remember( 'user_avatar_' . $user_id, function () use ( $user_id, $size, $css_class ) {
			$profile_image_id = get_user_meta( $user_id, 'profile_image_id', true );
			$avatar_html      = '';

			// 1. Check for custom uploaded image
			if ( $profile_image_id ) {
				$avatar_html = wp_get_attachment_image(
					$profile_image_id,
					[ $size, $size ],
					false,
					[
						'class'    => esc_attr( $css_class ),
						'loading'  => 'lazy',
						'decoding' => 'async'
					]
				);
			}

			// 2. Fallback to Gravatar if no custom image
			if ( empty( $avatar_html ) ) {
				$user = get_userdata( $user_id );
				if ( $user ) {
					$avatar_html = get_avatar(
						$user->user_email,
						$size,
						null,
						$user->display_name,
						[
							'class'    => esc_attr( $css_class ),
							'loading'  => 'lazy',
							'decoding' => 'async',
							'height'   => $size,
							'width'    => $size
						]
					);
				}
			}

			return $avatar_html ? : '';
		}, 'realhomes_user_avatars', 12 * HOUR_IN_SECONDS ); // Cache for 12 hours
	}
}

// Clear cache when icons are updated
add_action( 'updated_option', function ( $option ) {
	$action_buttons_customizer_options = [
		'realhomes_favourite_button_icon',
		'realhomes_compare_button_icon',
		'realhomes_video_button_icon',
		'realhomes_image_button_icon',
		'realhomes_print_button_icon',
		'realhomes_share_button_icon',
		'realhomes_report_button_icon',
	];

	if ( str_contains( $option, 'realhomes_' ) && str_contains( $option, '_property_meta_icon' ) ) {
		RealHomes_Cache::delete( 'property_meta_svg_icons', 'realhomes_property_meta_svg_icons' );
	} else if ( in_array( $option, $action_buttons_customizer_options, true ) ) {
		RealHomes_Cache::delete( 'property_action_buttons_svg_icons', 'property_action_buttons_svg_icons' );
	}
} );

add_action( 'update_option_avatar_default', function ( $old_value, $new_value ) {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	if ( $old_value === $new_value ) {
		return;
	}

	$user_ids = get_users( [ 'fields' => 'ID' ] );
	foreach ( $user_ids as $user_id ) {
		RealHomes_Cache::delete( 'user_avatar_' . $user_id, 'realhomes_user_avatars' );
	}
}, 10, 2 );

/**
 * Clear user avatar cache on profile update or profile image change.
 */
function clear_user_avatar_cache( $user_id ) {
	RealHomes_Cache::delete( 'user_avatar_' . $user_id, 'realhomes_user_avatars' );
}

add_action( 'profile_update', 'clear_user_avatar_cache' );
add_action( 'edit_user_profile_update', 'clear_user_avatar_cache' );
add_action( 'personal_options_update', 'clear_user_avatar_cache' ); // Optional: user edits own profile in admin

add_action( 'updated_user_meta', function ( $meta_id, $user_id, $meta_key ) {
	if ( 'profile_image_id' === $meta_key ) {
		clear_user_avatar_cache( $user_id );
	}
}, 10, 3 );