/**
 * RealHomes Property Print Handler
 *
 * @package     RealHomes
 * @subpackage  Print
 * @since       4.4.2
 */
( function ( $ ) {
    'use strict';
    let activePrintWindow = null; // Track the currently opened print window

    /**
     * Opens and prints the print-friendly property view
     */
    function openAndPrintProperty() {
        if ( typeof printProperty === 'undefined' || ! printProperty.printUrl ) {
            window.print?.(); // Fallback
            return;
        }

        // Close any previously opened print window
        if ( activePrintWindow && ! activePrintWindow.closed ) {
            activePrintWindow.close();
        }

        const width = window.innerWidth;
        const height = window.innerHeight;
        const left = window.screenX;
        const top = window.screenY;

        // Open new blank print window
        activePrintWindow = window.open( '', '', `width=${width},height=${height},top=${top},left=${left}` );
        activePrintWindow.document.write(`<p style="font-family: sans-serif; padding: 2rem;">${printProperty.loadingText}</p>`);

        fetch( printProperty.printUrl )
        .then( response => response.text() )
        .then( html => {
            activePrintWindow.document.open();
            activePrintWindow.document.write( html );
            activePrintWindow.document.close();

            activePrintWindow.onload = function () {
                activePrintWindow.focus();
                activePrintWindow.print();
                activePrintWindow.close();
            };
        } )
        .catch( () => {
            window.print?.(); // Fallback
        } );
    }

    // Handle print button click
    $( document ).on( 'click', '#property-print-button, a.print', function ( event ) {
        event.preventDefault();
        openAndPrintProperty();
    } );

    // Intercept Ctrl+P / Cmd+P globally
    $( document ).on( 'keydown', function ( event ) {
        const isMac           = navigator.platform.toUpperCase().indexOf( 'MAC' ) >= 0;
        const isPrintShortcut =
                  ( ! isMac && event.ctrlKey && event.key === 'p' ) ||
                  ( isMac && event.metaKey && event.key === 'p' );

        if ( isPrintShortcut ) {
            event.preventDefault();
            openAndPrintProperty();
        }
    } );

} )( jQuery );