/**
 * File to handle OTP template related js scripts
 *
 * @since 4.3.8
 */
document.addEventListener( 'DOMContentLoaded', function () {
    "use strict";

    // Select the OTP form
    let realhomesOtpForm = document.getElementById( 'ere-otp-verification-form' );

    if ( realhomesOtpForm ) {
        // Select all OTP input fields
        let otpInputs        = realhomesOtpForm.querySelectorAll( 'input[type="text"]' ),
            countdownElement = realhomesOtpForm.querySelector( '.countdown' ),
            cancelButton     = realhomesOtpForm.querySelector( 'button[name="ere_social_cancel_otp"]' ),
            verifyButton     = realhomesOtpForm.querySelector( 'button[name="ere_social_verify_otp"]' ),
            homeUrl          = cancelButton?.dataset.otpHomeUrl || '/',
            nonceField       = realhomesOtpForm.querySelector( 'input[name="ere_otp_nonce"]' ),
            userIdField      = realhomesOtpForm.querySelector( 'input[name="ere_otp_user_id"]' ),
            timerElement     = document.getElementById( 'otp-timer' ),
            redirectUrl      = '',
            enterAllOTPs     = '',
            verifyingText    = '',
            connectionError  = '',
            successMsg       = '',
            errorMsg         = '',
            attemptsOver     = '',
            otpSet = '';

        if ( typeof otpVerifyingStrings !== 'undefined' ) {
            redirectUrl     = otpVerifyingStrings.redirectUrl;
            enterAllOTPs    = otpVerifyingStrings.enterAllOTPs;
            verifyingText   = otpVerifyingStrings.verifying;
            connectionError = otpVerifyingStrings.connectionError;
            successMsg      = otpVerifyingStrings.successMsg;
            errorMsg        = otpVerifyingStrings.errorMsg;
            attemptsOver    = otpVerifyingStrings.attemptsOver;
        }

        otpInputs.forEach( ( input, index ) => {
            otpSet += ( `ere_otp_digit_${index + 1}`, input.value );
        } );

        // Get timer data from PHP
        const expiryTime     = countdownElement?.dataset.expiryTime ? parseInt( countdownElement.dataset.expiryTime ) : 0;
        const currentTime    = countdownElement?.dataset.currentTime ? parseInt( countdownElement.dataset.currentTime ) : Math.floor( Date.now() / 1000 );
        const expiredMessage = countdownElement?.dataset.expiredMessage | '';

        // Implement the focus navigation logic
        otpInputs.forEach( function ( input, index ) {
            input.addEventListener( 'input', function ( event ) {
                let value = event.target.value;

                // Ensure only one digit is entered
                if ( value.length > 1 ) {
                    event.target.value = value.charAt( 0 );
                }

                // Move to next field if not last field and input is valid
                if ( value && index < otpInputs.length - 1 ) {
                    otpInputs[index + 1].focus();
                }

                // Check if all inputs are filled, then focus on Verify button
                if ( [...otpInputs].every( input => input.value.length === 1 ) ) {
                    verifyButton?.focus();
                }
            } );

            input.addEventListener( 'keydown', function ( event ) {
                if ( event.key === 'Backspace' && input.value === '' && index > 0 ) {
                    otpInputs[index - 1].focus();
                }
            } );

            input.addEventListener( 'keypress', function ( event ) {
                let charCode = event.charCode || event.keyCode;
                if ( charCode < 48 || charCode > 57 ) {
                    event.preventDefault();
                }
            } );
        } );

        // Dynamic countdown timer based on PHP values
        if ( timerElement && expiryTime > 0 ) {
            let timeRemaining = Math.max( 0, expiryTime - currentTime );

            function updateTimerDisplay() {
                let minutes = Math.floor( timeRemaining / 60 );
                let seconds = timeRemaining % 60;

                // Format time as MM:SS
                timerElement.textContent =
                    ( minutes < 10 ? '0' : '' ) + minutes + ':' +
                    ( seconds < 10 ? '0' : '' ) + seconds;
            }

            function countdown() {
                if ( timeRemaining > 0 ) {
                    timeRemaining -= 1;
                    updateTimerDisplay();
                } else {
                    // Timer reached zero, handle expiration
                    timerElement.textContent = expiredMessage;
                    clearInterval( timerInterval );

                    // Disable verify button when expired
                    if ( verifyButton ) {
                        verifyButton.remove();
                    }
                }
            }

            // Initialize and start the timer
            updateTimerDisplay();
            let timerInterval = setInterval( countdown, 1000 );
        } else if ( timerElement ) {
            // Fallback if no expiry time is set
            timerElement.textContent = expiredMessage;
            if ( verifyButton ) {
                verifyButton.disabled = true;
            }
        }

        if ( cancelButton ) {
            cancelButton.addEventListener( 'click', function () {
                window.location.href = homeUrl;
            } );
        }

        if ( verifyButton ) {
            // Get the form element (assuming verifyButton is inside the form)

            if ( verifyButton ) {
                verifyButton.addEventListener( 'click', function ( e ) {
                    e.preventDefault();

                    // Collect values
                    const nonce  = nonceField.value;
                    const userId = userIdField.value;

                    // Verify OTP digits
                    let otpCode = '';
                    otpInputs.forEach( input => {
                        if ( input.value && input.value.length === 1 ) {
                            otpCode += input.value;
                        }
                    } );

                    if ( otpCode.length !== otpInputs.length ) {
                        alert( enterAllOTPs );
                        return;
                    }

                    // Set processing state
                    this.classList.add( 'processing' );
                    const originalText = this.innerHTML;
                    this.innerHTML     = '<span class="rh-spinner"></span> ' + verifyingText;

                    // Prepare form data
                    const formData = new FormData();
                    formData.append( 'action', 'ere_verify_social_otp' );
                    formData.append( 'ere_otp_nonce', nonce );
                    formData.append( 'user_id', userId );

                    // Add OTP digits
                    otpInputs.forEach( ( input, index ) => {
                        formData.append( `ere_otp_digit_${index + 1}`, input.value );
                    } );

                    // Make AJAX request
                    fetch( ajaxurl, {
                        method : 'POST',
                        body   : formData
                    } )
                    .then( response => {
                        if ( ! response.ok ) {
                            otp_manage_message_container( connectionError );
                        }
                        return response.json(); // parse JSON only if response was ok
                    } )
                    .then(response => {
                        const { success, data } = response;

                        if ( success ) {
                            window.location.href = redirectUrl;
                            this.innerHTML = successMsg;
                            otp_manage_message_container( successMsg );
                        } else if ( data ) {
                            const attemptsSpan      = document.querySelector( '.otp-attempts span' );
                            let remainingAttempts   = data.remaining_attempts !== 'undefined' ? data.remaining_attempts : 0;

                            if ( attemptsSpan ) {
                                attemptsSpan.textContent = remainingAttempts;

                                // Error notification
                                otp_manage_message_container( errorMsg );

                                if ( 1 > remainingAttempts ) {
                                    window.location.href = redirectUrl;
                                }
                            }


                            // Show error message
                            console.error( data.message || errorMsg );
                        } else {
                            console.error( errorMsg );
                        }
                    } )
                    .catch( error => {
                        console.error( 'Fetch error:', error, connectionError );
                    } )
                    .finally( () => {
                        if ( this ) {
                            this.classList.remove( 'processing' );
                            this.innerHTML = originalText;
                        }
                    } );

                } );
            }
        }
    }

    function otp_manage_message_container(statement = '', timer = 10000) {
        const messageDiv = document.getElementById('otp-message-container');

        if (statement && messageDiv) {
            // Set text and show
            messageDiv.textContent = statement;
            messageDiv.style.opacity = '0';
            messageDiv.style.display = 'block';
            messageDiv.style.transition = 'opacity 0.4s';

            // Fade in
            requestAnimationFrame(() => {
                messageDiv.style.opacity = '1';
            });

            // Fade out after timer
            setTimeout(() => {
                messageDiv.style.opacity = '0';
                // Hide after fadeOut completes
                setTimeout(() => {
                    messageDiv.style.display = 'none';
                }, 400);
            }, timer);
        }
    }

} );